// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPSupplierOrders.model.ArticleDetail', {

    extend: 'Shopware.apps.ViisonPickwareERPSupplierCommon.model.ArticleDetail',
    fields: [
        { name: 'supplierId', type: 'int', useNull: true },
        { name: 'orderedQuantity', type: 'int' },
        { name: 'deliveredQuantity', type: 'int' },
        { name: 'price', type: 'float' },
        { name: 'statusId', type: 'int' },
        { name: 'supplierArticleNumber', type: 'string' },
        {
            name: 'totalPrice',
            type: 'float',
            convert: function (value, record) {
                return record.get('price') * record.get('orderedQuantity');
            },
        },
    ],

    /**
     * @Override
     */
    set: function (field) {
        this.callParent(arguments);

        // Keep total price in sync with price and ordered quantity properties
        if (field === 'price' || field === 'orderedQuantity') {
            this.set('totalPrice', this.get('price') * this.get('orderedQuantity'));
        }
    },

    /**
     * Selects the first supplier that is mapped to this article detail (only in raw data),
     * which is the default supplier, since they are already ordered so that the default
     * supplier is first in the list. Finally the ID of that supplier is returned.
     *
     * @return int|null
     */
    getDefaultSupplierId: function () {
        return (this.raw.suppliers.length > 0) ? this.raw.suppliers[0].supplierId : null;
    },

    /**
     * Returns and array containing the IDs of all suppliers that are mapped to this article
     * detail (only in raw data).
     *
     * @return int[]
     */
    getAssignedSupplierIds: function () {
        var supplierIds = [];
        Ext.each(this.raw.suppliers, function (supplier) {
            supplierIds.push(supplier.supplierId);
        });

        return supplierIds;
    },

    /**
     * @param int supplierId
     * @return boolean
     */
    isSupplierIdAssigned: function (supplierId) {
        return this.getAssignedSupplierIds().indexOf(supplierId) > -1;
    },

    /**
     * Tries to find the supplier with the given ID in the mappings (only in raw data)
     * and, if found, returns the raw data.
     *
     * @param int supplierId
     * @return object
     */
    getSupplierDataForSupplierId: function (supplierId) {
        // Find the supplier assignment data
        var supplierData = null;
        Ext.each(this.raw.suppliers, function (supplier) {
            if (supplier.supplierId === supplierId) {
                supplierData = supplier;

                return false;
            }

            return undefined;
        });

        return supplierData;
    },

    /**
     * Tries to find the default supplier of this article and, if found,
     * sets it as the assigned supplier and applies assignment data to this article.
     */
    applyDefaultSupplier: function () {
        var defaultSupplierId = this.getDefaultSupplierId();
        if (defaultSupplierId) {
            this.applySupplierWithId(defaultSupplierId);
        }
    },

    /**
     * Tries to find the supplier assignment with the given supplier ID, if found,
     * sets it as the assigned supplier and applies assignment data to this article.
     *
     * @param {int} supplierId
     */
    applySupplierWithId: function (supplierId) {
        this.set('supplierId', supplierId);
        var supplierData = this.getSupplierDataForSupplierId(supplierId);
        if (supplierData) {
            this.set('orderedQuantity', Math.max(
                1,
                this.get('suggestedReorderQuantity'),
                this.get('orderedQuantity')
            ));
            this.set('price', supplierData.purchasePrice);
            this.set('supplierArticleNumber', supplierData.supplierArticleNumber);
        }
    },

    /**
     * Calculates the 'suggestedReorderQuantity' based on the per supplier configured packing unit size.
     *
     * @param {int|null} supplierId
     * @returns {int}
     */
    getSuggestedReorderQuantityForSupplier: function (supplierId) {
        if (!supplierId) {
            supplierId = this.getDefaultSupplierId();
        }
        var supplierData = this.getSupplierDataForSupplierId(supplierId);
        if (!supplierData) {
            return this.get('suggestedReorderQuantity');
        }

        var orderAmount = Math.max(this.get('suggestedReorderQuantity'), supplierData.minimumOrderAmount);

        return Math.ceil(orderAmount / supplierData.packingUnit) * supplierData.packingUnit;
    },

});
