// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

/**
 * Overrides the default readRecords implementation to fix the article
 * and article detail IDs, because the actual article ID is nested in an
 * 'articleDetail' object, while the article detail ID is toplevel as 'articleDetailId'.
 */
Ext.define('Shopware.apps.ViisonPickwareERPSupplierOrders.reader.OrderArticleJson', {

    extend: 'Ext.data.reader.Json',
    type: 'json',
    root: 'data',
    totalProperty: 'total',

    /**
     * @Override
     */
    readRecords: function (data) {
        if (data && data.data) {
            Ext.each(data.data, function (item) {
                item.id = item.articleDetailId;
                item.articleId = (item.articleDetail) ? item.articleDetail.articleId : null;
            });
        }

        return this.callParent([data]);
    },

});

/**
 * Overrides the default getRecordData implementation to set the article detail ID,
 * that is held in the 'id' field, as the 'articleDetailId'.
 */
Ext.define('Shopware.apps.ViisonPickwareERPSupplierOrders.writer.OrderArticleJson', {

    extend: 'Ext.data.writer.Json',
    type: 'json',
    root: 'data',
    allowSingle: false,

    /**
     * @Override
     */
    getRecordData: function () {
        var data = this.callParent(arguments);
        data.articleDetailId = data.id;

        return data;
    },

});

/**
 * Although this store extends the customised version of the common ArticleDetail store,
 * it communicates with the ViisonPickwareERPSupplierOrders and reads/writes supplier order article
 * details. However, these supplier order article details are locally handled as ArticleDetail
 * model instances. This allows us to use drag'n'drop from an ArticleDetail grid to one
 * that actually holds supplier order article details. The communication with the different
 * API is possible by using the OrderArticleJson readers and writers, which 'translate'
 * between the data format of supplier order article detail and (customised) article details.
 */
Ext.define('Shopware.apps.ViisonPickwareERPSupplierOrders.store.AssignedArticleDetail', {

    extend: 'Shopware.apps.ViisonPickwareERPSupplierOrders.store.ArticleDetail',
    pageSize: 1000,
    remoteSort: false,
    sorters: [
        { property: 'article.name' },
    ],
    constructor: function () {
        this.setProxy({
            type: 'ajax',
            batchActions: true,
            api: {
                read: ViisonCommonApp.assembleBackendUrl(
                    'ViisonPickwareERPSupplierOrders/getOrderArticleList'
                ),
                create: ViisonCommonApp.assembleBackendUrl(
                    'ViisonPickwareERPSupplierOrders/createOrderArticles'
                ),
                update: ViisonCommonApp.assembleBackendUrl(
                    'ViisonPickwareERPSupplierOrders/updateOrderArticles'
                ),
                destroy: ViisonCommonApp.assembleBackendUrl(
                    'ViisonPickwareERPSupplierOrders/deleteOrderArticles'
                ),
            },
            reader: Ext.create('Shopware.apps.ViisonPickwareERPSupplierOrders.reader.OrderArticleJson', {}),
            writer: Ext.create('Shopware.apps.ViisonPickwareERPSupplierOrders.writer.OrderArticleJson', {}),
        });

        this.callParent();
    },

});
