// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

/**
 * Overrides the default readRecords implementation to nest the supplier data,
 * which is handled a fake 'hasMany' association and has therefore to be
 * nested in an array before reading any data.
 */
Ext.define('Shopware.apps.ViisonPickwareERPSupplierOrders.reader.OrderJson', {

    extend: 'Ext.data.reader.Json',
    type: 'json',
    root: 'data',
    totalProperty: 'total',

    /**
     * @Override
     */
    readRecords: function (data) {
        if (data && data.data) {
            Ext.each(data.data, function (item) {
                if (item.supplier) {
                    item.supplier = [
                        item.supplier,
                    ];
                }
            });
        }

        return this.callParent([data]);
    },

});

/**
 * Overrides the default getRecordData implementation to normalise the supplier
 * association, which is only a fake 'hasMany' association and has therefore to be
 * fixed before sending any data.
 */
Ext.define('Shopware.apps.ViisonPickwareERPSupplierOrders.writer.OrderJson', {

    extend: 'Ext.data.writer.Json',
    type: 'json',
    root: 'data',
    allowSingle: false,

    /**
     * @Override
     */
    getRecordData: function () {
        var data = this.callParent(arguments);
        if (Ext.isArray(data.supplier) && data.supplier.length === 1) {
            // Normalise the supplier array (containing only one object) to a direct object association
            data.supplier = data.supplier[0];
        }

        return data;
    },

});

Ext.define('Shopware.apps.ViisonPickwareERPSupplierOrders.store.Order', {

    extend: 'Ext.data.Store',
    model: 'Shopware.apps.ViisonPickwareERPSupplierOrders.model.Order',
    autoLoad: false,
    remoteFilter: true,
    remoteSort: true,
    sorters: [
        { property: 'created', direction: 'DESC' },
    ],
    proxy: {
        type: 'ajax',
        batchActions: true,
        api: {
            read: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPSupplierOrders/getOrderList'),
            create: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPSupplierOrders/createOrders'),
            update: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPSupplierOrders/updateOrders'),
            destroy: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPSupplierOrders/deleteOrders'),
        },
        reader: Ext.create('Shopware.apps.ViisonPickwareERPSupplierOrders.reader.OrderJson', {}),
        writer: Ext.create('Shopware.apps.ViisonPickwareERPSupplierOrders.writer.OrderJson', {}),
        encodeSorters: function (sorters) {
            // Replace any sorters on 'warehouseId' with a sorter on 'warehouse.name'
            sorters.forEach(function (sorter) {
                if (sorter.property === 'warehouseId') {
                    sorter.property = 'warehouse.name';
                }
            });

            // Use a default AJAX proxy to encode the sorters, because 'callParent' does not
            // work in this context for some reason
            var proxy = Ext.create('Ext.data.proxy.Ajax', {});
            var encodedSorters = proxy.encodeSorters(sorters);

            return encodedSorters;
        },
    },

});
