// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.ArticleAssignment', {

    extend: 'Shopware.apps.ViisonPickwareERPSupplierCommon.view.AssignmentPanel',
    alias: 'widget.viison_pickware_erp_supplier_orders-detail-article_assignment',
    cls: 'viison_pickware_erp_supplier_orders-detail-article_assignment',

    sourceWarehouseIds: [],
    targetWarehouse: null,
    leftTitle: '',
    rightTitle: '',

    /**
     * @Override
     */
    initComponent: function () {
        this.initializeStateManagement();
        this.callParent(arguments);

        // Add tooltip
        this.rightPanel.getView().on('render', function (view) {
            view.tooltip = Ext.create('Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.ArticleTooltip', {
                articleGridView: view,
                supplierOrder: this.record,
                supplierStore: this.supplierStore,
                itemStatusStore: this.itemStatusStore,
            });
        }, this);

        if (!this.record) {
            // Update the layout of the panel containing the assignment buttons
            Ext.apply(this.items[1], {
                layout: {
                    type: 'vbox',
                    pack: 'start',
                },
            });

            // Listen on clicks on the grouping header
            this.rightPanel.on('groupclick', function (view, node, group, e) {
                var supplierId = parseInt(group, 10);
                // Determine which element was clicked
                var action = e.getTarget().getAttribute('data-action');
                if (action === 'createSingleOrder') {
                    // Find the group records
                    var records;
                    Ext.each(this.rightStore.getGroups(), function (rightGroup) {
                        if (rightGroup.name === supplierId) {
                            records = rightGroup.children;

                            return false;
                        }

                        return undefined;
                    });

                    this.fireEvent('createSingleOrder', this, records, supplierId);
                }
            }, this);
        } else {
            // Fill comment form with record data
            this.rightPanel.nextSibling('form').loadRecord(this.record);
            this.setTargetWarehouse(this.record.getWarehouse());
        }
    },

    /**
     * @Override
     */
    getLeftPanelConfig: function () {
        var config = this.callParent(arguments);
        // Create a supplier store for the supplier selection
        var filterSupplierStore = Ext.create('Shopware.apps.ViisonPickwareERPSupplierManagement.store.Supplier', {
            pageSize: 10,
        });
        filterSupplierStore.load();

        config.stateful = true;
        config.stateId = 'viison_pickware_erp_supplier_orders-detail-article_assignment--left';

        // Customize the top toolbar to add a supplier selection
        config.dockedItems[0].items.splice(1, 0, {
            xtype: 'combobox',
            emptyText: ViisonCommonApp.getSnippet(
                'edit/article_assignment/supplier_selection/placeholder',
                'backend/viison_pickware_erp_supplier_orders/main'
            ),
            store: filterSupplierStore,
            displayField: 'name',
            valueField: 'id',
            itemId: 'supplierSelection',
            pageSize: 10,
            typeAhead: false,
            autoSelect: false,
            matchFieldWidth: false,
            margin: '0 5 0 2',
            width: 200,
            listConfig: {
                width: 250,
                getInnerTpl: function (displayField) {
                    // Use unicode curly braces
                    return '<tpl if="id == 0"><span style="color:#6c818f;"></tpl>\u007B' + displayField + '\u007D<tpl if="id == 0"></span></tpl>';
                },
            },
            listeners: {
                change: function (combo, newValue, oldValue) {
                    if ((Ext.isNumeric(newValue) || newValue === null) && newValue !== oldValue) {
                        var record = (Ext.isNumeric(newValue)) ? combo.findRecord('id', newValue) : null;
                        this.fireEvent('supplierFilterChanged', record, this.leftStore);
                    }
                },
                focus: function (combo) {
                    combo.expand();
                },
                scope: this,
            },
        }, {
            xtype: 'tbspacer',
            width: 35,
        });

        return config;
    },

    /**
     * @Override
     */
    createLeftColumns: function () {
        var columns = this.createSharedArticleColumns();
        // Prepend a column for showing a warning icon in case the article needs to be re-ordered
        var reorderColumns = [{
            header: '',
            dataIndex: 'reorder',
            hideable: false,
            tdCls: 'has--icon',
            width: 32,
            renderer: function (value, metaData, record) {
                if (value === 2) {
                    var urgentReorderInfo = ViisonCommonApp.getSnippet(
                        'edit/article_assignment/tooltip/urgent_reorder',
                        'backend/viison_pickware_erp_supplier_orders/main'
                    );
                    metaData.tdAttr = 'data-qtip="' + urgentReorderInfo + '"';
                    metaData.tdCls = 'is--error';
                } else if (value === 1) {
                    var reorderInfo = ViisonCommonApp.getSnippet(
                        'edit/article_assignment/tooltip/reorder',
                        'backend/viison_pickware_erp_supplier_orders/main'
                    );
                    metaData.tdAttr = 'data-qtip="' + reorderInfo + '"';
                    metaData.tdCls = 'is--warning';
                } else if (record.get('dropShippingArticle')) {
                    var dropShippingInfo = ViisonCommonApp.getSnippet(
                        'edit/article_assignment/tooltip/drop_shipping',
                        'backend/viison_pickware_erp_supplier_orders/main'
                    );
                    metaData.tdAttr = 'data-qtip="' + dropShippingInfo + '"';
                    metaData.tdCls = 'is--box';
                }

                return '';
            },
        }];

        // Insert a column showing the already incoming stock between the available stock and the minimum stock columns
        var incomingStockColumns = [
            {
                header: ViisonCommonApp.getSnippet(
                    'edit/article_assignment/column/available_stock',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                tooltip: this.getViisonSnippet(
                    'edit/article_assignment/column/available_stock/help_text',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                dataIndex: 'availableStock',
                align: 'right',
                width: 50,
            }, {
                header: ViisonCommonApp.getSnippet(
                    'edit/article_assignment/column/minimum_stock',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                tooltip: this.getViisonSnippet(
                    'edit/article_assignment/column/minimum_stock/help_text',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                dataIndex: 'minimumStock',
                align: 'right',
                width: 50,
            }, {
                header: ViisonCommonApp.getSnippet(
                    'edit/article_assignment/column/target_stock',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                tooltip: this.getViisonSnippet(
                    'edit/article_assignment/column/target_stock/help_text',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                dataIndex: 'targetStock',
                align: 'right',
                width: 50,
            }, {
                header: ViisonCommonApp.getSnippet(
                    'edit/article_assignment/column/incoming_stock',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                tooltip: this.getViisonSnippet(
                    'edit/article_assignment/column/incoming_stock/help_text',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                dataIndex: 'incomingStock',
                align: 'right',
                width: 80,
            }, {
                header: ViisonCommonApp.getSnippet(
                    'edit/article_assignment/column/order_suggestion',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                tooltip: this.getViisonSnippet(
                    'edit/article_assignment/column/order_suggestion/help_text',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                dataIndex: 'suggestedReorderQuantity',
                scope: this,
                align: 'right',
                tdCls: 'is--proposal-column',
                width: 70,
            },
        ];

        // Merge columns:  columns[0-4] + stock columns + columns[4-End]
        columns = reorderColumns
            .concat(columns.slice(0, 4))
            .concat(incomingStockColumns)
            .concat(columns.slice(4, columns.length));

        return columns;
    },

    /**
     * @Override
     */
    createRightContainer: function () {
        if (!this.record) {
            return this.callParent(arguments);
        }

        return {
            xtype: 'panel',
            flex: 1,
            layout: {
                type: 'vbox',
                align: 'stretch',
            },
            border: false,
            bodyStyle: 'border-bottom:none;',
            items: [
                this.rightPanel,
                {
                    xtype: 'form',
                    cls: 'is--viison-common--form',
                    record: this.record,
                    height: 200,
                    border: false,
                    style: {
                        backgroundColor: '#f0f2f4',
                    },
                    layout: {
                        type: 'hbox',
                        align: 'stretch',
                    },
                    padding: '15 20 15 20',
                    items: [{
                        xtype: 'fieldset',
                        flex: 1,
                        margin: '0 8 0 0',
                        padding: 0,
                        border: false,
                        items: [{
                            xtype: 'label',
                            style: {
                                display: 'block',
                                fontWeight: 'bold',
                                paddingTop: '1px',
                            },
                            text: ViisonCommonApp.getSnippet(
                                'edit/article_assignment/field/document_comment/title',
                                'backend/viison_pickware_erp_supplier_orders/main'
                            ) + ':',
                            margin: '0 0 8 0',
                        }, {
                            xtype: 'textareafield',
                            name: 'documentComment',
                            height: 105,
                            anchor: '100%',
                            supportText: ViisonCommonApp.getSnippet(
                                'edit/article_assignment/field/document_comment/info',
                                'backend/viison_pickware_erp_supplier_orders/main'
                            ),
                            hideLabel: true,
                        }],
                    }, {
                        xtype: 'fieldset',
                        flex: 1,
                        margin: '0 0 0 8',
                        padding: 0,
                        border: false,
                        items: [{
                            xtype: 'label',
                            style: {
                                display: 'block',
                                fontWeight: 'bold',
                                paddingTop: '1px',
                            },
                            text: ViisonCommonApp.getSnippet(
                                'edit/article_assignment/field/comment/title',
                                'backend/viison_pickware_erp_supplier_orders/main'
                            ) + ':',
                            margin: '0 0 8 0',
                        }, {
                            xtype: 'textareafield',
                            name: 'comment',
                            height: 105,
                            anchor: '100%',
                            supportText: ViisonCommonApp.getSnippet(
                                'edit/article_assignment/field/comment/info',
                                'backend/viison_pickware_erp_supplier_orders/main'
                            ),
                            hideLabel: true,
                        }],
                    }],
                },
            ],
            dockedItems: [{
                xtype: 'toolbar',
                dock: 'bottom',
                ui: 'shopware-ui',
                cls: 'is--viison-common--button-panel is--primary',
                items: [
                    '->',
                    {
                        text: ViisonCommonApp.getSnippet(
                            'edit/article_assignment/toolbar/button/cancel',
                            'backend/viison_pickware_erp_supplier_orders/main'
                        ),
                        cls: 'secondary',
                        action: 'cancelEditing',
                        handler: function () {
                            this.fireEvent('cancelEditing', this);
                        },
                        scope: this,
                    }, {
                        text: ViisonCommonApp.getSnippet(
                            'edit/article_assignment/toolbar/button/copy',
                            'backend/viison_pickware_erp_supplier_orders/main'
                        ),
                        cls: 'secondary',
                        action: 'copyOrder',
                        disabled: (this.record.getSupplier() === null),
                        handler: function () {
                            this.fireEvent('copyOrder', this);
                        },
                        scope: this,
                    }, {
                        text: ViisonCommonApp.getSnippet(
                            'edit/article_assignment/toolbar/button/save',
                            'backend/viison_pickware_erp_supplier_orders/main'
                        ),
                        cls: 'primary',
                        action: 'saveOrder',
                        handler: function () {
                            this.fireEvent('saveOrder', this);
                        },
                        scope: this,
                    },
                ],
            }],
        };
    },

    /**
     * @Override
     */
    getRightPanelConfig: function () {
        var config = this.callParent(arguments);

        // Use a bug in ExtJS that removes the background of the row editor and hence makes all columns visible
        config.cls = 'x-tabpanel-child viison-common--grid has--vertical-lines has--no-borders';

        // Preserve scroll position when a record is updated
        Ext.apply(config.viewConfig, {
            preserveScrollOnRefresh: true,
        });

        // Allow row editing
        this.rowEditor = Ext.create('Ext.grid.plugin.RowEditing', {
            clicksToMoveEditor: 1,
            autoCancel: false,
            listeners: {
                scope: this,
                edit: function (editor, context) {
                    // Explicitly set the price to fix conversion errors
                    context.record.set('price', context.newValues.price);
                    context.record.commit();
                    this.fireEvent('articleRowEdited', this, context.record);
                },
            },
        });
        config.plugins = config.plugins || [];
        config.plugins.push(this.rowEditor);

        // Add grouping feature
        this.rightPanelGroupingFeature = Ext.create('ViisonPickwareERP.Ext.grid.feature.Grouping', {
            hideGroupedHeader: false,
            enableGroupingMenu: false,
            collapsible: false,
            groupHeaderTpl: [
                '<div class="is--viison-group-header">',
                '<div class="is--viison-item-left">',
                '{groupValue:this.formatSupplier} ({groupValue:this.formatOrderValue})',
                '</div>',
                '<div class="is--viison-item-right" style="display: {groupValue:this.selectDisplayMode};">',
                '<div class="x-btn primary small x-toolbar-item x-btn-default-toolbar-small x-noicon x-btn-noicon x-btn-default-toolbar-small-noicon">',
                '<em>',
                '<button class="x-btn-center" hidefocus="true" role="button" data-action="createSingleOrder">',
                '<span class="x-btn-inner" data-action="createSingleOrder">' + ViisonCommonApp.getSnippet('edit/article_assignment/grouping/header/create_order_button', 'backend/viison_pickware_erp_supplier_orders/main') + '</span>',
                '</button>',
                '</em>',
                '</div>',
                '</div>',
                '</div>',
                {
                    formatSupplier: function (supplierId) {
                        // Try to find the supplier with the given ID
                        var supplier = this.supplierStore.getById(supplierId);

                        return (supplier) ? supplier.get('name') : ViisonCommonApp.getSnippet('edit/article_assignment/no_supplier', 'backend/viison_pickware_erp_supplier_orders/main');
                    }.bind(this),
                    formatOrderValue: function (supplierId) {
                        var records = this.rightStore.queryBy(function (record) {
                            return record.data.supplierId === supplierId;
                        });

                        var orderValue = 0;
                        var articleCount = 0;
                        records.each(function (record) {
                            orderValue += record.data.price * record.data.orderedQuantity;
                            articleCount += record.data.orderedQuantity;
                        });

                        return Ext.String.format(
                            ViisonCommonApp.getSnippet(
                                'edit/article_assignment/supplier_order_info_header',
                                'backend/viison_pickware_erp_supplier_orders/main'
                            ),
                            records.length,
                            articleCount,
                            this.renderAmountWithCurrency(orderValue, supplierId)
                        );
                    }.bind(this),
                    selectDisplayMode: function (supplierId) {
                        return (supplierId && !this.record) ? 'inline-block' : 'none';
                    }.bind(this),
                },
            ],
        });
        config.features = config.features || [];
        config.features.push(this.rightPanelGroupingFeature);
        config.stateful = true;

        if (!this.record) {
            config.stateId = 'viison_pickware_erp_supplier_orders-detail-article_assignment--create--right';
            config.border = false;
            // Add bottom aligned toolbar
            config.dockedItems.push({
                xtype: 'toolbar',
                dock: 'bottom',
                cls: 'shopware-ui is--viison-common--button-panel has--no-top-border',
                items: [
                    '->',
                    {
                        text: ViisonCommonApp.getSnippet(
                            'edit/article_assignment/toolbar/button/create_all_orders',
                            'backend/viison_pickware_erp_supplier_orders/main'
                        ),
                        cls: 'primary',
                        action: 'createAllOrders',
                        scope: this,
                        handler: function () {
                            this.fireEvent('createAllOrders', this);
                        },
                    },
                ],
            });
        } else {
            config.stateId = 'viison_pickware_erp_supplier_orders-detail-article_assignment--edit--right';
            config.style = { marginTop: '-1px !important' };
        }

        // Add buttons and search field to a top toolbar.
        var topToolbar = {
            xtype: 'toolbar',
            itemId: 'topToolbar',
            dock: 'top',
            ui: 'shopware-ui is--viison-common--toolbar',
            items: [
                {
                    xtype: 'button',
                    itemId: 'deleteButton',
                    iconCls: 'sprite-picture--minus',
                    text: ViisonCommonApp.getSnippet(
                        'list/toolbar/button/delete_articles/all',
                        'backend/viison_pickware_erp_supplier_orders/main'
                    ),
                    margin: '0 5 0 0',
                    scope: this,
                    handler: function () {
                        var selectionModel = this.rightPanel.getSelectionModel();
                        this.fireEvent('rightPanelRemoveArticles', this, selectionModel);
                    },
                },
                {
                    xtype: 'button',
                    itemId: 'barcodeLabelButton',
                    iconCls: 'sprite-barcode',
                    margin: '0 5 0 0',
                    text: ViisonCommonApp.getSnippet(
                        'list/toolbar/button/save_for_barcode_label_printing/all',
                        'backend/viison_pickware_erp_supplier_orders/main'
                    ),
                    scope: this,
                    handler: function () {
                        var selectionModel = this.rightPanel.getSelectionModel();
                        this.fireEvent('rightPanelAddArticleLabels', this, selectionModel);
                    },
                },
            ],
        };

        if (!this.record) {
            topToolbar.items.push({
                xtype: 'button',
                itemId: 'assignSupplierButton',
                iconCls: 'sprite-truck-box-label',
                margin: '0 5 0 0',
                text: ViisonCommonApp.getSnippet(
                    'list/toolbar/button/assign_supplier',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                scope: this,
                disabled: true,
                handler: function () {
                    this.fireEvent('rightPanelAssignSupplier', this, this.rightPanel);
                },
            });

            topToolbar.items.push('->');
            topToolbar.items.push({
                xtype: 'textfield',
                name: 'searchfield',
                cls: 'searchfield',
                width: 170,
                enableKeyEvents: true,
                emptyText: ViisonCommonApp.getSnippet(
                    'search_field/placeholder',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                listeners: {
                    buffer: 500,
                    change: function (field) {
                        this.fireEvent('rightPanelSearchFieldChanged', this, field.value);
                    },
                    scope: this,
                },
            });
        }
        config.dockedItems.push(topToolbar);

        return config;
    },

    /**
     * @Override
     */
    createRightColumns: function () {
        var defaultColumns = this.createSharedArticleColumns();
        var defaultActionColumn = defaultColumns.pop();
        var columns = [];

        // Add read-only row editor fields for the article name, order number and fabricator number
        Ext.each([0, 1, 3], function (index) {
            defaultColumns[index].editor = {
                xtype: 'textfield',
                readOnly: true,
            };
        });

        // Add icon column at first place. Hovering over this column will display the article tooltip
        columns.push({
            header: '',
            hideable: false,
            tdCls: 'is--info-icon-column viison--show-tooltip',
            width: 24,
        });

        columns = columns.concat(defaultColumns);

        if (!this.record) {
            // Insert a supplier column right after the fabricator number column
            var supplierColumn = {
                header: ViisonCommonApp.getSnippet(
                    'edit/article_assignment/column/supplier',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                dataIndex: 'supplierId',
                flex: 1,
                scope: this,
                renderer: function (value, metaData, record) {
                    var supplierId = record.get('supplierId');
                    var supplier = this.supplierStore.getById(supplierId);
                    if (!supplier) {
                        return ViisonCommonApp.getSnippet(
                            'edit/article_assignment/no_supplier',
                            'backend/viison_pickware_erp_supplier_orders/main'
                        );
                    }
                    var supplierName = supplier.get('name');
                    var defaultSupplier = record.getDefaultSupplierId();
                    var isDefaultSupplier = (defaultSupplier !== null && defaultSupplier === supplierId);
                    if (!isDefaultSupplier && defaultSupplier !== null) {
                        var tooltip = ViisonCommonApp.getSnippet(
                            'edit/article_assignment/column/non_default_supplier_tooltip',
                            'backend/viison_pickware_erp_supplier_orders/main'
                        );
                        supplierName += '<span class="sprite-lightning" style="margin-left: 5px; padding-left: 16px; padding-bottom: 2px; padding-top: 2px;" data-qtip="' + tooltip + '"></span>';
                    }

                    return supplierName;
                },
            };

            // Create a supplier store for the row editor
            var rowEditorSupplierStore = Ext.create(
                'Shopware.apps.ViisonPickwareERPSupplierManagement.store.Supplier',
                {
                    pageSize: 10,
                }
            );
            // Make sure our supplier store is up to date with the items that
            // can be chosen in the row editor supplier store
            rowEditorSupplierStore.on('load', function () {
                this.supplierStore.load();
            }, this);
            rowEditorSupplierStore.load();

            // Make the supplier column editable
            supplierColumn.editor = {
                xtype: 'combo',
                store: rowEditorSupplierStore,
                displayField: 'name',
                valueField: 'id',
                pageSize: 10,
                typeAhead: false,
                autoSelect: false,
                editable: false,
                emptyText: ViisonCommonApp.getSnippet(
                    'edit/article_assignment/no_supplier',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                matchFieldWidth: false,
                listConfig: {
                    width: 220,
                    cls: 'viison_pickware_erp_supplier_orders-supplier_assignment',
                },
                listeners: {
                    scope: this,
                    select: function (combo, records) {
                        var selection = this.rightPanel.getSelectionModel().getSelection();
                        if (records.length > 0 && selection.length > 0) {
                            // Get current editor values
                            var editorForm = this.rowEditor.editor.getForm();
                            var values = editorForm.getValues();

                            // Try to get values from supplier assignment
                            var articleDetail = selection[0];
                            var supplierId = records[0].get('id');
                            if (articleDetail.isSupplierIdAssigned(supplierId)) {
                                // Use the assignment values
                                var supplierData = articleDetail.getSupplierDataForSupplierId(supplierId);
                                values.orderedQuantity = Math.max(
                                    1,
                                    values.orderedQuantity,
                                    articleDetail.getSuggestedReorderQuantityForSupplier(supplierId)
                                );
                                values.price = supplierData.purchasePrice;
                                values.supplierArticleNumber = supplierData.supplierArticleNumber;
                            }

                            // Update the row editor form
                            this.rowEditor.editor.getForm().setValues(values);
                        }
                    },
                    render: function (combo) {
                        var picker = combo.getPicker();
                        picker.pagingToolbar.on('change', function (pagingToolBar, changeEvent, options) {
                            this.displayGroupedSuppliers(options.combo);
                        }, this, {
                            combo: combo,
                        });
                    },
                    expand: function (combo) {
                        this.displayGroupedSuppliers(combo);
                    },
                },
                tpl: new Ext.XTemplate(
                    '{%this.currentlyAssigned = null%}',
                    '<tpl for=".">',
                    '<tpl if="this.shouldShowHeader(assigned)">',
                    '<div class="group-header">{[this.getHeaderName(values.assigned)]}</div>',
                    '</tpl>',
                    '<div class="x-boundlist-item">',
                    '<tpl if="default">',
                    '<div class="is--default">{name}<span class="sprite-star" style="margin-left: 5px; padding-left: 16px; padding-bottom: 2px; padding-top: 2px;"></span></div>',
                    '<tpl else>',
                    '{name}',
                    '</tpl>',
                    '</div>',
                    '</tpl>',
                    {
                        /**
                         * @param {bool} assigned True if the current supplier is assigned to the
                         * article, false otherwise
                         * @returns {boolean}
                         */
                        shouldShowHeader: function (assigned) {
                            return this.currentlyAssigned !== assigned;
                        },
                        /**
                         * @param {bool} assigned True if the current supplier is assigned to the
                         * article, false otherwise
                         * @returns {boolean}
                         */
                        getHeaderName: function (assigned) {
                            this.currentlyAssigned = assigned;
                            if (assigned === true) {
                                return ViisonCommonApp.getSnippet(
                                    'edit/article_assignment/supplier_grouping/header/assigned',
                                    'backend/viison_pickware_erp_supplier_orders/main'
                                );
                            }

                            return ViisonCommonApp.getSnippet(
                                'edit/article_assignment/supplier_grouping/header/others',
                                'backend/viison_pickware_erp_supplier_orders/main'
                            );
                        },
                    }
                ),
            };
            columns.push(supplierColumn);
        }

        columns.push({
            header: ViisonCommonApp.getSnippet(
                'edit/article_assignment/column/supplier_article_number',
                'backend/viison_pickware_erp_supplier_orders/main'
            ),
            dataIndex: 'supplierArticleNumber',
            flex: 1,
            hidden: true,
            editor: {
                xtype: 'textfield',
                readOnly: true,
            },
        });

        columns = columns.concat([
            {
                header: ViisonCommonApp.getSnippet(
                    'edit/article_assignment/column/online_physical_stock',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                tooltip: ViisonCommonApp.getSnippet(
                    'edit/article_assignment/column/online_physical_stock/tooltip',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                dataIndex: 'onlinePhysicalStock',
                hidden: true,
                width: 70,
                align: 'right',
            }, {
                header: ViisonCommonApp.getSnippet(
                    'edit/article_assignment/column/online_available_stock',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                tooltip: ViisonCommonApp.getSnippet(
                    'edit/article_assignment/column/online_available_stock/tooltip',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                dataIndex: 'onlineAvailableStock',
                width: 70,
                align: 'right',
            },
        ]);

        if (this.record) {
            columns.push({
                // Insert a column for editing the delivered quantity right before the ordered quantity
                header: ViisonCommonApp.getSnippet(
                    'edit/article_assignment/column/delivered_quantity',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                dataIndex: 'deliveredQuantity',
                width: 80,
                align: 'right',
                editor: {
                    xtype: 'numberfield',
                    allowBlank: true,
                    minValue: 0,
                    step: 1,
                },
            });
        }

        columns = columns.concat([
            {
                header: ViisonCommonApp.getSnippet(
                    'edit/article_assignment/column/order_quantity',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                dataIndex: 'orderedQuantity',
                width: 80,
                align: 'right',
                editor: {
                    xtype: 'numberfield',
                    allowBlank: false,
                    minValue: 1,
                    step: 1,
                },
            }, {
                header: ViisonPickwarePurchasePriceHelper.purchasePriceLabel(
                    ViisonCommonApp.getSnippet(
                        'edit/article_assignment/column/price',
                        'backend/viison_pickware_erp_supplier_orders/main'
                    )
                ),
                dataIndex: 'price',
                width: 75,
                align: 'right',
                editor: {
                    xtype: 'numberfield',
                    allowBlank: true,
                    minValue: 0,
                    step: 0.01,
                },
                scope: this,
                renderer: this.customCurrencyRenderer,
            }, {
                header: ViisonPickwarePurchasePriceHelper.purchasePriceLabel(
                    ViisonCommonApp.getSnippet(
                        'edit/article_assignment/column/total_price',
                        'backend/viison_pickware_erp_supplier_orders/main'
                    )
                ),
                dataIndex: 'totalPrice',
                hidden: true,
                width: 90,
                align: 'right',
                scope: this,
                renderer: this.customCurrencyRenderer,
            },
        ]);

        if (this.record) {
            // Insert a column for editing the item status right before the action column
            columns.push({
                header: ViisonCommonApp.getSnippet(
                    'edit/article_assignment/column/status',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                dataIndex: 'statusId',
                width: 55,
                editor: {
                    xtype: 'combobox',
                    store: this.itemStatusStore,
                    queryMode: 'local',
                    valueField: 'id',
                    displayField: 'description',
                    editable: false,
                    triggerAction: 'all',
                    selectOnTab: true,
                    lazyRender: true,
                    listClass: 'x-combo-list-small',
                },
                scope: this,
                renderer: function (value) {
                    var record = this.itemStatusStore.getById(value);

                    return (record) ? record.get('description') : value;
                },
            });
        }

        // Add a 'barcode label' and 'delete' icons to the action column
        defaultActionColumn.width += 50;
        defaultActionColumn.items.push(
            {
                iconCls: 'sprite-barcode',
                tooltip: ViisonCommonApp.getSnippet(
                    'edit/article_assignment/tooltip/save_for_barcode_label_printing',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                scope: this,
                handler: function (grid, rowIndex) {
                    this.fireEvent('saveArticleDetailForBarcodeLabelPrinting', this, grid.getStore().getAt(rowIndex));
                },
            },
            {
                iconCls: 'sprite-minus-circle',
                action: 'removeArticleDetail',
                tooltip: ViisonCommonApp.getSnippet(
                    'edit/article_assignment/tooltip/remove_item',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                scope: this,
                handler: function (grid, rowIndex) {
                    this.fireEvent('removeArticleDetail', this, grid.getStore().getAt(rowIndex));
                },
            }
        );
        columns.push(defaultActionColumn);

        return columns;
    },

    /**
     * @Override
     */
    createMiddleContainer: function () {
        if (this.record) {
            return this.callParent(arguments);
        }

        return {
            xtype: 'panel',
            border: 1,
            layout: {
                type: 'vbox',
                pack: 'center',
            },
            bodyStyle: {
                backgroundColor: '#ebedef',
                borderTop: 'none',
                borderBottom: 'none',
            },
            items: [{
                xtype: 'panel',
                border: 0,
                margin: '48 0 0 0',
                layout: {
                    type: 'vbox',
                    pack: 'top',
                },
                bodyStyle: {
                    backgroundColor: '#ebedef',
                },
                items: [{
                    xtype: 'button',
                    cls: Ext.baseCSSPrefix + 'form-itemselector-btn',
                    iconCls: 'c-sprite-add-urgent-reorder-articles',
                    tooltip: ViisonCommonApp.getSnippet(
                        'edit/article_assignment/tooltip/assign_urgent_reorder_articles',
                        'backend/viison_pickware_erp_supplier_orders/main'
                    ),
                    navBtn: true,
                    margin: '4 4 2 4',
                    scope: this,
                    handler: function () {
                        this.fireEvent('assignUrgentReorderArticles', this);
                    },
                }, {
                    xtype: 'button',
                    cls: Ext.baseCSSPrefix + 'form-itemselector-btn',
                    iconCls: 'c-sprite-add-reorder-articles',
                    tooltip: ViisonCommonApp.getSnippet(
                        'edit/article_assignment/tooltip/assign_reorder_articles',
                        'backend/viison_pickware_erp_supplier_orders/main'
                    ),
                    navBtn: true,
                    margin: '2 4 4 4',
                    scope: this,
                    handler: function () {
                        this.fireEvent('assignReorderArticles', this);
                    },
                }],
            }, {
                xtype: 'panel',
                border: 0,
                flex: 1,
                layout: {
                    type: 'vbox',
                    pack: 'center',
                },
                bodyStyle: {
                    backgroundColor: '#ebedef',
                },
                items: this.createMiddleButtons(),
            }],
        };
    },

    /**
     * @return Ext.grid.column.Column[]
     */
    createSharedArticleColumns: function () {
        return [{
            header: ViisonCommonApp.getSnippet(
                'edit/article_assignment/column/order_number',
                'backend/viison_pickware_erp_supplier_orders/main'
            ),
            dataIndex: 'orderNumber',
            width: 90,
        }, {
            header: ViisonCommonApp.getSnippet(
                'edit/article_assignment/column/name',
                'backend/viison_pickware_erp_supplier_orders/main'
            ),
            dataIndex: 'name',
            flex: 2,
        }, {
            header: ViisonCommonApp.getSnippet(
                'edit/article_assignment/column/fabricator_name',
                'backend/viison_pickware_erp_supplier_orders/main'
            ),
            dataIndex: 'manufacturerName',
            hidden: true,
            flex: 2,
        }, {
            header: ViisonCommonApp.getSnippet(
                'edit/article_assignment/column/fabricator_number',
                'backend/viison_pickware_erp_supplier_orders/main'
            ),
            dataIndex: 'manufacturerArticleNumber',
            hidden: true,
            flex: 1,
        }, {
            xtype: 'actioncolumn',
            hideable: false,
            align: 'center',
            width: 35,
            items: [{
                iconCls: 'sprite-sticky-notes-pin',
                action: 'showArticleDetail',
                tooltip: ViisonCommonApp.getSnippet(
                    'edit/article_assignment/tooltip/show_article_details',
                    'backend/viison_pickware_erp_supplier_orders/main'
                ),
                scope: this,
                getClass: function (value, meta, record) {
                    return (!record.get('articleId')) ? 'x-hide-visibility' : '';
                },
                handler: function (grid, rowIndex) {
                    this.fireEvent('showArticleDetail', grid.getStore().getAt(rowIndex));
                },
            }],
        }];
    },

    /**
     * By default Shopware does not register a custom state provider when the backend
     * is initialized. Unfortunately the default state provider is non-persistent. Hence
     * we need to replace the default state provider with a persistent one (local storage)
     * in order to support "real" state persistence. Since Shopware itself replaces
     * the default provider in some of its components (e.g. the article list), we
     * replace the registered state provider only in case that it is non persistent.
     */
    initializeStateManagement: function () {
        var currentStateProvider = Ext.state.Manager.getProvider();

        if (!(currentStateProvider instanceof Ext.state.LocalStorageProvider
            || currentStateProvider instanceof Ext.state.CookieProvider)) {
            Ext.state.Manager.setProvider(new Ext.state.LocalStorageProvider());
        }
    },

    /**
     * Helper renderer to render a row in the article detail grid. Formats the purchase price with the currency of the
     * current order (edit view) or currency of the assigned supplier of the article detail (create).
     *
     * @param {number} value
     * @param meta
     * @param {Shopware.apps.ViisonPickwareERPSupplierManagement.model.ArticleDetail} record
     * @returns {string}
     */
    customCurrencyRenderer: function (value, meta, record) {
        return this.renderAmountWithCurrency(value, record.get('supplierId'));
    },

    /**
     * Renders the given value with the currency of the current order (edit view) or currency of any assigned supplier
     * given by the supplierId (create view), or uses the default currency if no other currency is assigned.
     *
     * @param {number} value
     * @param {int} supplierId
     * @returns {string}
     */
    renderAmountWithCurrency: function (value, supplierId) {
        var currency;
        if (this.record) {
            // This is an existing order. Use the currency of the order (this.record)
            currency = this.record.getCurrency();
        } else if (supplierId) {
            // This is an order creation. Use the currency of the supplier (given by supplierId)
            var supplier = this.supplierStore.getById(supplierId);
            currency = supplier.getCurrency();
        } else {
            // No order or supplier id is given (e.g. order creation with an article without an assigned supplier). Use
            // Default currency.
            return ViisonCurrencyFormatter.renderer(value);
        }

        return ViisonCurrencyFormatter.formatWithSymbolAndPosition(
            value,
            currency.raw.symbol,
            currency.raw.symbolPosition
        );
    },

    /**
     * Prepares a list of assigned and unassigned suppliers and loads the list into the template. Assigned suppliers
     * will be loaded from the complete supplierStore all other suppliers will be taken from the
     * paginated version of the store from the combobox.
     * @param {Ext.form.field.ComboBox} combo
     */
    displayGroupedSuppliers: function (combo) {
        var globalSupplierStore = this.supplierStore;
        var comboSupplierStore = combo.getStore();
        var currentSelection = this.rightPanel.getSelectionModel().getSelection()[0];
        var assignedIds = currentSelection.getAssignedSupplierIds();
        var defaultSupplierId = currentSelection.getDefaultSupplierId();
        var supplierData = [];

        globalSupplierStore.each(function (record) {
            if (assignedIds.includes(record.data.id)) {
                supplierData.push(record);
            }
        });

        comboSupplierStore.each(function (record) {
            if (!assignedIds.includes(record.data.id)) {
                supplierData.push(record);
            }
        });

        supplierData.forEach(function (record) {
            record.data.assigned = false;
            record.data.default = false;
            if (assignedIds.includes(record.data.id)) {
                record.data.assigned = true;
                if (record.data.id === defaultSupplierId) {
                    record.data.default = true;
                }
            }
        });

        supplierData.sort(function (o1, o2) {
            var value1 = o1.get('assigned');
            var value2 = o2.get('assigned');

            return value2 - value1;
        });

        // For some reason when you switch from page 2 to 1 extjs tries to select a second item which collapses the
        // combobox. To prevent that collapse we save the last selected value allow multiselect and reverted the changes
        // after loading the new data.
        var lastSelection = combo.lastValue;
        combo.multiSelect = true;
        combo.store.loadData(supplierData);
        combo.multiSelect = false;
        combo.select(lastSelection);
    },
});
