// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.ArticleTooltip', {

    extend: 'Ext.tip.ToolTip',
    alias: 'widget.viison_pickware_erp_supplier_orders-detail-article_tooltip',
    cls: 'viison_pickware_erp_supplier_orders-detail-article_tooltip is--viison-common-tooltip',

    renderTo: Ext.getBody(),
    anchor: 'top',
    dismissDelay: 20000, // 20 sec
    showDelay: 0,
    shadow: false,

    /**
     * @property {Ext.grid.View}
     */
    articleGridView: null,

    /**
     * @property {Shopware.apps.ViisonPickwareERPSupplierOrders.model.Order}
     */
    supplierOrder: null,

    /**
     * @property {Shopware.apps.ViisonPickwareERPSupplierManagement.store.Supplier}
     */
    supplierStore: null,

    /**
     * @property {Shopware.apps.ViisonPickwareERPSupplierOrders.store.order.ArticleStatus}
     */
    itemStatusStore: null,

    /**
     * @Override
     */
    initComponent: function () {
        this.tpl = this.createTooltipTemplate();
        this.target = this.articleGridView.el;
        this.delegate = '.x-grid-cell.viison--show-tooltip';

        this.on('beforeshow', this.updateTooltip, this);

        this.callParent(arguments);
    },

    /**
     * Updates the tooltip by collecting all necessary data and rerendering the tooltip.
     *
     * @param {Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.ArticleTooltip} tooltip
     */
    updateTooltip: function (tooltip) {
        var rowElement = tooltip.triggerElement.parentElement;
        var rowRecord = this.articleGridView.getRecord(rowElement);
        var data = Ext.applyIf({}, rowRecord.data);

        // Prepare data
        var supplier = this.supplierStore.getById(rowRecord.get('supplierId'));
        data.supplierName = supplier ? supplier.get('name') : null;

        data.price = this.formatPurchasePriceForOrderOrSupplier(
            rowRecord.get('price') || 0.0,
            this.supplierOrder,
            supplier
        );
        data.totalPrice = this.formatPurchasePriceForOrderOrSupplier(
            rowRecord.get('totalPrice') || 0.0,
            this.supplierOrder,
            supplier
        );

        if (this.supplierOrder) {
            // Display status and delivered quantity for already ordered supplier orders only
            var statusRecord = this.itemStatusStore.getById(rowRecord.get('statusId'));
            data.status = (statusRecord) ? statusRecord.get('description') : rowRecord.get('statusId');
            data.showStatus = true;
            data.showDeliveredQuantity = true;
        }

        tooltip.update(data);
    },

    /**
     * Creates and returns the XTemplate used to render the article tooltip.
     *
     * @returns {Ext.XTemplate}
     */
    createTooltipTemplate: function () {
        /* eslint-disable indent */
        return new Ext.XTemplate(
            '<tpl for=".">',
                '<div class="is--tooltip-content">',
                '<div class="is--header is--large">{name}</div>',
                '<tpl if="showStatus">',
                    '<div class="is--row is--single-row">',
                        '<span class="is--label">',
                            ViisonCommonApp.getSnippet(
                                'edit/article_assignment/column/status',
                                'backend/viison_pickware_erp_supplier_orders/main'
                            ) + ':',
                        '</span>',
                        '<span class="is--value">',
                            '{status}',
                        '</span>',
                    '</div>',
                '</tpl>',
                '<div class="has--table-structure has--value-size-large">',
                    '<div class="is--table-cell">',
                        '<span class="is--label">',
                            ViisonCommonApp.getSnippet(
                                'edit/article_assignment/column/online_physical_stock',
                                'backend/viison_pickware_erp_supplier_orders/main'
                            ),
                        '</span>',
                        '<span class="is--value">',
                            '<tpl if="onlinePhysicalStock">',
                                '{onlinePhysicalStock}',
                            '<tpl else>',
                                '0',
                            '</tpl>',
                        '</span>',
                    '</div>',
                    '<div class="is--table-cell">',
                        '<span class="is--label">',
                            ViisonCommonApp.getSnippet(
                                'edit/article_assignment/column/online_available_stock',
                                'backend/viison_pickware_erp_supplier_orders/main'
                            ),
                        '</span>',
                        '<span class="is--value">',
                            '<tpl if="onlineAvailableStock">',
                                '{onlineAvailableStock}',
                            '<tpl else>',
                                '0',
                            '</tpl>',
                        '</span>',
                    '</div>',
                    '<tpl if="showDeliveredQuantity">',
                        '<div class="is--table-cell">',
                            '<span class="is--label">',
                                ViisonCommonApp.getSnippet(
                                    'edit/article_assignment/column/delivered_quantity',
                                    'backend/viison_pickware_erp_supplier_orders/main'
                                ),
                            '</span>',
                            '<span class="is--value">',
                                '<tpl if="deliveredQuantity">',
                                    '{deliveredQuantity}',
                                '<tpl else>',
                                    '0',
                                '</tpl>',
                            '</span>',
                        '</div>',
                    '</tpl>',
                    '<div class="is--table-cell">',
                        '<span class="is--label">',
                            ViisonCommonApp.getSnippet(
                                'edit/article_assignment/column/order_quantity',
                                'backend/viison_pickware_erp_supplier_orders/main'
                            ),
                        '</span>',
                        '<span class="is--value">',
                            '<tpl if="orderedQuantity">',
                                '{orderedQuantity}',
                            '<tpl else>',
                                '0',
                            '</tpl>',
                        '</span>',
                    '</div>',
                    '<div class="is--table-cell">',
                        '<span class="is--label">',
                            ViisonPickwarePurchasePriceHelper.purchasePriceLabel(
                                ViisonCommonApp.getSnippet(
                                    'edit/article_assignment/column/price',
                                    'backend/viison_pickware_erp_supplier_orders/main'
                                )
                            ),
                        '</span>',
                        '<span class="is--value">',
                            '{price}',
                        '</span>',
                    '</div>',
                    '<div class="is--table-cell">',
                        '<span class="is--label">',
                            ViisonPickwarePurchasePriceHelper.purchasePriceLabel(
                                ViisonCommonApp.getSnippet(
                                    'edit/article_assignment/column/total_price',
                                    'backend/viison_pickware_erp_supplier_orders/main'
                                )
                            ),
                        '</span>',
                        '<span class="is--value">',
                            '{totalPrice}',
                        '</span>',
                    '</div>',
                '</div>',
                '<div class="is--row has--divider has--padding">',
                    '<span class="is--label">',
                        ViisonCommonApp.getSnippet(
                            'edit/article_assignment/column/order_number',
                            'backend/viison_pickware_erp_supplier_orders/main'
                        ) + ':',
                    '</span>',
                    '<span class="is--value">{orderNumber}</span>',
                '</div>',
                '<div class="is--row has--divider has--padding">',
                    '<span class="is--label">',
                        ViisonCommonApp.getSnippet(
                            'edit/article_assignment/column/fabricator_name',
                            'backend/viison_pickware_erp_supplier_orders/main'
                        ) + ':',
                    '</span>',
                    '<span class="is--value">',
                        '<tpl if="manufacturerName">',
                            '{manufacturerName}',
                        '<tpl else>',
                            '-',
                        '</tpl>',
                    '</span>',
                '</div>',
                '<div class="is--row has--divider has--padding">',
                    '<span class="is--label">',
                        ViisonCommonApp.getSnippet(
                            'edit/article_assignment/column/fabricator_number',
                            'backend/viison_pickware_erp_supplier_orders/main'
                        ) + ':',
                    '</span>',
                    '<span class="is--value">',
                        '<tpl if="manufacturerArticleNumber">',
                            '{manufacturerArticleNumber}',
                        '<tpl else>',
                            '-',
                        '</tpl>',
                    '</span>',
                '</div>',
                '<div class="is--row has--divider has--padding">',
                    '<span class="is--label">',
                        ViisonCommonApp.getSnippet(
                            'edit/article_assignment/column/supplier',
                            'backend/viison_pickware_erp_supplier_orders/main'
                        ) + ':',
                    '</span>',
                    '<span class="is--value">',
                        '<tpl if="supplierName">',
                            '{supplierName}',
                        '<tpl else>',
                            '-',
                        '</tpl>',
                    '</span>',
                '</div>',
                '<div class="is--row has--padding">',
                    '<span class="is--label">',
                        ViisonCommonApp.getSnippet(
                            'edit/article_assignment/column/supplier_article_number',
                            'backend/viison_pickware_erp_supplier_orders/main'
                        ) + ':',
                    '</span>',
                    '<span class="is--value">',
                        '<tpl if="supplierArticleNumber">',
                            '{supplierArticleNumber}',
                        '<tpl else>',
                            '-',
                        '</tpl>',
                    '</span>',
                '</div>',
                '</div>',
            '</tpl>'
        );
        /* eslint-enable indent */
    },

    /**
     * Formats the given purchase price in the currency of the given order (if set), or the currency of the supplier, or
     * use the default currency, if no supplier or order is assigned.
     *
     * @param {number} purchasePrice
     * @param {Shopware.apps.ViisonPickwareERPSupplierOrders.model.Order} order
     * @param {Shopware.apps.ViisonPickwareERPSupplierManagement.model.Supplier} supplier
     * @returns {string}
     */
    formatPurchasePriceForOrderOrSupplier: function (purchasePrice, order, supplier) {
        var currency;
        if (order) {
            currency = order.getCurrency();
        } else if (supplier) {
            currency = supplier.getCurrency();
        } else {
            return ViisonCurrencyFormatter.renderer(purchasePrice);
        }

        return ViisonCurrencyFormatter.formatWithSymbolAndPosition(
            purchasePrice,
            currency.raw.symbol,
            currency.raw.symbolPosition
        );
    },

});
