// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.Configuration', {

    extend: 'Ext.container.Container',
    alias: 'widget.viison_pickware_erp_supplier_orders-detail-configuration',
    cls: 'viison_pickware_erp_supplier_orders-detail-configuration',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_supplier_orders/detail/configuration',

    layout: {
        type: 'hbox',
        align: 'stretch',
    },

    targetWarehouseStoreIsLoaded: false,
    sourceWarehousesStoreIsLoaded: false,

    /**
     * @property {Shopware.apps.ViisonPickwareERPWarehouseManagement.model.Warehouse|null}
     */
    targetWarehouse: null,
    orderStatus: null,

    /**
     * @Override
     */
    initComponent: function () {
        this.targetWarehouseStore = Ext.create('Shopware.apps.ViisonPickwareERPSupplierOrders.store.TargetWarehouses', {
            listeners: {
                scope: this,
                load: function () {
                    if (this.targetWarehouse === null) {
                        this.selectDefaultWarehouseAsTargetWarehouse(this.targetWarehouseComboBox);
                    }

                    this.targetWarehouseStoreIsLoaded = true;
                    this.checkWarehouseStoresLoaded();
                },
            },
        });
        this.targetWarehouseComboBox = Ext.create('Ext.form.field.ComboBox', {
            flex: 1,
            itemId: 'targetWarehouseComboBox',
            cls: 'is--pickware-erp--target-warehouse-combobox',
            store: this.targetWarehouseStore,
            queryMode: 'local', // Prevent combobox from reloading the store
            displayField: 'displayName',
            valueField: 'id',
            value: this.targetWarehouse !== null ? this.targetWarehouse.get('id') : null,
            disabled: this.orderStatus === 3 || this.orderStatus === 4,
            listeners: {
                scope: this,
                change: this.onTargetWarehouseSelectionChanged,
            },
        });

        this.sourceWarehousesStore = Ext.create('Shopware.apps.ViisonPickwareERPSupplierOrders.store.SourceWarehouses', {
            listeners: {
                scope: this,
                load: function () {
                    if (this.targetWarehouse === null) {
                        this.selectDefaultWarehouseAsTargetWarehouse(this.targetWarehouseComboBox);
                    }

                    this.sourceWarehousesStoreIsLoaded = true;
                    this.checkWarehouseStoresLoaded();
                },
            },
        });
        this.sourceWarehousesComboBox = Ext.create('Shopware.apps.ViisonPickwareERPSupplierCommon.view.WarehouseMultiselectionComboBox', {
            flex: 1,
            itemId: 'sourceWarehousesComboBox',
            store: this.sourceWarehousesStore,
            queryMode: 'local', // Prevent combobox from reloading the store
            value: 0, // Set first entry (ID = 0) as default value ??
            listeners: {
                scope: this,
                selectedWarehouseIdsChanged: this.onSourceWarehousesSelectionChanged,
            },
        });

        this.items = [
            {
                xtype: 'container',
                flex: 3,
                layout: {
                    type: 'vbox',
                    align: 'stretch',
                },
                padding: '13 20 8 15',
                style: {
                    borderRight: '1px solid #b7c7d1',
                    borderBottom: '1px solid #a4b5c0',
                    backgroundColor: '#dae0e5',
                },
                items: [
                    {
                        xtype: 'label',
                        cls: 'is--viison-common--label',
                        text: this.getViisonSnippet('target_warehouse/label') + ':',
                        margin: '8 12 8 3',
                    },
                    this.targetWarehouseComboBox,
                ],
            }, {
                xtype: 'container',
                flex: 8,
                layout: 'fit',
                style: {
                    borderBottom: '1px solid #a4b5c0',
                    backgroundColor: '#f5f7f8',
                },
                padding: '13 30 10 20',
                items: [
                    {
                        xtype: 'container',
                        layout: {
                            type: 'hbox',
                            align: 'stretch',
                        },
                        items: [
                            {
                                xtype: 'container',
                                flex: 1,
                                layout: {
                                    type: 'vbox',
                                    align: 'stretch',
                                },
                                padding: '0 20 0 0',
                                style: {
                                    borderRight: '1px solid #d2dfe8',
                                },
                                items: [
                                    {
                                        xtype: 'label',
                                        cls: 'is--viison-common--label',
                                        html: this.getViisonSnippet('source_warehouses/label') + ':',
                                        margin: '8 4 8 3',
                                    },
                                    this.sourceWarehousesComboBox,
                                ],
                            }, {
                                xtype: 'container',
                                layout: 'vbox',
                                margin: '3 0 0 0',
                                padding: '0 0 0 20',
                                items: [
                                    {
                                        xtype: 'checkbox',
                                        itemId: 'considerOpenOrdersCheckbox',
                                        cls: 'viison-common--checkbox-with-boxlabel',
                                        boxLabel: this.getViisonSnippet('consider_open_orders/label'),
                                        helpText: this.getViisonSnippet('consider_open_orders/helptext'),
                                        checked: true,
                                        uncheckedValue: false,
                                        listeners: {
                                            scope: this,
                                            change: this.onConsiderOpenOrdersChanged,
                                        },
                                    },
                                    {
                                        xtype: 'checkbox',
                                        itemId: 'useTotalIncomingStockCheckbox',
                                        cls: 'viison-common--checkbox-with-boxlabel',
                                        boxLabel: this.getViisonSnippet('use_total_incoming_stock/label'),
                                        helpText: this.getViisonSnippet('use_total_incoming_stock/helptext'),
                                        checked: false,
                                        uncheckedValue: false,
                                        listeners: {
                                            scope: this,
                                            change: this.onUseTotalIncomingStockChanged,
                                        },
                                    },
                                ],
                            },
                        ],
                    },
                ],
            },
        ];

        this.callParent(arguments);
    },

    /**
     * @param {Shopware.apps.ViisonPickwareERPSupplierCommon.view.WarehouseMultiselectionComboBox} combobox
     * @param {Array} selectedWarehouseIds
     */
    onSourceWarehousesSelectionChanged: function (combobox, selectedWarehouseIds) {
        this.blockEvents = true;
        if (this.sourceWarehouseSelectionIncludesWarehouseWhoseStockIsAvailableOnline(combobox, selectedWarehouseIds)) {
            this.down('#considerOpenOrdersCheckbox').setDisabled(false);
            this.down('#considerOpenOrdersCheckbox').setValue(true);
        } else {
            this.down('#considerOpenOrdersCheckbox').setDisabled(true);
            this.down('#considerOpenOrdersCheckbox').setValue(false);
        }
        this.blockEvents = false;

        this.fireEvent('sourceWarehouseSelectionChanged', this, selectedWarehouseIds);
    },

    /**
     * @param {Ext.form.field.ComboBox} combobox
     * @param {Number} selectedWarehouseId
     */
    onTargetWarehouseSelectionChanged: function (combobox, selectedWarehouseId) {
        this.fireEvent('targetWarehouseSelectionChanged', this, combobox.getStore().getById(selectedWarehouseId));
    },

    /**
     * @param {Ext.form.field.Checkbox} checkbox
     * @param {Boolean} newValue
     */
    onConsiderOpenOrdersChanged: function (checkbox, newValue) {
        if (!this.blockEvents) {
            this.fireEvent('considerOpenOrdersValueChanged', this, newValue);
        }
    },

    /**
     * @param {Ext.form.field.Checkbox} checkbox
     * @param {Boolean} newValue
     */
    onUseTotalIncomingStockChanged: function (checkbox, newValue) {
        this.fireEvent('useTotalIncomingStockValueChanged', this, newValue);
    },

    getSelectedTargetWarehouseId: function () {
        return this.down('#targetWarehouseComboBox').getValue();
    },

    getSelectedSourceWarehouseIds: function () {
        var selectedWarehouseIds = this.down('#sourceWarehousesComboBox').getValue();
        var warehouseStore = this.down('#sourceWarehousesComboBox').getStore();
        var expandedWarehouseIds = [];

        if (selectedWarehouseIds.length > 0) {
            Ext.Array.each(selectedWarehouseIds, function (warehouseId) {
                if (warehouseId === 0) {
                    expandedWarehouseIds = Ext.Array.merge(
                        expandedWarehouseIds,
                        warehouseStore.getIdsOfWarehousesWhoseStockIsAvailableOnline()
                    );
                } else {
                    expandedWarehouseIds = Ext.Array.merge(
                        expandedWarehouseIds,
                        [warehouseId]
                    );
                }
            });
        }

        return expandedWarehouseIds;
    },

    getConsiderOpenOrdersValue: function () {
        return this.down('#considerOpenOrdersCheckbox').getValue();
    },

    getUseTotalIncomingStockValue: function () {
        return this.down('#useTotalIncomingStockCheckbox').getValue();
    },

    getConfigurationValues: function () {
        return {
            targetWarehouse: this.getSelectedTargetWarehouseId(),
            sourceWarehouses: this.getSelectedSourceWarehouseIds(),
            considerOpenOrders: this.getConsiderOpenOrdersValue(),
            useTotalIncomingStock: this.getUseTotalIncomingStockValue(),
        };
    },

    /**
     * @param {Ext.form.field.ComboBox} combobox
     */
    selectDefaultWarehouseAsTargetWarehouse: function (combobox) {
        if (combobox !== null) {
            var defaultWarehouse = combobox.getStore().getDefaultWarehouse();
            if (defaultWarehouse) {
                combobox.setValue(defaultWarehouse.get('id'));
            }
        }
    },

    /**
     * @param {Shopware.apps.ViisonPickwareERPSupplierCommon.view.WarehouseMultiselectionComboBox} combobox
     * @param {Array} selectedWarehouseIds
     */
    sourceWarehouseSelectionIncludesWarehouseWhoseStockIsAvailableOnline: function (combobox, selectedWarehouseIds) {
        var hasWarehouseWhoseStockIsAvailableOnline = false;

        if (selectedWarehouseIds.length === 0) {
            // If no specific warehouse is selected, all warehouses are taken into consideration
            selectedWarehouseIds = combobox.getStore().getWarehouseIds();
        }

        Ext.each(selectedWarehouseIds, function (warehouseId) {
            Ext.each(combobox.store.data.items, function (warehouse) {
                if (warehouse.get('id') === warehouseId && warehouse.get('stockAvailableForSale')) {
                    hasWarehouseWhoseStockIsAvailableOnline = true;
                }
            }, this);
        }, this);

        return hasWarehouseWhoseStockIsAvailableOnline;
    },

    checkWarehouseStoresLoaded: function () {
        if (this.targetWarehouseStoreIsLoaded && this.sourceWarehousesStoreIsLoaded) {
            this.fireEvent('configurationPanelReady', this);
        }
    },

});

