// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPSupplierOrders.view.main.List', {

    extend: 'Ext.grid.Panel',
    alias: 'widget.viison_pickware_erp_supplier_orders-main-list',
    // Use a bug in ExtJS that removes the background of the row editor and hence makes all columns visible
    cls: 'x-tabpanel-child viison_pickware_erp_supplier_orders-main-list viison-common--grid has--vertical-lines has--no-borders',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_supplier_orders/main',

    border: true,
    autoScroll: true,
    viewConfig: {
        enableTextSelection: true,
    },

    /**
     * @Override
     */
    initComponent: function () {
        // Add a dynamic 'warning icon' to the rows, if the payment due date is exceeded without paying
        // or the delivery date is exceeded without complete delivery
        this.viewConfig = {
            getRowClass: function (record) {
                // Determine current date
                var today = new Date();
                today.setHours(0, 0, 0, 0);

                var showWarning = false;
                // Compare payment status and due date
                showWarning = showWarning || (record.get('paymentStatus') !== 1 && record.get('paymentDueDate') && today.getTime() > record.get('paymentDueDate').getTime());
                // Compare delivery status and date
                showWarning = showWarning || (record.get('statusId') < 4 && record.get('deliveryDate') && today.getTime() > record.get('deliveryDate').getTime());

                return (showWarning) ? 'x-grid-row-viison-warning' : '';
            },
        };

        // Add row editing
        this.plugins = [
            Ext.create('Ext.grid.plugin.RowEditing', {
                clicksToMoveEditor: 1,
                autoCancel: false,
                listeners: {
                    edit: function (editor, event) {
                        // Check if the user tried to change the status of a completely delivered order. We do not
                        // support this since usually stock entries have been generated for a completely delivered order
                        // (the user has the choice)
                        if (event.originalValues.statusId === 4 && event.newValues.statusId !== event.originalValues.statusId) {
                            // Reset the state to 'completely delivered' and show a notification to the user
                            event.record.set('statusId', 4);
                            Shopware.Msg.createGrowlMessage(
                                this.getViisonSnippet('notification/error/title'),
                                this.getViisonSnippet('notification/error/message/status_change_rejected'),
                                'Pickware'
                            );
                        }

                        // Set the date values again to work around wrong date values (2015-05-01 vs. 2015-01-05)
                        event.record.set('deliveryDate', event.newValues.deliveryDate);
                        event.record.set('paymentDueDate', event.newValues.paymentDueDate);
                        this.fireEvent('saveOrder', event.record, event.originalValues);
                    },
                    scope: this,
                },
            }),
        ];

        this.columns = this.createColumns();
        this.dockedItems = this.createDockedItems();

        this.callParent(arguments);
    },

    /**
     * @return Ext.grid.column.Column[]
     */
    createColumns: function () {
        return [{
            xtype: 'datecolumn',
            header: this.getViisonSnippet('order/field/created'),
            dataIndex: 'created',
            width: 130,
            format: 'd.m.Y H:i',
        }, {
            xtype: 'gridcolumn',
            header: this.getViisonSnippet('order/field/order_number'),
            dataIndex: 'orderNumber',
            width: 85,
        }, {
            xtype: 'gridcolumn',
            header: this.getViisonSnippet('order/field/supplier_name'),
            dataIndex: 'supplierId',
            flex: 3,
            renderer: function (value, meta, record) {
                return (record.raw.supplier) ? record.getSupplier().get('name') : this.getViisonSnippet('list/column/supplier_name/unknown');
            },
        }, {
            xtype: 'gridcolumn',
            header: this.getViisonSnippet('order/field/warehouse_name'),
            dataIndex: 'warehouseId',
            flex: 2,
            scope: this,
            renderer: function (value) {
                var warehouseIndex = this.warehouseStore.findExact('id', value);

                return (warehouseIndex > -1) ? this.warehouseStore.getAt(warehouseIndex).get('displayName') : this.getViisonSnippet('list/column/warehouse_name/unknown');
            },
        }, {
            xtype: 'gridcolumn',
            header: this.getViisonSnippet('order/field/status'),
            dataIndex: 'statusId',
            flex: 2,
            editor: {
                xtype: 'combo',
                store: this.orderStatusStore,
                valueField: 'id',
                displayField: 'description',
                queryMode: 'local',
                selectOnTab: true,
                typeAhead: true,
                triggerAction: 'all',
            },
            renderer: function (value) {
                var record = this.orderStatusStore.getById(value);

                return (record === null) ? '' : record.get('description');
            },
        }, {
            xtype: 'datecolumn',
            header: this.getViisonSnippet('order/field/delivery_date'),
            dataIndex: 'deliveryDate',
            width: 80,
            format: 'd.m.Y',
            editor: {
                xtype: 'datefield',
                allowBlank: true,
            },
        }, {
            header: this.getViisonSnippet('order/field/total'),
            dataIndex: 'total',
            width: 90,
            align: 'right',
            scope: this,
            renderer: function (value, meta, record) {
                var currency;
                try {
                    // When creating a supplier order with an article that had no supplier assigned by default
                    // (unassigned article), the order is placed in the store which re-renders the grid. At some point
                    // the grid is then rendered where the currency of the order ist not persisted correctly and the
                    // getCurrency() call fails (does not return null. fails.). The store is then reloaded at another
                    // time which fixes this. I do not know how to prevent this bug other than working around it.
                    currency = record.getCurrency();
                } catch (exception) {
                    return ViisonCurrencyFormatter.renderer(value);
                }

                return ViisonCurrencyFormatter.formatWithSymbolAndPosition(
                    value,
                    currency.raw.symbol,
                    currency.raw.symbolPosition
                );
            },
        }, {
            header: this.getViisonSnippet('order/field/total_in_default_currency'),
            dataIndex: 'totalInDefaultCurrency',
            width: 90,
            align: 'right',
            hidden: true,
            sortable: false,
            renderer: ViisonCurrencyFormatter.renderer,
        }, {
            xtype: 'gridcolumn',
            header: this.getViisonSnippet('order/field/payment_status'),
            dataIndex: 'paymentStatus',
            width: 90,
            editor: {
                xtype: 'combo',
                store: this.paymentStatusStore,
                valueField: 'id',
                displayField: 'description',
                queryMode: 'local',
                selectOnTab: true,
            },
            renderer: function (value) {
                var record = this.paymentStatusStore.getById(value);

                return (record === null) ? '' : record.get('description');
            },
        }, {
            xtype: 'datecolumn',
            header: this.getViisonSnippet('order/field/payment_due_date'),
            dataIndex: 'paymentDueDate',
            width: 80,
            format: 'd.m.Y',
            editor: {
                xtype: 'datefield',
                allowBlank: true,
            },
        }, {
            xtype: 'actioncolumn',
            hideable: false,
            align: 'center',
            width: 135,
            items: [{
                iconCls: 'sprite-mail-send',
                action: 'sendOrderMail',
                tooltip: this.getViisonSnippet('list/tooltip/send_mail'),
                handler: function (grid, rowIndex) {
                    this.fireEvent('sendOrderMail', grid.getStore().getAt(rowIndex));
                },
                scope: this,
            }, {
                iconCls: 'sprite-inbox-download',
                action: 'downloadOrderPDF',
                tooltip: this.getViisonSnippet('list/tooltip/download_pdf'),
                handler: function (grid, rowIndex) {
                    this.fireEvent('downloadOrderPDF', grid.getStore().getAt(rowIndex));
                },
                scope: this,
            }, {
                iconCls: 'sprite-arrow-continue-270-left',
                action: 'downloadOrderCSV',
                tooltip: this.getViisonSnippet('list/tooltip/download_csv'),
                handler: function (grid, rowIndex) {
                    this.fireEvent('downloadOrderCSV', grid.getStore().getAt(rowIndex));
                },
                scope: this,
            }, {
                iconCls: 'sprite-pencil',
                action: 'editOrder',
                tooltip: this.getViisonSnippet('list/tooltip/edit'),
                handler: function (grid, rowIndex) {
                    this.fireEvent('editOrder', grid.getStore().getAt(rowIndex));
                },
                scope: this,
            }, {
                iconCls: 'sprite-minus-circle',
                action: 'deleteOrder',
                tooltip: this.getViisonSnippet('list/tooltip/delete'),
                handler: function (grid, rowIndex) {
                    this.fireEvent('deleteOrder', grid.getStore().getAt(rowIndex));
                },
                scope: this,
            }],
        }];
    },

    /**
     * Creates a button toolbar located at the top as well as a paging toolbar located at the bottom.
     *
     * @return Ext.toolbar.Toolbar[]
     */
    createDockedItems: function () {
        // Create a toolbar located at the top, containing a button for creating a new order
        var topToolbar = Ext.create('Ext.toolbar.Toolbar', {
            dock: 'top',
            ui: 'shopware-ui is--viison-common--toolbar',
            items: [{
                xtype: 'button',
                text: this.getViisonSnippet('list/toolbar/button/add'),
                iconCls: 'sprite-plus-circle-frame',
                action: 'addOrder',
            }, {
                xtype: 'tbspacer',
                width: 20,
            }, '->', {
                xtype: 'textfield',
                name: 'searchfield',
                cls: 'searchfield',
                width: 170,
                enableKeyEvents: true,
                emptyText: this.getViisonSnippet('search_field/placeholder'),
                listeners: {
                    buffer: 500,
                    change: function (field) {
                        this.fireEvent('searchFieldChanged', field.value);
                    },
                    scope: this,
                },
            }],
        });

        // Create a paging toolbar located at the bottom
        var bottomToolbar = {
            xtype: 'viison_common_pagination_toolbar-toolbar',
            store: this.store,
        };

        return [
            topToolbar,
            bottomToolbar,
        ];
    },

});
