// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPWarehouseManagement.controller.BinLocationCreation', {

    extend: 'Ext.app.Controller',
    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_warehouse_management/main',

    /**
     * @Override
     */
    init: function () {
        this.control({
            'viison_pickware_erp_warehouse_management-edit-bin_locations-bin_location_creation button[action=save]': {
                click: this.onSave,
            },
            'viison_pickware_erp_warehouse_management-edit-bin_locations-bin_location_creation button[action=cancel]': {
                click: this.onCancel,
            },
        });

        this.callParent(arguments);
    },

    /**
     * Creates and shows a new window for creating a single bin location, which is passed the given
     * warehouse, binLocationStore and optional callback.
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.model.Warehouse warehouse
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.store.BinLocation binLocationStore
     * @param Function callback (optional)
     */
    createWindow: function (warehouse, binLocationStore, callback) {
        Ext.create('Shopware.apps.ViisonPickwareERPWarehouseManagement.view.edit.BinLocations.BinLocationCreation', {
            warehouse: warehouse,
            binLocationStore: binLocationStore,
            saveCallback: callback || Ext.emptyFn,
        }).show();
    },

    /**
     * Closes the window, which contains the given button.
     *
     * @param Ext.button.Button button
     */
    onSave: function (button) {
        var creationWindow = button.up('viison_pickware_erp_warehouse_management-edit-bin_locations-bin_location_creation');
        var form = creationWindow.down('form').getForm();
        if (!form.isValid()) {
            return;
        }

        // Create a new bin location using the form data
        var binLocationCode = form.getValues().code;
        var binLocation = Ext.create('Shopware.apps.ViisonPickwareERPWarehouseManagement.model.BinLocation', {});
        binLocation.set('code', binLocationCode);
        binLocation.set('warehouseId', creationWindow.warehouse.get('id'));

        // Add the new bin location to the store and save it
        creationWindow.setLoading(true);
        creationWindow.binLocationStore.add(binLocation);
        creationWindow.binLocationStore.sync({
            scope: this,
            callback: function (batch) {
                creationWindow.setLoading(false);
                // Check the response
                var operation = batch.operations[0];
                if (operation.success) {
                    creationWindow.close();
                    var successMessage = Ext.String.format(this.getViisonSnippet('notification/success/create_bin_location/message'), binLocationCode);
                    Shopware.Notification.createGrowlMessage(this.getViisonSnippet('notification/success/create_bin_location/title'), successMessage, 'ViisonPickwareERP');
                } else {
                    // Revert changes and show error notification
                    creationWindow.binLocationStore.rejectChanges();
                    var duplicateLocation = operation.request.proxy.reader.jsonData.uniqueConstraintViolation;
                    var errorMessage = (duplicateLocation) ? this.getViisonSnippet('notification/error/message/create_bin_location/duplicate_location') : this.getViisonSnippet('notification/error/message/create_bin_location/unknown_error');
                    errorMessage = Ext.String.format(errorMessage, binLocationCode);
                    Shopware.Notification.createGrowlMessage(this.getViisonSnippet('notification/error/title'), errorMessage, 'ViisonPickwareERP');
                }
            },
        });
    },

    /**
     * Closes the window, which contains the given button.
     *
     * @param Ext.button.Button button
     */
    onCancel: function (button) {
        button.up('viison_pickware_erp_warehouse_management-edit-bin_locations-bin_location_creation').close();
    },

});
