// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPWarehouseManagement.controller.BinLocationGenerator', {

    extend: 'Ext.app.Controller',
    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_warehouse_management/main',

    /**
     * @Override
     */
    init: function () {
        this.control({
            'viison_pickware_erp_warehouse_management-edit-bin_locations-bin_location_generator': {
                addComponent: this.onAddComponent,
                deleteComponent: this.onDeleteComponent,
                formChanged: Ext.Function.createBuffered(this.onFormChanged, 1000, this),
            },
            'viison_pickware_erp_warehouse_management-edit-bin_locations-bin_location_generator button[action=save]': {
                click: this.onSave,
            },
            'viison_pickware_erp_warehouse_management-edit-bin_locations-bin_location_generator button[action=cancel]': {
                click: this.onCancel,
            },
            'viison_pickware_erp_warehouse_management-edit-bin_locations-bin_location_generator button[action=generateBinLocations]': {
                click: this.onGenerateBinLocations,
            },
        });

        this.callParent(arguments);
    },

    /**
     * Creates and shows a new window for generating bin locations, which is passed the given
     * warehouse and optional callback.
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.model.Warehouse warehouse
     * @param Function callback (optional)
     */
    createWindow: function (warehouse, callback) {
        // Create a store containing copies of the current format components of the warehouse
        var formatComponents = [];
        warehouse.binLocationFormatComponents().each(function (formatComponent) {
            formatComponents.push(Ext.create('Shopware.apps.ViisonPickwareERPWarehouseManagement.model.BinLocationFormatComponent', formatComponent.raw));
        });
        var formatComponentStore = Ext.create('Ext.data.Store', {
            model: 'Shopware.apps.ViisonPickwareERPWarehouseManagement.model.BinLocationFormatComponent',
            data: formatComponents,
        });
        if (formatComponentStore.count() === 0) {
            this.addNewComponentToStore(formatComponentStore, 0);
        }

        // Create and show the window
        Ext.create('Shopware.apps.ViisonPickwareERPWarehouseManagement.view.edit.BinLocations.BinLocationGenerator', {
            warehouse: warehouse,
            store: formatComponentStore,
            saveCallback: callback || Ext.emptyFn,
        }).show();
    },

    /**
     * Creates a new bin location format component instance, adds it to the store and
     * renders a form in the new grid row.
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.view.edit.BinLocations.BinLocationGenerator generatorWindow
     * @param Number index
     */
    onAddComponent: function (generatorWindow, index) {
        // Insert a new entry into the store
        var newRecord = this.addNewComponentToStore(generatorWindow.store, index);

        // Explicitely render the row for the new record
        generatorWindow.renderRow(newRecord);
    },

    /**
     * Creates a new bin location format component instance and inserts it into the store.
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.store.BinLocationFormatComponent store
     * @param Number index
     * @return Shopware.apps.ViisonPickwareERPWarehouseManagement.model.BinLocationFormatComponent
     */
    addNewComponentToStore: function (store, index) {
        var prevIndex = Math.max(0, index - 1);
        var recordData = {
            type: (!store.getAt(prevIndex) || store.getAt(prevIndex).get('type') === 'fixed') ? 'letters' : 'fixed',
            length: 1,
        };
        var newRecord = Ext.create('Shopware.apps.ViisonPickwareERPWarehouseManagement.model.BinLocationFormatComponent', recordData);
        store.insert(index, [
            newRecord,
        ]);

        return newRecord;
    },

    /**
     * Removes the given record from the store and reloads the example location data.
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.view.edit.BinLocations.BinLocationGenerator generatorWindow
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.model.BinLocationFormatComponent record
     */
    onDeleteComponent: function (generatorWindow, record) {
        // Remove the record and its form from the store and grid respectively
        generatorWindow.store.remove(record);
        delete generatorWindow.rowForms[record.internalId];

        // Reload example location
        this.onFormChanged(generatorWindow);
    },

    /**
     * Performs a basic validation on all forms that are contained in the bin location
     * generator grid and, if all are valid, sends a request to retrieve an example location
     * as well as the possible number of locations for the given format, which are then
     * displayed in the generator window.
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.view.edit.BinLocations.BinLocationGenerator generatorWindow
     */
    onFormChanged: function (generatorWindow) {
        // Check for any components
        if (generatorWindow.store.count() === 0) {
            generatorWindow.updateExampleBinLocationLabels({
                success: true,
                binLocation: '-',
                numPossibleLocations: '-',
            });

            return;
        }

        // Validate all forms and don't request an example location if not all fields are valid
        if (!this.validateFormatComponentForms(generatorWindow.rowForms)) {
            return;
        }

        // Request example bin location
        Ext.Ajax.request({
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPWarehouseManagement/getExampleBinLocation'),
            method: 'POST',
            jsonData: {
                components: this.getFormatComponentFormData(generatorWindow),
            },
            scope: this,
            success: function (response) {
                var responseData = Ext.JSON.decode(response.responseText, true);
                if (responseData) {
                    generatorWindow.updateExampleBinLocationLabels(responseData);
                    if (typeof responseData.numPossibleLocations !== 'undefined') {
                        this.estimatedNumberOfPossibleBinLocations = responseData.numPossibleLocations;
                    }
                } else {
                    Shopware.Notification.createGrowlMessage(this.getViisonSnippet('notification/error/title'), this.getViisonSnippet('notification/error/message/get_example_bin_location'), 'ViisonPickwareERPWarehouseManagement');
                }
            },
        });
    },

    /**
     * Performs a basic validation on all forms that are contained in the bin location
     * generator grid and, if all are valid, sends a request to save the current format
     * components in the warehouse that is being edited.
     *
     * @param Ext.button.Button button
     */
    onSave: function (button) {
        var generatorWindow = button.up('viison_pickware_erp_warehouse_management-edit-bin_locations-bin_location_generator');
        if (!this.validateFormatComponentForms(generatorWindow.rowForms)) {
            Shopware.Notification.createGrowlMessage(this.getViisonSnippet('notification/error/title'), this.getViisonSnippet('notification/error/message/save_bin_location_format/validation'), 'ViisonPickwareERPWarehouseManagement');

            return;
        }

        // POST all components
        Ext.Ajax.request({
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPWarehouseManagement/saveBinLocationFormatComponents'),
            method: 'PUT',
            jsonData: {
                warehouseId: generatorWindow.warehouse.get('id'),
                components: this.getFormatComponentFormData(generatorWindow),
            },
            scope: this,
            success: function (response) {
                var responseData = Ext.JSON.decode(response.responseText, true);
                if (responseData && responseData.success) {
                    Shopware.Notification.createGrowlMessage(this.getViisonSnippet('notification/success/save_bin_location_format/title'), this.getViisonSnippet('notification/success/save_bin_location_format/message'), 'ViisonPickwareERPWarehouseManagement');

                    // Copy all components to the warehouse record's store
                    generatorWindow.warehouse.binLocationFormatComponents().removeAll();
                    generatorWindow.store.each(function (formatComponent) {
                        generatorWindow.warehouse.binLocationFormatComponents().add(Ext.create('Shopware.apps.ViisonPickwareERPWarehouseManagement.model.BinLocationFormatComponent', formatComponent.raw));
                    });
                } else {
                    Shopware.Notification.createGrowlMessage(this.getViisonSnippet('notification/error/title'), this.getViisonSnippet('notification/error/message/save_bin_location_format'), 'ViisonPickwareERPWarehouseManagement');
                }
            },
        });
    },

    /**
     * Closes the window, which contains the given button.
     *
     * @param Ext.button.Button button
     */
    onCancel: function (button) {
        button.up('viison_pickware_erp_warehouse_management-edit-bin_locations-bin_location_generator').close();
    },

    /**
     * Performs a basic validation on all forms that are contained in the bin location
     * generator grid and, if all are valid, sends a request to save the current format
     * components in the warehouse that is being edited.
     *
     * @param Ext.button.Button button
     */
    onGenerateBinLocations: function (button) {
        var generatorWindow = button.up('viison_pickware_erp_warehouse_management-edit-bin_locations-bin_location_generator');
        if (!this.validateFormatComponentForms(generatorWindow.rowForms)) {
            Shopware.Notification.createGrowlMessage(this.getViisonSnippet('notification/error/title'), this.getViisonSnippet('notification/error/message/generate_bin_locations/validation'), 'ViisonPickwareERPWarehouseManagement');

            return;
        }

        // Ask for confirmation
        Ext.Msg.confirm(this.getViisonSnippet('alert/title'), Ext.String.format(this.getViisonSnippet('alert/message/generate_bin_locations'), this.estimatedNumberOfPossibleBinLocations), function (buttonId) {
            if (buttonId !== 'yes') {
                return;
            }

            // POST all components
            generatorWindow.setLoading(true);
            Ext.Ajax.request({
                url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPWarehouseManagement/generateBinLocations'),
                method: 'POST',
                jsonData: {
                    warehouseId: generatorWindow.warehouse.get('id'),
                    components: this.getFormatComponentFormData(generatorWindow),
                },
                scope: this,
                success: function (response) {
                    generatorWindow.setLoading(false);
                    var responseData = Ext.JSON.decode(response.responseText, true);
                    if (responseData && responseData.success) {
                        var message = Ext.String.format(this.getViisonSnippet('notification/success/generate_bin_locations/message'), responseData.numGeneratedBinLocations);
                        if (responseData.numSkippedBinLocations) {
                            message += ' ' + Ext.String.format(this.getViisonSnippet('notification/success/generate_bin_locations/message/skipped_locations_addition'), responseData.numSkippedBinLocations);
                        }
                        Shopware.Notification.createGrowlMessage(this.getViisonSnippet('notification/success/generate_bin_locations/title'), message, 'ViisonPickwareERPWarehouseManagement');
                    } else {
                        Shopware.Notification.createGrowlMessage(this.getViisonSnippet('notification/error/title'), this.getViisonSnippet('notification/error/message/generate_bin_locations'), 'ViisonPickwareERPWarehouseManagement');
                    }
                },
                failure: function () {
                    generatorWindow.setLoading(false);
                    Shopware.Notification.createGrowlMessage(this.getViisonSnippet('notification/error/title'), this.getViisonSnippet('notification/error/message/generate_bin_locations'), 'ViisonPickwareERPWarehouseManagement');
                },
            });
        }, this);
    },

    /**
     * Checks the validity of each of the given forms.
     *
     * @param Ext.form.Panel[] forms
     * @return boolean
     */
    validateFormatComponentForms: function (forms) {
        var valid = true;
        Ext.Object.each(forms, function (key, form) {
            valid = valid && form.getForm().isValid();
        });

        return valid;
    },

    /**
     * Finds the form for each record of the given window's store and packs their
     * values in an array, which is returned.
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.view.edit.BinLocations.BinLocationGenerator generatorWindow
     * @return object[]
     */
    getFormatComponentFormData: function (generatorWindow) {
        var componentData = [];
        generatorWindow.store.each(function (record) {
            var values = generatorWindow.rowForms[record.internalId].getForm().getValues();
            values.leadingZeros = !!values.leadingZeros;
            componentData.push(values);
        });

        return componentData;
    },

});
