// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPWarehouseManagement.controller.Edit', {

    extend: 'Ext.app.Controller',
    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_warehouse_management/main',
    /**
     * Contains the lastly calculated number of possible bin locations to be used
     * in the confirmation alert that is shown before generating any bin locations.
     */
    estimatedNumberOfPossibleBinLocations: 0,

    /**
     * @Override
     */
    init: function () {
        this.control({
            'viison_pickware_erp_warehouse_management-edit-base button[action=cancel]': {
                click: this.onCancel,
            },
            'viison_pickware_erp_warehouse_management-edit-base button[action=save]': {
                click: this.onSaveBase,
            },
            'viison_pickware_erp_warehouse_management-edit-bin_locations': {
                searchFieldChanged: this.onBinLocationSearchFieldChanged,
            },
            'viison_pickware_erp_warehouse_management-edit-bin_locations button[action=addBinLocation]': {
                click: this.onAddBinLocation,
            },
            'viison_pickware_erp_warehouse_management-edit-bin_locations button[action=generateBinLocations]': {
                click: this.onGenerateBinLocations,
            },
            'viison_pickware_erp_warehouse_management-edit-bin_locations button[action=deleteSelectedBinLocations]': {
                click: this.onDeleteSelectedBinLocations,
            },
            'viison_pickware_erp_warehouse_management-edit-bin_locations-bin_location_list': {
                binLocationSelectionChanged: this.onBinLocationSelectionChanged,
                saveBinLocation: this.onSaveBinLocation,
                deleteBinLocation: this.onDeleteBinLocation,
            },
            'viison_pickware_erp_warehouse_management-edit-bin_locations-article_detail_mapping_list': {
                showArticle: this.onShowArticle,
            },
        });

        this.callParent(arguments);
    },

    /**
     * Creates a new edit window for the given warehouse record.
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.model.Warehouse record
     * @return Shopware.apps.ViisonPickwareERPWarehouseManagement.view.Edit
     */
    createEditWindow: function (record) {
        // Create the edit window
        var title = (record.get('name').length > 0) ? (this.getViisonSnippet('edit/window/title/edit') + ' - ' + record.get('name')) : this.getViisonSnippet('edit/window/title/add');
        var editWindow = this.getView('Edit').create({
            title: title,
            record: record,
        });

        return editWindow;
    },

    /**
     * Closes the edit window the given button is part of.
     *
     * @param Ext.button.Button button
     */
    onCancel: function (button) {
        button.up('viison_pickware_erp_warehouse_management-edit').close();
    },

    /**
     * Saves the base data in the warehouse record associated with the window,
     * the given button is part of.
     *
     * @param Ext.button.Button button
     */
    onSaveBase: function (button) {
        var baseForm = button.up('viison_pickware_erp_warehouse_management-edit-base');
        if (!baseForm.getForm().isValid()) {
            return;
        }

        // Update the edited record with the form date
        baseForm.getForm().updateRecord(baseForm.record);
        // Set boolean/checkbox values manually to work around a bug in Ext JS
        baseForm.record.set('stockAvailableForSale', baseForm.getForm().getFieldValues().stockAvailableForSale);
        baseForm.record.set('defaultWarehouse', baseForm.getForm().getFieldValues().defaultWarehouse);
        baseForm.record.set('defaultReturnShipmentWarehouse', baseForm.getForm().getFieldValues().defaultReturnShipmentWarehouse);

        // Sync the changes to the server
        if (baseForm.record.phantom) {
            // Add record to store first
            this.getController('Main').warehouseStore.add(baseForm.record);
        }
        this.getController('Main').syncWarehouses(this.getViisonSnippet('notification/error/message/save'), function (success, jsonResponse) {
            if (success) {
                button.up('viison_pickware_erp_warehouse_management-edit').close();
            } else if (jsonResponse.uniqueConstraintViolation) {
                Shopware.Notification.createGrowlMessage(this.getViisonSnippet('notification/error/title'), this.getViisonSnippet('notification/error/message/save/unique_constraint_violation'), 'ViisonPickwareERPWarehouseManagement');
            }
        });
    },

    /**
     * Saves the query in panel's bin location store and reloads its data.
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.view.edit.BinLocations panel
     * @param String query
     */
    onBinLocationSearchFieldChanged: function (panel, query) {
        var store = panel.down('viison_pickware_erp_warehouse_management-edit-bin_locations-bin_location_list').store;
        store.getProxy().extraParams.query = query;
        store.loadPage(1);
    },

    /**
     * Creates a new bin location creation window for the warehouse that is being edited.
     *
     * @param Ext.button.Button button
     */
    onAddBinLocation: function (button) {
        var mainPanel = button.up('viison_pickware_erp_warehouse_management-edit-bin_locations');
        var listPanel = mainPanel.down('viison_pickware_erp_warehouse_management-edit-bin_locations-bin_location_list');
        this.getController('BinLocationCreation').createWindow(mainPanel.record, listPanel.store);
    },

    /**
     * Creates a new bin location generator window for the warehouse that is being edited.
     *
     * @param Ext.button.Button button
     */
    onGenerateBinLocations: function (button) {
        var mainPanel = button.up('viison_pickware_erp_warehouse_management-edit-bin_locations');
        this.getController('BinLocationGenerator').createWindow(mainPanel.record);
    },

    /**
     * Ask the user for confirmation of the deletion and, if given, deletes all selected bin location from
     * the store and syncs the changes.
     *
     * @param Ext.button.Button button
     */
    onDeleteSelectedBinLocations: function (button) {
        var listPanel = button.up('viison_pickware_erp_warehouse_management-edit-bin_locations').down('viison_pickware_erp_warehouse_management-edit-bin_locations-bin_location_list');
        // Ask for confirmation
        Ext.Msg.confirm(this.getViisonSnippet('alert/title'), (this.getViisonSnippet('alert/message/delete_selected_bin_locations') + ' ' + this.getViisonSnippet('alert/message/delete_bin_location/info')), function (buttonId) {
            if (buttonId !== 'yes') {
                return;
            }

            // Remove the given bin location from the store and sync the changes
            var selectedBinLocations = listPanel.selModel.getSelection();
            listPanel.store.remove(selectedBinLocations);
            this.syncBinLocations(listPanel, this.getViisonSnippet('notification/error/message/delete_selected_bin_locations'));
        }, this);
    },

    /**
     * Checks the new selection and updates the UI accordingly. That is, the detail panel is reloaded
     * if exactly one row is selected and cleared otherwise. Finally the 'delete' button is enabled
     * or disabled if at least one record is selected or no records are selected respectively.
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.view.edit.BinLocations.BinLocationList panel
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.model.BinLocation[] selection
     */
    onBinLocationSelectionChanged: function (panel, selection) {
        var mainPanel = panel.up('viison_pickware_erp_warehouse_management-edit-bin_locations');
        // Update the article detail mapping grid
        var mappingListPanel = mainPanel.down('viison_pickware_erp_warehouse_management-edit-bin_locations-article_detail_mapping_list');
        mappingListPanel.store.clearFilter(true);
        if (selection.length === 1) {
            mappingListPanel.setDisabled(false);
            // Filter the store to show the mappings of the selected location
            mappingListPanel.store.filter([
                { property: 'binLocationMapping.binLocationId', value: selection[0].get('id') },
            ]);
        } else {
            mappingListPanel.setDisabled(true);
            // Clear the store
            mappingListPanel.store.removeAll();
        }

        // Enable/disable the 'delete' button based on the selection
        mainPanel.down('button[action=deleteSelectedBinLocations]').setDisabled(selection.length === 0);
    },

    /**
     * TODO
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.view.edit.BinLocations.BinLocationList panel
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.model.BinLocation binLocation
     */
    onSaveBinLocation: function (panel, binLocation) {
        var newCode = binLocation.get('code');
        var mainPanel = panel.up('viison_pickware_erp_warehouse_management-edit-bin_locations');
        mainPanel.setLoading(true);
        panel.store.sync({
            scope: this,
            success: function () {
                mainPanel.setLoading(false);
            },
            failure: function (batch) {
                mainPanel.setLoading(false);
                // Revert any changes like added, updated or removed records
                panel.store.rejectChanges();
                // Display an error notification
                var message = this.getViisonSnippet('notification/error/message/save_bin_location');
                if (batch.proxy.reader.jsonData && batch.proxy.reader.jsonData.uniqueConstraintViolation) {
                    message = Ext.String.format(this.getViisonSnippet('notification/error/message/save_bin_location/unique_constraint_violation'), mainPanel.record.get('displayName'), newCode);
                }
                Shopware.Notification.createGrowlMessage(this.getViisonSnippet('notification/error/title'), message, 'ViisonPickwareERPWarehouseManagement');
            },
        });
    },

    /**
     * Ask the user for confirmation of the deletion and, if given, deletes the given bin location from
     * the store and syncs the changes.
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.view.edit.BinLocations.BinLocationList panel
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.model.BinLocation binLocation
     */
    onDeleteBinLocation: function (panel, binLocation) {
        // Ask for confirmation
        Ext.Msg.confirm('Achtung', Ext.String.format((this.getViisonSnippet('alert/message/delete_bin_location') + ' ' + this.getViisonSnippet('alert/message/delete_bin_location/info')), binLocation.get('code')), function (buttonId) {
            if (buttonId !== 'yes') {
                return;
            }

            // Remove the given bin location from the store and sync the changes
            panel.store.remove(binLocation);
            this.syncBinLocations(panel, this.getViisonSnippet('notification/error/message/delete_bin_location'));
        }, this);
    },

    /**
     * Launches a new article sub application showing the details of the given
      *article detail mapping record.
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.view.edit.BinLocations.ArticleDetailMappingList panel
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.model.ArticleDetailMapping articleDetailMapping
     */
    onShowArticle: function (panel, articleDetailMapping) {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.Article',
            action: 'detail',
            params: {
                articleId: articleDetailMapping.get('articleId'),
            },
        });
    },

    /**
     * Synchronizes the warehouses store and, in case of failure, shows a growl notification.
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.view.edit.BinLocations.BinLocationList listPanel
     * @param string errorMessage
     * @param function callback (optional)
     */
    syncBinLocations: function (listPanel, errorMessage, callback) {
        // Check the store for any changes
        var numChanges = listPanel.store.getNewRecords().length + listPanel.store.getModifiedRecords().length + listPanel.store.getRemovedRecords().length;
        if (numChanges === 0) {
            if (Ext.isFunction(callback)) {
                callback(true);
            }

            return;
        }

        // Sync the store
        listPanel.selModel.deselectAll();
        listPanel.setLoading(true);
        listPanel.store.sync({
            scope: this,
            success: function () {
                listPanel.setLoading(false);
                if (Ext.isFunction(callback)) {
                    callback(true);
                }
            },
            failure: function (batch) {
                listPanel.setLoading(false);
                // Revert any changes like added, updated or removed records
                listPanel.store.rejectChanges();
                // Display an error notification
                var message = errorMessage || '';
                Shopware.Notification.createGrowlMessage(this.getViisonSnippet('notification/error/title'), message, 'ViisonPickwareERPWarehouseManagement');
                if (Ext.isFunction(callback)) {
                    callback(false, batch.proxy.reader.jsonData);
                }
            },
        });
    },

});
