// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPWarehouseManagement.controller.Main', {

    extend: 'Ext.app.Controller',
    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    refs: [
        { ref: 'detailView', selector: 'viison_pickware_erp_warehouse_management-main-details' },
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_warehouse_management/main',

    /**
     * @Override
     */
    init: function () {
        this.control({
            'viison_pickware_erp_warehouse_management-main-list button[action=addWarehouse]': {
                click: this.onAddWarehouse,
            },
            'viison_pickware_erp_warehouse_management-main-list': {
                editWarehouse: this.onEditWarehouse,
                deleteWarehouse: this.onDeleteWarehouse,
                select: this.onWarehouseSelected,
                searchFieldChanged: this.onSearchFieldChanged,
            },
        });

        // Create the main warehouse store
        this.warehouseStore = Ext.create('Shopware.apps.ViisonPickwareERPWarehouseManagement.store.Warehouse', {});

        // Check for a passed warehouse ID
        if (this.subApplication && this.subApplication.params && Ext.isNumeric(this.subApplication.params.warehouseId)) {
            // Load only the warehouse whose ID was passed to this app
            this.warehouseStore.filters.add('singleWarehouse', Ext.create('Ext.util.Filter', {
                property: 'warehouse.id',
                value: this.subApplication.params.warehouseId,
            }));
            this.warehouseStore.load({
                scope: this,
                callback: function (records) {
                    // Try to open the requested warehouse in an edit window
                    this.warehouseStore.filters.removeAtKey('singleWarehouse');
                    if (records.length === 1) {
                        var editWindow = this.getController('Edit').createEditWindow(records[0]);
                        if (this.subApplication.params.binLocationCode) {
                            // Activate the bin locations tab and search for the code
                            var tabPanel = editWindow.down('tabpanel');
                            var binLocationsTab = tabPanel.items.getAt(1);
                            tabPanel.setActiveTab(binLocationsTab);
                            binLocationsTab.down('toolbar').down('textfield[name=searchfield]').setValue(
                                this.subApplication.params.binLocationCode
                            );
                        }
                    } else {
                        Shopware.Notification.createGrowlMessage('Hinweis', Ext.String.format('Lager nicht gefunden', this.subApplication.params.warehouseId), 'ViisonPickwareERP');
                    }
                },
            });
        } else {
            // Just load the stroe and create the main window
            this.warehouseStore.on('load', this.clearDetailView, this);
            this.warehouseStore.load();
            this.mainView = this.getView('Main').create({
                warehouseStore: this.warehouseStore,
            });
        }

        this.callParent(arguments);
    },

    /**
     * Creates an empty warehouse model instance and passes it to a new edit window.
     */
    onAddWarehouse: function () {
        // Create an empty warehouse record
        var newWarehouseRecord = Ext.create('Shopware.apps.ViisonPickwareERPWarehouseManagement.model.Warehouse', {});

        this.getController('Edit').createEditWindow(newWarehouseRecord);
    },

    /**
     * Opens a new edit window using the given warehouse record.
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.model.Warehouse record
     */
    onEditWarehouse: function (record) {
        this.getController('Edit').createEditWindow(record);
    },

    /**
     * Ask the user for confirmation of the deletion and, if confirmed, deletes
     * the given record from the store and syncs the store afterwards.
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.model.Warehouse record
     */
    onDeleteWarehouse: function (record) {
        // Present a confirm alert before deleting the warehouse
        Ext.Msg.confirm(this.getViisonSnippet('alert/title'), this.getViisonSnippet('alert/message/delete'), function (buttonId) {
            if (buttonId !== 'yes') {
                return;
            }

            // Remove the given record from the store and sync the changes
            this.warehouseStore.remove(record);
            this.syncWarehouses(this.getViisonSnippet('notification/error/message/delete'), function (success, responseData) {
                if (!success && responseData.issues) {
                    // Could not delete warehouses because some pre-conditions are not met
                    var glue = '<br> &bull; ';
                    Shopware.Notification.createStickyGrowlMessage({
                        title: this.getViisonSnippet('notification/error/title'),
                        text: this.getViisonSnippet('notification/error/message/delete/failed_with_issues') + glue + responseData.issues[record.get('id')].join(glue),
                    });
                }
            }.bind(this));
        }, this);
    },

    /**
     * Updates the content of the detail panel using the data of the given record
     * and expands the detail panel.
     *
     * @param Ext.selection.DataViewModel selection
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.model.Warehouse record
     */
    onWarehouseSelected: function (selection, record) {
        this.getDetailView().dataView.update(record.data);
        this.getDetailView().expand();
    },

    /**
     * Sets the given query as an extra parameter to the warehouse store's proxy
     * and reloads the store.
     *
     * @param string query
     */
    onSearchFieldChanged: function (query) {
        this.warehouseStore.getProxy().extraParams.query = query;
        this.warehouseStore.loadPage(1);
    },

    /**
     * Synchronizes the warehouses store and, in case of failure, shows a growl notification.
     *
     * @param string errorMessage
     * @param function callback (optional)
     */
    syncWarehouses: function (errorMessage, callback) {
        // Check the store for any changes
        var numChanges = this.warehouseStore.getNewRecords().length + this.warehouseStore.getModifiedRecords().length + this.warehouseStore.getRemovedRecords().length;
        if (numChanges === 0) {
            if (Ext.isFunction(callback)) {
                callback(true);
            }

            return;
        }

        // Sync the store
        this.selectedRow = null;
        if (this.mainView) {
            this.mainView.setLoading(true);
        }
        this.warehouseStore.sync({
            scope: this,
            success: function () {
                this.warehouseStore.reload({
                    callback: function () {
                        if (this.mainView) {
                            this.mainView.setLoading(false);
                            this.clearDetailView();
                        }
                    },
                    scope: this,
                });
                if (Ext.isFunction(callback)) {
                    callback(true);
                }
            },
            failure: function (batch) {
                if (this.mainView) {
                    this.mainView.setLoading(false);
                }
                // Revert any changes like added, updated or removed records
                this.warehouseStore.rejectChanges();
                // Display an error notification
                var message = errorMessage || '';
                Shopware.Notification.createGrowlMessage(this.getViisonSnippet('notification/error/title'), message, 'ViisonPickwareERPWarehouseManagement');
                if (Ext.isFunction(callback)) {
                    callback(false, batch.proxy.reader.jsonData);
                }
            },
        });
    },

    /**
     * Clears the detail view.
     */
    clearDetailView: function () {
        this.getDetailView().dataView.update({});
    },

});
