// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPWarehouseManagement.view.BinLocationSelectionAlert', {

    extend: 'Ext.window.MessageBox',
    alias: 'widget.viison_pickware_erp_warehouse_management-bin_location_selection_alert',
    alternateClassName: 'ViisonPickwareBinLocationSelectionAlert',
    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_warehouse_management/bin_location_selection_alert',

    /**
     * @Override
     */
    constructor: function () {
        this.callParent(arguments);

        // Locale files only translate the button text of the message box
        // singelton Ext.Msg (= Ext.MessageBox), but not of the parent class
        // Ext.window.MessageBox. Hence we need to apply the translation
        // manually.
        this.buttonText = Ext.Msg.buttonText;
    },

    statics: {
        /**
         * Creates and shows a new instance of this class, that contains an icon, message and warehouse combo box.
         *
         * @param String title
         * @param String message
         * @param Number buttons
         * @param Function callback
         * @param Object scope
         * @return Shopware.apps.ViisonPickwareERPWarehouseManagement.view.BinLocationSelectionAlert
         */
        selectWarehouse: function (title, message, buttons, callback, scope) {
            var alert = Ext.create('Shopware.apps.ViisonPickwareERPWarehouseManagement.view.BinLocationSelectionAlert', {});

            return alert.show({
                title: title || this.getViisonSnippet('select_warehouse/title'),
                msg: message || this.getViisonSnippet('select_warehouse/message'),
                icon: alert.QUESTION,
                buttons: buttons || alert.OKCANCEL,
                showWarehouseSelection: true,
                showBinLocationSelection: false,
                callback: function (buttonId) {
                    if (Ext.isFunction(callback)) {
                        callback.call(scope, buttonId, alert.warehouseSelection.getSelectedRecord());
                    }
                },
            });
        },

        /**
         * Creates and shows a new instance of this class, that contains an icon, message as well as warehouse and bin
         * location combo boxes.
         *
         * @param String title
         * @param String message
         * @param Number buttons
         * @param Function callback
         * @param Object scope
         * @return Shopware.apps.ViisonPickwareERPWarehouseManagement.view.BinLocationSelectionAlert
         */
        selectWarehouseAndBinLocation: function (title, message, buttons, callback, scope) {
            var alert = Ext.create('Shopware.apps.ViisonPickwareERPWarehouseManagement.view.BinLocationSelectionAlert', {});

            return alert.show({
                title: title || this.getViisonSnippet('select_warehouse_and_bin_location/title'),
                msg: message || this.getViisonSnippet('select_warehouse_and_bin_location/message'),
                icon: alert.QUESTION,
                buttons: buttons || alert.OKCANCEL,
                showWarehouseSelection: true,
                showBinLocationSelection: true,
                callback: function (buttonId) {
                    if (Ext.isFunction(callback)) {
                        callback.call(
                            scope,
                            buttonId,
                            alert.warehouseSelection.getSelectedRecord(),
                            alert.binLocationSelection.getSelectedRecord()
                        );
                    }
                },
            });
        },
    },

    /**
     * @Override
     */
    initComponent: function () {
        this.callParent(arguments);

        // Apply shopware styling
        this.msg.allowHtml = true;
        this.bottomTb.addCls('shopware-toolbar');
        this.bottomTb.setUI('shopware-ui');
        Ext.each(this.msgButtons, function (button) {
            var cls = (button.itemId === 'ok' || button.itemId === 'yes') ? 'primary' : 'secondary';
            button.addCls(cls);
        });

        // Create and add the warehouse and bin location combo boxes
        this.warehouseSelection = Ext.create('Shopware.apps.ViisonPickwareERPWarehouseManagement.view.WarehouseComboBox', {
            name: 'warehouse',
            margin: '10 0 0 0',
            hidden: true,
        });
        this.promptContainer.insert(1, this.warehouseSelection);
        this.binLocationSelection = Ext.create('Shopware.apps.ViisonPickwareERPWarehouseManagement.view.BinLocationComboBox', {
            name: 'binLocation',
            margin: '10 0 0 0',
            hidden: true,
        });
        this.promptContainer.insert(2, this.binLocationSelection);

        // Listen on warehouse changes to automatically update the bin location filter accordingly
        this.warehouseSelection.on('warehouseChanged', function (warehouse) {
            this.binLocationSelection.warehouseId = (warehouse) ? warehouse.get('id') : null;
            if (!this.binLocationSelection.isHidden()) {
                this.binLocationSelection.loadStore();
            }
        }, this);
    },

    /**
     * @Override
     */
    reconfigure: function (cfg) {
        // Show/hide the warehouse and bin location selection
        if (cfg.showWarehouseSelection) {
            this.warehouseSelection.show();
            this.warehouseSelection.store.load();
        } else {
            this.warehouseSelection.hide();
        }
        if (cfg.showBinLocationSelection) {
            this.binLocationSelection.show();
            this.binLocationSelection.loadStore();
        } else {
            this.binLocationSelection.hide();
        }

        return this.callParent(arguments);
    },

});
