// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPWarehouseManagement.view.edit.BinLocations.BinLocationGenerator', {

    extend: 'Ext.window.Window',
    alias: 'widget.viison_pickware_erp_warehouse_management-edit-bin_locations-bin_location_generator',
    cls: 'viison_pickware_erp_warehouse_management-edit-bin_locations-bin_location_generator viison-common--window has--no-border',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_warehouse_management/main',

    border: false,
    layout: 'fit',
    modal: true,
    width: '90%',
    height: '80%',

    /**
     * @Override
     */
    initComponent: function () {
        this.title = this.getViisonSnippet('edit/bin_location_generator/title');

        // Add an object mapping the internal IDs of the currently existing components to their
        // respective forms
        this.rowForms = {};

        this.items = this.createItems();
        this.dockedItems = this.createDockedItems();

        // Listen on the grid's 'viewready' to be able to render the correct form elements into the grid row
        this.gridPanel.on('viewready', function () {
            this.fireEvent('formChanged', this);
            this.store.each(function (record) {
                this.renderRow(record);
            }, this);
        }, this);

        this.callParent(arguments);
    },

    /**
     * @return Ext.component.Component[]
     */
    createItems: function () {
        // Create and save a grid panel
        this.gridPanel = Ext.create('Ext.grid.Panel', {
            cls: 'is--editor-grid',
            split: true,
            border: false,
            autoScroll: true,
            viewConfig: {
                enableTextSelection: true,
                markDirty: false,
            },
            columns: this.createColumns(),
            store: this.store,
        });

        return [
            this.gridPanel,
        ];
    },

    /**
     * @return Ext.grid.column.Column[]
     */
    createColumns: function () {
        return [{
            xtype: 'gridcolumn',
            dataIndex: 'fixed',
            header: this.getViisonSnippet('edit/bin_location_generator/list/column/component'),
            sortable: false,
            flex: 1,
            renderer: function (value, metaData) {
                // Mark the column for custom rendering
                metaData.tdCls += ' component-form';

                return '';
            },
        }, {
            xtype: 'actioncolumn',
            align: 'right',
            width: 130,
            items: [{
                iconCls: 'is--icon-add',
                action: 'deleteComponent',
                tooltip: this.getViisonSnippet('edit/bin_location_generator/list/tooltip/add'),
                scope: this,
                handler: function (grid, rowIndex) {
                    this.fireEvent('addComponent', this, rowIndex + 1);
                },
            }, {
                iconCls: 'is--icon-remove',
                action: 'deleteComponent',
                tooltip: this.getViisonSnippet('edit/bin_location_generator/list/tooltip/delete'),
                scope: this,
                handler: function (grid, rowIndex) {
                    this.fireEvent('deleteComponent', this, grid.getStore().getAt(rowIndex));
                },
            }],
        }];
    },

    /**
     * @return Ext.toolbar.Toolbar[]
     */
    createDockedItems: function () {
        // Create two template labels for the top toolbar
        this.exampleBinLocationLabel = Ext.create('Ext.Component', {
            flex: 1,
            cls: 'is--info-text is--bin-location-example',
            data: {
                binLocation: '-',
            },
            tpl: '<b>' + this.getViisonSnippet('edit/bin_location_generator/toolbar/label/example') + ':</b><span>{binLocation}</span>',
        });
        this.numberOfPossibleBinLocationsLabel = Ext.create('Ext.Component', {
            cls: 'is--info-text is--number-of-possible-bin-locations',
            data: {
                numPossibleLocations: '-',
            },
            tpl: '<b>' + this.getViisonSnippet('edit/bin_location_generator/toolbar/label/number_of_possible_locations') + ':</b><span>{numPossibleLocations}</span>',
        });
        this.binLocationFormatValidationErrorLabel = Ext.create('Ext.Component', {
            hidden: true,
            flex: 1,
            cls: 'is--editor-error',
            data: {
                message: this.getViisonSnippet('edit/bin_location_generator/format_validation/default_error_message'),
            },
            tpl: '<b>' + this.getViisonSnippet('edit/bin_location_generator/toolbar/label/format_validation_error') + ':</b><span>{message}</span>',
        });

        // Create top and bottom toolbars
        return [{
            xtype: 'toolbar',
            dock: 'top',
            ui: 'shopware-ui is--viison-common--toolbar is--header-toolbar',
            layout: {
                type: 'hbox',
                align: 'stretch',
            },
            items: [
                {
                    xtype: 'button',
                    text: this.getViisonSnippet('edit/bin_location_generator/toolbar/button/add'),
                    iconCls: 'sprite-plus-circle-frame',
                    margin: '0 5 0 0',
                    action: 'addComponent',
                    scope: this,
                    handler: function () {
                        this.fireEvent('addComponent', this, this.store.count());
                    },
                }, {
                    xtype: 'button',
                    text: this.getViisonSnippet('edit/bin_location_generator/toolbar/button/save'),
                    iconCls: 'sprite-sd-memory-card',
                    margin: '0 5 0 0',
                    action: 'save',
                }, {
                    xtype: 'tbspacer',
                    width: 8,
                }, {
                    xtype: 'container',
                    cls: 'is--info-container',
                    flex: 1,
                    layout: {
                        type: 'hbox',
                        align: 'stretch',
                    },
                    items: [
                        this.exampleBinLocationLabel,
                        this.binLocationFormatValidationErrorLabel,
                        this.numberOfPossibleBinLocationsLabel,
                    ],
                },
            ],
        }, {
            xtype: 'toolbar',
            dock: 'bottom',
            ui: 'shopware-ui is--viison-common--button-panel',
            items: [
                '->',
                {
                    xtype: 'button',
                    text: this.getViisonSnippet('edit/bin_location_generator/toolbar/button/cancel'),
                    cls: 'secondary',
                    action: 'cancel',
                }, {
                    xtype: 'button',
                    text: this.getViisonSnippet('edit/bin_location_generator/toolbar/button/generate'),
                    cls: 'primary',
                    action: 'generateBinLocations',
                },
            ],
        }];
    },

    /**
     * Creates a new form panel containing all given 'items' plus an item for changing making
     * the component dynamic/fixed and renders it to the given element. The form is then saved
     * to have an easy access to it. When creating the form it is rendered to a dedicated cell
     * in the row of the record.
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.model.BinLocationFormatComponent record
     */
    renderRow: function (record) {
        // Find the row node
        var recordIndex = this.store.indexOf(record);
        var cells = Ext.DomQuery.select('.component-form', this.gridPanel.view.getNode(recordIndex));
        if (cells.length === 0) {
            return;
        }

        // Create fields required for the record and prepend them with the type selection
        var items = [{
            xtype: 'label',
            cls: 'is--viison-common--label',
            text: this.getViisonSnippet('warehouse/bin_location_format_component/field/type') + ':',
            margin: '7 10 0 0',
        }, {
            xtype: 'combobox',
            name: 'type',
            hideLabel: true,
            width: 150,
            margin: '0 25 0 0',
            store: Ext.create('Ext.data.Store', {
                fields: [
                    { name: 'value', type: 'string' },
                    { name: 'label', type: 'string' },
                ],
                data: [
                    { value: 'letters', label: this.getViisonSnippet('edit/bin_location_generator/form/field/type/letters') },
                    { value: 'digits', label: this.getViisonSnippet('edit/bin_location_generator/form/field/type/digits') },
                    { value: 'fixed', label: this.getViisonSnippet('edit/bin_location_generator/form/field/type/fixed') },
                ],
            }),
            queryMode: 'local',
            displayField: 'label',
            valueField: 'value',
            forceSelection: true,
            editable: false,
            value: record.get('type'),
            listeners: {
                scope: this,
                change: function (field, newValue) {
                    // Clear start and end values
                    record.raw.start = '';
                    record.raw.end = '';
                    this.createRowUpdateListener(record, this)(field, newValue);
                    this.fireEvent('formChanged', this);
                },
            },
        }];
        switch (record.get('type')) {
            case 'letters':
                items = items.concat(this.createLettersComponentFields(record));
                break;
            case 'digits':
                items = items.concat(this.createDigitsComponentFields(record));
                break;
            case 'fixed':
                items = items.concat(this.createFixedComponentFields(record));
                break;
            default:
                break;
        }

        // Create and save the form and render it to the cell
        var cellEl = cells[0];
        cellEl.innerHTML = '';
        this.rowForms[record.internalId] = Ext.create('Ext.form.Panel', {
            width: '100%',
            border: false,
            bodyStyle: 'background: transparent;',
            padding: 3,
            renderTo: cellEl,
            layout: {
                type: 'hbox',
            },
            items: items,
        });

        // Validate the form once to show validation hints right away when adding a new component
        this.rowForms[record.internalId].getForm().isValid();
    },

    /**
     * Creates and returns the fields required for a letters component form.
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.model.BinLocationFormatComponent record
     * @return object[]
     */
    createLettersComponentFields: function (record) {
        return [{
            xtype: 'label',
            cls: 'is--viison-common--label',
            minWidth: 40,
            text: this.getViisonSnippet('warehouse/bin_location_format_component/field/length') + ':',
            margin: '7 10 0 0',
        }, {
            xtype: 'numberfield',
            name: 'length',
            hideLabel: true,
            width: 80,
            margin: '0 25 0 0',
            value: record.get('length'),
            minValue: 1,
            editable: false,
            listeners: {
                change: this.createRowUpdateListener(record, this),
            },
        }, {
            xtype: 'label',
            cls: 'is--viison-common--label',
            text: this.getViisonSnippet('warehouse/bin_location_format_component/field/start/label') + ':',
            margin: '7 10 0 0',
        }, {
            xtype: 'textfield',
            name: 'start',
            hideLabel: true,
            emptyText: this.getViisonSnippet('warehouse/bin_location_format_component/field/start/empty_text/prefix') + ' A',
            width: 80,
            margin: '0 25 0 0',
            value: record.get('start'),
            allowBlank: false,
            regex: /^[A-Z]+$/,
            regexText: this.getViisonSnippet('warehouse/bin_location_format_component/field/text_validation_error'),
            minLength: record.get('length'),
            maxLength: record.get('length'),
            listeners: {
                scope: this,
                change: function (field, newValue) {
                    field.setRawValue(newValue.toUpperCase());
                    record.raw.start = field.getRawValue();
                    this.fireEvent('formChanged', this);
                },
            },
        }, {
            xtype: 'label',
            cls: 'is--viison-common--label',
            text: this.getViisonSnippet('warehouse/bin_location_format_component/field/end/label') + ':',
            margin: '7 10 0 0',
        }, {
            xtype: 'textfield',
            name: 'end',
            hideLabel: true,
            emptyText: this.getViisonSnippet('warehouse/bin_location_format_component/field/end/empty_text/prefix') + ' Z',
            width: 80,
            margin: '0 25 0 0',
            value: record.get('end'),
            allowBlank: false,
            regex: /^[A-Z]+$/,
            regexText: this.getViisonSnippet('warehouse/bin_location_format_component/field/text_validation_error'),
            minLength: record.get('length'),
            maxLength: record.get('length'),
            listeners: {
                scope: this,
                change: function (field, newValue) {
                    field.setRawValue(newValue.toUpperCase());
                    record.raw.end = field.getRawValue();
                    this.fireEvent('formChanged', this);
                },
            },
        }];
    },

    /**
     * Creates and returns the fields required for a digits component form.
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.model.BinLocationFormatComponent record
     * @return object[]
     */
    createDigitsComponentFields: function (record) {
        // Determine the max value
        var endMaxValue = 0;
        for (var i = 0; i < record.get('length'); i += 1) {
            endMaxValue += 9 * Math.pow(10, i);
        }

        return [{
            xtype: 'label',
            cls: 'is--viison-common--label',
            minWidth: 40,
            text: this.getViisonSnippet('warehouse/bin_location_format_component/field/length') + ':',
            margin: '7 10 0 0',
        }, {
            xtype: 'numberfield',
            name: 'length',
            hideLabel: true,
            width: 80,
            margin: '0 25 0 0',
            value: record.get('length'),
            minValue: 1,
            editable: false,
            listeners: {
                change: this.createRowUpdateListener(record, this),
            },
        }, {
            xtype: 'label',
            cls: 'is--viison-common--label',
            text: this.getViisonSnippet('warehouse/bin_location_format_component/field/start/label') + ':',
            margin: '7 10 0 0',
        }, {
            xtype: 'numberfield',
            name: 'start',
            hideLabel: true,
            emptyText: this.getViisonSnippet('warehouse/bin_location_format_component/field/start/empty_text/prefix') + ' 0',
            width: 80,
            margin: '0 25 0 0',
            value: record.get('start'),
            minValue: 0,
            maxValue: (record.get('end')) ? record.get('end') : endMaxValue,
            allowBlank: false,
            listeners: {
                scope: this,
                change: function (field) {
                    record.raw.start = field.getRawValue();
                    this.fireEvent('formChanged', this);
                },
            },
        }, {
            xtype: 'label',
            cls: 'is--viison-common--label',
            text: this.getViisonSnippet('warehouse/bin_location_format_component/field/end/label') + ':',
            margin: '7 10 0 0',
        }, {
            xtype: 'numberfield',
            name: 'end',
            hideLabel: true,
            emptyText: this.getViisonSnippet('warehouse/bin_location_format_component/field/end/empty_text/prefix') + ' 9',
            width: 80,
            margin: '0 25 0 0',
            value: record.get('end'),
            minValue: (record.get('start')) ? record.get('start') : 0,
            maxValue: endMaxValue,
            allowBlank: false,
            listeners: {
                scope: this,
                change: function (field) {
                    record.raw.end = field.getRawValue();
                    this.fireEvent('formChanged', this);
                },
            },
        }, {
            xtype: 'label',
            cls: 'is--viison-common--label',
            text: this.getViisonSnippet('warehouse/bin_location_format_component/field/leading_zeros') + ':',
            margin: '7 10 0 0',
        }, {
            xtype: 'checkbox',
            name: 'leadingZeros',
            cls: 'is--editor-grid-checkbox',
            hideLabel: true,
            value: true,
            inputValue: true,
            checked: record.get('leadingZeros'),
            listeners: {
                scope: this,
                change: function (field, newValue) {
                    record.raw.leadingZeros = newValue;
                    this.fireEvent('formChanged', this);
                },
            },
        }];
    },

    /**
     * Creates and returns the fields required for a fixed component form.
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.model.BinLocationFormatComponent record
     * @return object[]
     */
    createFixedComponentFields: function (record) {
        return [{
            xtype: 'label',
            cls: 'is--viison-common--label',
            minWidth: 40,
            text: this.getViisonSnippet('warehouse/bin_location_format_component/field/value') + ':',
            margin: '7 10 0 0',
        }, {
            xtype: 'textfield',
            name: 'value',
            hideLabel: true,
            width: 80,
            margin: '0 25 0 0',
            emptyText: this.getViisonSnippet('warehouse/bin_location_format_component/field/value/empty_text/prefix') + ' -',
            value: record.get('value'),
            maskRe: /[^\s]/,
            listeners: {
                scope: this,
                change: function (field, newValue) {
                    record.raw.value = newValue;
                    this.fireEvent('formChanged', this);
                },
            },
        }];
    },

    /**
     * Creates a function that expects two parameters 'field' and 'newValue', which can
     * be used as a 'change' listener customised for the given format component record.
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.model.BinLocationFormatComponent record
     * @param Object scope
     * @return function
     */
    createRowUpdateListener: function (record, scope) {
        return function (field, newValue) {
            // Apply all raw data on the record incl. the new value of the field triggering the update event
            record.raw[field.name] = newValue;
            Ext.Object.each(record.raw, function (key, value) {
                record.set(key, value);
            });
            this.renderRow(record);
        }.bind(scope);
    },

    /**
     * Shows/hides the example bin location, number of possible locations as well as
     * format validation error labels based on the 'success' flag contained in the
     * given data and updates these fields using the same data.
     *
     * @param object data
     */
    updateExampleBinLocationLabels: function (data) {
        this.exampleBinLocationLabel.update(data);
        this.exampleBinLocationLabel.setVisible(data.success);
        this.numberOfPossibleBinLocationsLabel.update(data);
        this.numberOfPossibleBinLocationsLabel.setVisible(data.success);
        this.binLocationFormatValidationErrorLabel.update({
            message: data.message || this.getViisonSnippet('edit/bin_location_generator/format_validation/default_error_message'),
        });
        this.binLocationFormatValidationErrorLabel.setVisible(!data.success);
    },

});
