// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPWarehouseManagementBarcodeLabelPrinting.controller.Edit', {

    override: 'Shopware.apps.ViisonPickwareERPWarehouseManagement.controller.Edit',

    /**
     * @Override
     */
    init: function () {
        this.control({
            'viison_pickware_erp_warehouse_management-edit-bin_locations button[action=saveBinLocationsForBarcodeLabelPrinting]': {
                click: this.onSaveBinLocationsForBarcodeLabelPrinting,
            },
            'viison_pickware_erp_warehouse_management-edit-bin_locations-bin_location_list': {
                saveBinLocationForBarcodeLabelPrinting: this.onSaveBinLocationForBarcodeLabelPrinting,
            },
        });

        this.callParent(arguments);
    },

    /**
     * @Override
     */
    onBinLocationSelectionChanged: function (panel, selection) {
        var mainPanel = panel.up('viison_pickware_erp_warehouse_management-edit-bin_locations');
        // Update the label of the 'print barcode labels' button based on the selection
        var text;
        if (selection.length === 0) {
            text = ViisonCommonApp.getSnippet(
                'edit/bin_locations/toolbar/button/save_all_bin_locations_for_barcode_label_printing',
                'backend/viison_pickware_erp_warehouse_management_barcode_label_printing/main'
            );
        } else {
            text = ViisonCommonApp.getSnippet(
                'edit/bin_locations/toolbar/button/save_selected_bin_locations_for_barcode_label_printing',
                'backend/viison_pickware_erp_warehouse_management_barcode_label_printing/main'
            );
        }
        mainPanel.down('button[action=saveBinLocationsForBarcodeLabelPrinting]').setText(text);

        this.callParent(arguments);
    },

    /**
     * Determines the current selection of bin locations in the list and, depending on the selection,
     * either adds all selected bin locations to the barcode label printing or, if no rows are
     * selected, adds all bin locations of the current warehouse, if they match the query entered
     * in the search field.
     *
     * @param Ext.button.Button button
     */
    onSaveBinLocationsForBarcodeLabelPrinting: function (button) {
        var panel = button.up('viison_pickware_erp_warehouse_management-edit-bin_locations');
        var binLocationListPanel = panel.down('viison_pickware_erp_warehouse_management-edit-bin_locations-bin_location_list');
        var selection = binLocationListPanel.selModel.getSelection();
        // Perpare a common callback
        var callback = function (options, success) {
            panel.setLoading(false);

            var title;
            var message;

            if (success) {
                title = ViisonCommonApp.getSnippet(
                    'notification/save_bin_locations_for_barcode_label_printing/success/title',
                    'backend/viison_pickware_erp_warehouse_management_barcode_label_printing/main'
                );
                message = ViisonCommonApp.getSnippet(
                    'notification/save_bin_locations_for_barcode_label_printing/success/message',
                    'backend/viison_pickware_erp_warehouse_management_barcode_label_printing/main'
                );
            } else {
                title = ViisonCommonApp.getSnippet(
                    'notification/save_bin_locations_for_barcode_label_printing/error/title',
                    'backend/viison_pickware_erp_warehouse_management_barcode_label_printing/main'
                );
                message = ViisonCommonApp.getSnippet(
                    'notification/save_bin_locations_for_barcode_label_printing/error/message',
                    'backend/viison_pickware_erp_warehouse_management_barcode_label_printing/main'
                );
            }

            Shopware.Msg.createGrowlMessage(title, message, 'ViisonPickwareERPArticleListBarcodeLabelPrinting');
        };

        panel.setLoading(true);
        if (selection.length > 0) {
            // Add only the selected bin locations
            var binLocationIds = [];
            Ext.each(selection, function (binLocation) {
                binLocationIds.push(binLocation.get('id'));
            });
            Ext.Ajax.request({
                method: 'POST',
                url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPBarcodeLabelPrinting/enqueueBarcodeLabel'),
                jsonData: {
                    type: 'bin_location',
                    identifiers: binLocationIds,
                },
                callback: callback,
            });
        } else {
            // Add all bin locations of the current warehouse, matching the search query
            Ext.Ajax.request({
                method: 'POST',
                url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPBarcodeLabelPrinting/addAllFilteredBinLocationBarcodeLabels'),
                jsonData: {
                    warehouseId: panel.record.get('id'),
                    searchQuery: binLocationListPanel.store.getProxy().extraParams.query,
                },
                callback: callback,
            });
        }
    },

    /**
     * POSTs the ID of the given bin location them to the main barcode label
     * printing controller to add them to the barcode label printing list.
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.view.edit.BinLocations.BinLocationList listPanel
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.model.BinLocation binLocation
     */
    onSaveBinLocationForBarcodeLabelPrinting: function (listPanel, binLocation) {
        var panel = listPanel.up('viison_pickware_erp_warehouse_management-edit-bin_locations');
        // Add the bin locations to the barcode label printing list
        panel.setLoading(true);
        Ext.Ajax.request({
            method: 'POST',
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPBarcodeLabelPrinting/enqueueBarcodeLabel'),
            jsonData: {
                type: 'bin_location',
                identifiers: [
                    binLocation.get('id'),
                ],
            },
            callback: function (options, success) {
                panel.setLoading(false);

                var title;
                var message;

                if (success) {
                    title = ViisonCommonApp.getSnippet(
                        'notification/save_bin_location_for_barcode_label_printing/success/title',
                        'backend/viison_pickware_erp_warehouse_management_barcode_label_printing/main'
                    );
                    message = ViisonCommonApp.getSnippet(
                        'notification/save_bin_location_for_barcode_label_printing/success/message',
                        'backend/viison_pickware_erp_warehouse_management_barcode_label_printing/main'
                    );
                } else {
                    title = ViisonCommonApp.getSnippet(
                        'notification/save_bin_location_for_barcode_label_printing/error/title',
                        'backend/viison_pickware_erp_warehouse_management_barcode_label_printing/main'
                    );
                    message = ViisonCommonApp.getSnippet(
                        'notification/save_bin_location_for_barcode_label_printing/error/message',
                        'backend/viison_pickware_erp_warehouse_management_barcode_label_printing/main'
                    );
                }

                Shopware.Msg.createGrowlMessage(title, message, 'ViisonPickwareERPArticleListBarcodeLabelPrinting');
            },
        });
    },

});
