<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonCommon\Classes\Document;

/**
 * Class that represents a page layout for a label page
 */
class PaperLayout
{
    const PAGE_ORIENTATION_PORTRAIT = 'portrait';
    const PAGE_ORIENTATION_LANDSCAPE = 'landscape';

    const PAGE_FORMAT_DIN_A3 = 'din_a3';
    const PAGE_FORMAT_DIN_A4 = 'din_a4';
    const PAGE_FORMAT_DIN_A5 = 'din_a5';
    const PAGE_FORMAT_DIN_A6 = 'din_a6';

    /**
     * General snippet namespace for all grid paper layouts.
     */
    const SNIPPET_NAMESPACE = 'backend/viison_common/paper_layouts';

    /**
     * @var string
     */
    public $identifier;

    /**
     * @var string $pageFormat
     */
    public $pageFormat;

    /**
     * @var string $pageOrientation
     */
    public $pageOrientation;

    /**
     * @var float $pageHeightInMillimeters
     */
    public $pageHeightInMillimeters;

    /**
     * @var float $pageWidthInMillimeters
     */
    public $pageWidthInMillimeters;

    /**
     * @var string $snippetNamespace
     */
    public $snippetNamespace;

    /**
     * @var string $snippetName
     */
    public $snippetName;

    /**
     * Returns the PaperLayout for $layoutIdentifier
     *
     * @param string $layoutIdentifier
     * @return PaperLayout
     * @throws \InvalidArgumentException When the identifier is not a known PaperLayout
     */
    public static function createDefaultPaperLayout($layoutIdentifier)
    {
        $layout = new self();
        $layout->snippetNamespace = static::SNIPPET_NAMESPACE;

        // Normalize the identifier
        $layoutIdentifier = mb_strtolower(trim($layoutIdentifier));
        $layoutIdentifier = preg_replace('|[-\\s]|', '_', $layoutIdentifier);

        switch ($layoutIdentifier) {
            case 'a3_l':
            case 'a3_landscape':
            case 'din_a3_landscape':
                $layout->identifier = 'din_a3_landscape';
                $layout->pageOrientation = self::PAGE_ORIENTATION_LANDSCAPE;
                $layout->pageFormat = self::PAGE_FORMAT_DIN_A3;
                $layout->pageWidthInMillimeters = 420;
                $layout->pageHeightInMillimeters = 297;
                $layout->snippetName = 'paper_format/din_a3_landscape';
                break;
            case 'a3':
            case 'a3_p':
            case 'a3_portrait':
            case 'din_a3_portrait':
                $layout->identifier = 'din_a3_portrait';
                $layout->pageOrientation = self::PAGE_ORIENTATION_PORTRAIT;
                $layout->pageFormat = self::PAGE_FORMAT_DIN_A3;
                $layout->pageWidthInMillimeters = 297;
                $layout->pageHeightInMillimeters = 420;
                $layout->snippetName = 'paper_format/din_a3_portrait';
                break;
            case 'a4_l':
            case 'a4_landscape':
            case 'din_a4_landscape':
                $layout->identifier = 'din_a4_landscape';
                $layout->pageOrientation = self::PAGE_ORIENTATION_LANDSCAPE;
                $layout->pageFormat = self::PAGE_FORMAT_DIN_A4;
                $layout->pageWidthInMillimeters = 297;
                $layout->pageHeightInMillimeters = 210;
                $layout->snippetName = 'paper_format/din_a4_landscape';
                break;
            case 'a4':
            case 'a4_p':
            case 'a4_portrait':
            case 'din_a4_portrait':
                $layout->identifier = 'din_a4_portrait';
                $layout->pageOrientation = self::PAGE_ORIENTATION_PORTRAIT;
                $layout->pageFormat = self::PAGE_FORMAT_DIN_A4;
                $layout->pageWidthInMillimeters = 210;
                $layout->pageHeightInMillimeters = 297;
                $layout->snippetName = 'paper_format/din_a4_portrait';
                break;
            case 'a5_l':
            case 'a5_landscape':
            case 'din_a5_landscape':
                $layout->identifier = 'din_a5_landscape';
                $layout->pageOrientation = self::PAGE_ORIENTATION_LANDSCAPE;
                $layout->pageFormat = self::PAGE_FORMAT_DIN_A5;
                $layout->pageWidthInMillimeters = 210;
                $layout->pageHeightInMillimeters = 148;
                $layout->snippetName = 'paper_format/din_a5_landscape';
                break;
            case 'a5':
            case 'a5_p':
            case 'a5_portrait':
            case 'din_a5_portrait':
                $layout->identifier = 'din_a5_portrait';
                $layout->pageOrientation = self::PAGE_ORIENTATION_PORTRAIT;
                $layout->pageFormat = self::PAGE_FORMAT_DIN_A5;
                $layout->pageWidthInMillimeters = 148;
                $layout->pageHeightInMillimeters = 210;
                $layout->snippetName = 'paper_format/din_a5_portrait';
                break;
            case 'a6_l':
            case 'a6_landscape':
            case 'din_a6_landscape':
                $layout->identifier = 'din_a6_landscape';
                $layout->pageOrientation = self::PAGE_ORIENTATION_LANDSCAPE;
                $layout->pageFormat = self::PAGE_FORMAT_DIN_A6;
                $layout->pageWidthInMillimeters = 148;
                $layout->pageHeightInMillimeters = 105;
                $layout->snippetName = 'paper_format/din_a6_landscape';
                break;
            case 'a6':
            case 'a6_p':
            case 'a6_portrait':
            case 'din_a6_portrait':
                $layout->identifier = 'din_a6_portrait';
                $layout->pageOrientation = self::PAGE_ORIENTATION_PORTRAIT;
                $layout->pageFormat = self::PAGE_FORMAT_DIN_A6;
                $layout->pageWidthInMillimeters = 105;
                $layout->pageHeightInMillimeters = 148;
                $layout->snippetName = 'paper_format/din_a6_portrait';
                break;
            default:
                throw new \InvalidArgumentException(sprintf(
                    'The identifier "%s" is not a valid identifier for a PaperLayout.',
                    $layoutIdentifier
                ));
        }

        return $layout;
    }

    /**
     * @return array
     */
    public function toArray()
    {
        return [
            'identifier' => $this->identifier,
            'pageFormat' => $this->pageFormat,
            'pageOrientation' => $this->pageOrientation,
            'pageHeightInMillimeters' => $this->pageHeightInMillimeters,
            'pageWidthInMillimeters' => $this->pageWidthInMillimeters,
            'snippetName' => $this->snippetName,
            'snippetNamespace' => $this->snippetNamespace,
        ];
    }
}
