<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonCommon\Classes\Document\RenderingContext;

use Shopware\Plugins\ViisonCommon\Classes\Document\PaperLayout;
use Shopware\Plugins\ViisonCommon\Classes\Document\RenderedDocument\RenderedDocument;
use Shopware\Plugins\ViisonCommon\Classes\Document\RenderingEngine\RenderingEngine;

/**
 * Abstract class for building a rendering context. A context should gather and deliver all information that is needed
 * to create the HTML that should be rendered
 */
abstract class AbstractRenderingContext
{
    /**
     * @var RenderingEngine
     */
    protected $renderingEngine;

    /**
     * @var PaperLayout
     */
    protected $paperLayout;

    /**
     * @param RenderingEngine $renderingEngine
     * @param $paperLayout $paperLayout
     */
    public function __construct(RenderingEngine $renderingEngine, PaperLayout $paperLayout)
    {
        $this->renderingEngine = $renderingEngine;
        $this->paperLayout = $paperLayout;
    }

    /**
     * Generates the HTML
     *
     * @return string The generated HTML output of the rendering context
     */
    abstract protected function generateHtml();

    /**
     * Generates and returns the PDF output of the rendering context using the assigned rendering engine.
     *
     * @return RenderedDocument
     */
    public function renderDocument()
    {
        $html = $this->generateHtml();
        $pdf = $this->renderingEngine->render($html, $this->paperLayout);

        return new RenderedDocument($html, $pdf);
    }

    /**
     * @return RenderingEngine
     */
    public function getRenderingEngine()
    {
        return $this->renderingEngine;
    }

    /**
     * @param RenderingEngine $renderingEngine
     */
    public function setRenderingEngine($renderingEngine)
    {
        $this->renderingEngine = $renderingEngine;
    }

    /**
     * @return PaperLayout
     */
    public function getPaperLayout()
    {
        return $this->paperLayout;
    }

    /**
     * @param PaperLayout $paperLayout
     */
    public function setPaperLayout($paperLayout)
    {
        $this->paperLayout = $paperLayout;
    }
}
