<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonCommon\Classes\Installation\ArticleAttributeElement;

use Doctrine\ORM\EntityManager;
use Shopware\Plugins\ViisonCommon\Classes\Installation\AttributeConfiguration;

class InstallationHelper
{

    /**
     * @todo Use `\Shopware\Models\Article\Element::class` once we can depend on PHP >= 5.5.
     */
    const ELEMENT_CLASS = 'Shopware\\Models\\Article\\Element';

    /**
     * @var EntityManager
     */
    private $entityManager;

    public function __construct(EntityManager $entityManager)
    {
        $this->entityManager = $entityManager;
    }

    /**
     * Creates a new Article\Element which is displayed in the article details module. This is only for Shopware < 5.2
     * and will never be called in Shopware >= 5.2.
     *
     * @todo Remove this and callers once we do not have to support Shopware < 5.2.
     *
     * @param string $type
     * @param string $name
     * @param string $label
     * @param string $helpText
     * @param int $position
     * @param string $default (optional, default null)
     * @param boolean $required (optional, default false)
     * @param boolean $variantable (optional, default true)
     * @param boolean $store (optional, default null)
     * @return void
     */
    private function createArticleAttributeElement($type, $name, $label, $helpText, $position, $default = null, $required = false, $variantable = true, $store = null)
    {
        // Create element with give values
        $articleElement = new \Shopware\Models\Article\Element();
        $articleElement->setType($type);
        $articleElement->setName($name);
        $articleElement->setLabel($label);
        $articleElement->setHelp($helpText);
        $articleElement->setPosition($position);
        $articleElement->setDefault($default);
        $articleElement->setRequired($required);
        $articleElement->setVariantable($variantable);
        $articleElement->setStore($store);

        // Save element
        $this->entityManager->persist($articleElement);
        $this->entityManager->flush();
    }

    /**
     * Creates a new Article\Element which is displayed in the article details module, unless an Article\Element with
     * the same name $name already exists. If one exists with the same name, does nothing (even if other values differ).
     * If Shopware >= 5.2 is used (that is, Shopware\Models\Attribute\Configuration is defined), the attribute
     * configuration helper is used to create an attribute configuration instead of an attribute element.
     *
     * @param string $type
     * @param string $name
     * @param string $label
     * @param string $helpText
     * @param int $position
     * @param string $default (optional, default null)
     * @param boolean $required (optional, default false)
     * @param boolean $variantable (optional, default true)
     * @param boolean $store (optional, default null)
     * @return void
     */
    public function createArticleAttributeElementUnlessNameExists($type, $name, $label, $helpText, $position, $default = null, $required = false, $variantable = true, $store = null)
    {
        if (class_exists('Shopware\\Models\\Attribute\\Configuration')) {
            // Shopware >= 5.2, hence use attribute configuration instead of article elements
            $attributeConfigurationHelper = new AttributeConfiguration\InstallationHelper($this->entityManager);
            $columnName = ConfigurationHelper::transformToColumnName($name);
            $attributeConfigurationHelper->createAttributeConfigurationUnlessExists('s_articles_attributes', $columnName, $type, $label, $helpText, $position);
        } else {
            // Check for existing element
            $existingElement = $this->entityManager->getRepository(static::ELEMENT_CLASS)->findOneBy([
                'name' => $name,
            ]);
            if (!$existingElement) {
                $this->createArticleAttributeElement($type, $name, $label, $helpText, $position, $default, $required, $variantable, $store);
            }
        }
    }
}
