<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonCommon\Classes\Localization;

use Enlight_Components_Session;
use Shopware\Components\DependencyInjection\Container;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util;

/**
 * A snippet manager that can be used while the plugin is not installed / active. This works by reading snippets directly from the plugin's Snippets/ directory.
 *
 * Use this inside install / update methods of the plugin Bootstrap.
 */
class BootstrapSnippetManager extends \Shopware_Components_Snippet_Manager
{
    /** @var array the directories to search for Snippet files */
    private $directories;

    /**
     * @param Container $container
     * @param string|array $directories
     */
    public function __construct($container, $directories)
    {
        if (is_string($directories)) {
            $directories = [$directories];
        }
        $this->directories = array_map(function ($directory) {
            // Enforce a trailing slash, otherwise the Shopware snippet manager fails silently.
            return rtrim($directory, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR;
        }, $directories);

        // Set the locale from the user session if it exists
        if (Util::assertMinimumShopwareVersion('5.7.0')) {
            $sessionExists = Shopware()->Container()->initialized('backendsession');
        } else {
            $sessionExists = Enlight_Components_Session::sessionExists();
        }
        if ($sessionExists) {
            $auth = $container->get('auth');
            if ($auth->hasIdentity()) {
                $user = $auth->getIdentity();
                if (isset($user->locale)) {
                    $this->setLocale($user->locale);
                }
            }
        }

        $snippetConfig = [
            'readFromIni' => true,
            'writeToIni' => false,
            'readFromDb' => false,
            'writeToDb' => false,
        ];

        // Yay, let's insert new constructor arguments in the middle of the argument list so we're sure to break
        // backwards compatibility!
        if (Util::assertMinimumShopwareVersion('5.2.0')) {
            $templateDirSet = $container->hasParameter('shopware.template.templateDir');
            $templateDir = $templateDirSet ? ($container->getParameter('shopware.template.templateDir') . '/Frontend') : null;

            parent::__construct(
                $container->get('models'),
                [],
                $snippetConfig,
                $templateDir
            );
        } else {
            parent::__construct(
                $container->get('models'),
                $snippetConfig
            );
        }
    }

    public function getConfigDirs($themeDir = null)
    {
        return $this->directories;
    }
}
