<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonCommon\Classes\Subscribers;

class ViewLoading extends AbstractBaseSubscriber
{
    /**
     * @var string $pluginViewsNamespace
     */
    private $pluginViewsNamespace;

    /**
     * @param \Shopware_Components_Plugin_Bootstrap $pluginBootstrap
     * @param string $pluginViewsNamespace
     */
    public function __construct(\Shopware_Components_Plugin_Bootstrap $pluginBootstrap, $pluginViewsNamespace)
    {
        parent::__construct($pluginBootstrap);
        $this->pluginViewsNamespace = $pluginViewsNamespace;
    }

    /**
     * @see \Shopware\Plugins\ViisonCommon\Classes\Subscribers\Base::getSubscribedEvents()
     */
    public static function getSubscribedEvents()
    {
        return [
            'Enlight_Controller_Action_PostDispatchSecure' => 'onPostDispatchSecure',
            'Theme_Inheritance_Template_Directories_Collected' => 'onFilterThemeInheritenceTemplateDirectories',
            'Theme_Compiler_Collect_Plugin_Less' => 'addLessFiles',
        ];
    }

    /**
     * Overwrite this method to return a list of all less files, which
     * should be added to the processing queue of shopware's less compiler.
     * Files must be specified relative to the plugin's root path, e.g.
     * 'Views/frontend/_public/src/less/all.less'.
     *
     * @return array
     */
    public static function getLessFiles()
    {
        return [];
    }

    /**
     * Appends the path to this plugin's views to the list of directories used for frontend
     * template inheritance in Shopware < 5 (and Shopware backend).
     *
     * @param \Enlight_Event_EventArgs $args
     */
    public function onPostDispatchSecure(\Enlight_Event_EventArgs $args)
    {
        $this->get('template')->addTemplateDir(
            $this->getPluginPath() . 'Views/',
            $this->pluginViewsNamespace
        );
    }

    /**
     * Appends the plugin's views path to the list of directories used for frontend template
     * inheritance in Shopware >= 5.
     *
     * @param \Enlight_Event_EventArgs $args
     * @return array
     */
    public function onFilterThemeInheritenceTemplateDirectories(\Enlight_Event_EventArgs $args)
    {
        $templateDirectories = $args->getReturn();
        $templateDirectories[] = $this->getPluginPath() . 'Views/';

        return $templateDirectories;
    }

    /**
     * Adds the plugin's style files to the processing queue of
     * shopware's less compiler.
     *
     * @param \Enlight_Event_EventArgs $args
     * @return \Doctrine\Common\Collections\ArrayCollection
     */
    public function addLessFiles(\Enlight_Event_EventArgs $args)
    {
        $lessFiles = static::getLessFiles();
        if (count($lessFiles) === 0) {
            // No less files to be processed
            return new \Doctrine\Common\Collections\ArrayCollection();
        }

        $less = new \Shopware\Components\Theme\LessDefinition(
            [],
            array_map(function ($lessFile) {
                return $this->getPluginPath() . $lessFile;
            }, $lessFiles),
            $this->getPluginPath()
        );

        return new \Doctrine\Common\Collections\ArrayCollection([$less]);
    }
}
