<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonCommon\Classes\Util;

use ReflectionClass;
use Shopware\Models\Shop\Locale;

/**
 * Some helpers for working with locales and language codes.
 */
class Localization
{
    // Lazy-init structure because PHP 5.3 does not support const arrays
    private static $languageToLocale = null;
    private static $languageToDateFormat = null;

    private static function getLanguageToLocaleMap()
    {
        if (self::$languageToLocale === null) {
            self::$languageToLocale = [
                // Offical languages of the EU
                // - see https://en.wikipedia.org/wiki/Languages_of_the_European_Union#Official_EU_languages
                // Please sort this array alphabetically by the english-language name of the language, not by key!
                'bg' => 'bg_BG', // Bulgarian - Bulgaria
                'hr' => 'hr_HR', // Croatian - Croatia
                'cs' => 'cs_CZ', // Czech - Czech Republic
                'da' => 'da_DK', // Danish - Denmark
                'nl' => 'nl_NL', // Dutch - Netherlands
                'en' => 'en_GB', // English - United Kingdom
                'et' => 'et_EE', // Estonian - Estonia
                'fi' => 'fi_FI', // Finnish - Finland
                'fr' => 'fr_FR', // French - France
                'de' => 'de_DE', // German - Germany
                'el' => 'el_GR', // Greek - Greece
                'hu' => 'hu_HU', // Hungarian - Hungary
                'ga' => 'ga_IE', // Irish - Ireland
                'it' => 'it_IT', // Italian - Italy
                'lv' => 'lv_LV', // Latvian - Latvia
                'mt' => 'mt_MT', // Maltese - Malta
                'pl' => 'pl_PL', // Polish - Poland
                'pt' => 'pt_PT', // Portuguese - Portugal
                'ro' => 'ro_RO', // Romanian - Romania
                'sk' => 'sk_SK', // Slovak - Slovakia
                'sl' => 'sl_SL', // Slovenian - Slovenia
                'es' => 'es_ES', // Spanish - Spain
                'sv' => 'sv_SE', // Swedish - Sweden
            ];
        }

        return self::$languageToLocale;
    }

    /**
     * Returns the Shopware-specific primary locale for a language. For example, when given "de", this returns "de_DE",
     * and when given "en", this returns "en_GB".
     *
     * @param string $languageCode an ISO 639-1 language code
     * @return string|null a locale identifier "xx_YY", where xx is an ISO 639-1 language code and YY is an
     * ISO 3166-1 alpha-2 country code, or null if the primary locale is unknown for that language
     */
    public static function getPrimaryLocaleForLanguage($languageCode)
    {
        $languageToLocaleMap = self::getLanguageToLocaleMap();

        return isset($languageToLocaleMap[$languageCode]) ? $languageToLocaleMap[$languageCode] : null;
    }

    /**
     * Returns a more general language code in the form "xx" for a given specific locale in the form "xx_YY". E.g. the
     * specific english locale "en_GB" returns the language code "en".
     *
     * @param string $locale
     * @return string
     */
    public static function getLanguageCode($locale)
    {
        $localeStringParts = explode('_', $locale);

        return $localeStringParts[0];
    }

    /**
     * Returns the date format for a given locale.
     *
     * @param string $locale
     * @return string
     */
    public static function getDateFormat($locale)
    {
        if (self::$languageToDateFormat === null) {
            self::$languageToDateFormat = [
                // Offical languages of the EU
                // - see https://en.wikipedia.org/wiki/Languages_of_the_European_Union#Official_EU_languages
                'bg' => 'd.m.Y', // Bulgarian - Bulgaria
                'hr' => 'd. m. Y.', // Croatian - Croatia
                'cs' => 'd. m. Y.', // Czech - Czech Republic
                'da' => 'd.m.Y', // Danish - Denmark
                'nl' => 'd-m-Y', // Dutch - Netherlands
                'en' => 'd/m/Y', // English - United Kingdom
                'et' => 'd.m.Y', // Estonian - Estonia
                'fi' => 'd.m.Y', // Finnish - Finland
                'fr' => 'd/m/Y', // French - France
                'de' => 'd.m.Y', // German - Germany
                'el' => 'd/m/Y', // Greek - Greece
                'hu' => 'Y. m. d.', // Hungarian - Hungary
                'ga' => 'd/m/Y', // Irish - Ireland
                'it' => 'd/m/Y', // Italian - Italy
                'lv' => 'd.m.Y', // Latvian - Latvia
                'mt' => 'd/m/Y', // Maltese - Malta
                'pl' => 'd.m.Y', // Polish - Poland
                'pt' => 'd/m/Y', // Portuguese - Portugal
                'ro' => 'd.m.Y', // Romanian - Romania
                'sk' => 'd.m.Y', // Slovak - Slovakia
                'sl' => 'd. m. Y.', // Slovenian - Slovenia
                'es' => 'd/m/Y', // Spanish - Spain
                'sv' => 'Y-m-d', // Swedish - Sweden
            ];
        }

        return self::$languageToDateFormat[mb_substr($locale, 0, 2)] ?: 'd/m/Y';
    }

    /**
     * @return Locale The Locale of the global snippet manager
     */
    public function getActiveLocale()
    {
        $snippetManager = Shopware()->Snippets();
        $reflectionClass = new ReflectionClass($snippetManager);

        return $reflectionClass->getProperty('locale')->getValue();
    }
}
