<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonCommon\Classes\Util;

use Shopware\Components\Model\ModelManager;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Models\Order\Detail as OrderDetail;

class OrderDetailUtil
{
    const ORDER_DETAIL_MODE_DEFAULT_ARTICLE = 0;
    const ORDER_DETAIL_MODE_PREMIUM_ARTICLE = 1;
    const ORDER_DETAIL_MODE_VOUCHER = 2;
    const ORDER_DETAIL_MODE_CUSTOMER_GROUP_DISCOUNT = 3;
    const ORDER_DETAIL_MODE_PAYMENT_SURCHARGE_DISCOUNT = 4;
    const ORDER_DETAIL_MODE_BUNDLE_DISCOUNT = 10;
    const ORDER_DETAIL_MODE_TRUSTED_SHOPS_ARTICLE = 12;

    const ORDER_DETAIL_MODES_NON_ARTICLE = [
        self::ORDER_DETAIL_MODE_VOUCHER,
        self::ORDER_DETAIL_MODE_CUSTOMER_GROUP_DISCOUNT,
        self::ORDER_DETAIL_MODE_PAYMENT_SURCHARGE_DISCOUNT,
        self::ORDER_DETAIL_MODE_BUNDLE_DISCOUNT,
        self::ORDER_DETAIL_MODE_TRUSTED_SHOPS_ARTICLE,
    ];

    const ORDER_DETAIL_STATUS_ID_OPEN = 0;
    const ORDER_DETAIL_STATUS_ID_IN_PROCESS = 1;
    const ORDER_DETAIL_STATUS_ID_CANCELLED_REJECTED = 2;
    const ORDER_DETAIL_STATUS_ID_COMPLETED = 3;

    /**
     * @param OrderDetail $orderDetail
     * @return bool true iff the order detail belongs to an article detail (and therefore is a shippable item).
     */
    public static function isOrderDetailForArticleDetail(OrderDetail $orderDetail)
    {
        return !in_array($orderDetail->getMode(), self::ORDER_DETAIL_MODES_NON_ARTICLE, true);
    }

    /**
     * Returns the ArticleDetail of an OrderDetail if it exists.
     *
     * The order number of the article_detail is used as reference.
     *
     * @param OrderDetail $orderDetail
     * @return ArticleDetail|null
     */
    public static function getArticleDetailForOrderDetail(OrderDetail $orderDetail)
    {
        // For several reason we will NO LONGER use $orderDetail->getArticleDetail() to get the article detail. Only
        // the reference via order number is used.
        // Reasons:
        // * Because of backwards compatibility and bugs in Shopware, there is no FK constraint for this reference. This
        //   leads to  broken references when deleting an article detail or regenerating the variants of an article.
        // * When changing the article in the order, the reference also breaks, because this only swaps the article
        //   number, name and price, but not the articleDetailId.
        // * Fixing issues because of a broken reference between articleDetail and orderDetail is not as easy as it
        //   was before, because now you need database access to edit the "hidden" reference. When using the reference
        //   via order number, the customers can fix the problem by themselves.
        // See also: https://github.com/VIISON/ShopwareCommon/pull/283 and
        // https://github.com/VIISON/ShopwarePickwareERP/issues/1233

        /** @var ModelManager $entityManager */
        $entityManager = Shopware()->Container()->get('models');

        return $entityManager->getRepository(ArticleDetail::class)->findOneBy([
            'number' => $orderDetail->getArticleNumber(),
        ]);
    }
}
