<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonCommon\Components\FileStorage;

use Shopware\Plugins\ViisonCommon\Classes\Exceptions\FileSystemExceptions\FileNotReadableException;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\FileSystemExceptions\FileNotStreamableException;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\FileSystemExceptions\FileNotWritableException;
use Shopware\Plugins\ViisonCommon\Classes\FileResponseStream;

/**
 * Interface to represent a class that can read and write files.
 */
interface FileStorage
{
    /**
     * @param string $fileName
     * @return string the content of the file given via parameter $fileName
     * @throws FileNotReadableException if file does not exists or there are no sufficient permissions to read the file
     */
    public function readFileContents($fileName);

    /**
     * Writes the $contents to the file given via $fileName.
     *
     * @param string $fileName
     * @param string $contents
     * @throws FileNotWritableException if file does not exists or there are no sufficient permissions to write the file
     */
    public function writeFileContents($fileName, $contents);

    /**
     * Streams the file given via $filename to the FileResponseStream $fileResponseStream.
     *
     * @param string $fileName
     * @param FileResponseStream $fileResponseStream
     * @throws FileNotReadableException if file does not exists or there are no sufficient permissions to read the file
     * @throws FileNotStreamableException if any error during the streaming occurs
     */
    public function streamFile($fileName, FileResponseStream $fileResponseStream);

    /**
     * Deletes the file given via $filename.
     *
     * @param string $fileName
     * @throws FileNotReadableException if file does not exists
     * @throws FileNotWritableException if there are no sufficient permissions to delete the file
     */
    public function deleteFile($fileName);

    /**
     * @param string $fileName
     * @return bool true iff the file given via parameter $fileName exists.
     */
    public function doesFileExist($fileName);
}
