<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonCommon\Subscribers\Components;

use Shopware\Plugins\ViisonCommon\Classes\Subscribers\TemplateMailComponent;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;

class TemplateMailComponentSubscriber extends TemplateMailComponent
{
    /**
     * This subscriber automatically sets the template context variable 'viisonConfigShop' for each newly created mail
     * object. Together with the \Smarty_Compiler_Config of ShopwareCommon, this makes the function {config=...}
     * available in mail templates. (And actually uses the correct config values of the shop the mail is created for.)
     *
     * It fixes the $shop argument by setting it to the active default shop if no shop is provided, since the mail
     * creation will fail in SW 5.4.3. This issue is known and already fixed, but will be released with SW 5.4.4.
     *
     * It fixes the $shop argument (if necessary), by filling empty properties of sub shops (i.e. theme
     * inheritance), because they are needed for the mail creation since SW 5.4.3
     *
     * @inheritdoc
     */
    public function handleBeforeCreateMail($mailModelName, \Enlight_Hook_HookArgs $args)
    {
        // In SW5.4.3 the TemplateMailComponents builds the theme for the mail of the given shop without checking
        // whether or not the shop parameter was given (and is not null). This issue is known and already fixed in
        // https://github.com/shopware/shopware/commit/89ad1bcbc26d99410e06c1f5bbdd992a310eeba3 but will be released
        // in SW5.4.4. Until then we fix this by providing the active default shop if no shop is set.
        $shop = $args->get('shop');
        $isShopwareVersion543 = ViisonCommonUtil::assertMinimumShopwareVersion('5.4.3') && !ViisonCommonUtil::assertMinimumShopwareVersion('5.4.4');
        if (!$shop && !$this->get('templatemail')->getShop() && $isShopwareVersion543) {
            $shop = $this->get('models')->getRepository('Shopware\\Models\\Shop\\Shop')->getActiveDefault();
            $args->set('shop', $shop);
        }

        // The TemplateMailComponent build the theme inheritance of a given shop when creation a mail (SW 5.4.3, commit:
        // https://github.com/shopware/shopware/commit/ed2e80e974470eb3451f1a9afd9d3dd1e34d2c14). If a given shop does
        // not have an own template (e.g. language sub shop) the mail creation would crash. We need to "fill" these
        // empty properties by using the Shop Repository helper function. We apply the fix for SW versions after 5.3.4
        // because the getById() function is available since then.
        $shop = $args->get('shop');
        if ($shop && !$shop->getTemplate() && ViisonCommonUtil::assertMinimumShopwareVersion('5.3.4')) {
            $shop = $this->get('models')->getRepository('Shopware\\Models\\Shop\\Shop')->getById($shop->getId());
            $args->set('shop', $shop);
        }

        // Set the shop argument as 'viisonConfigShop' template variable
        $context = $args->get('context');
        $context['viisonConfigShop'] = $shop;
        $args->set('context', $context);
    }
}
