<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonCommon\Subscribers;

use Monolog\Logger;
use Shopware\Components\DependencyInjection\Container;
use Shopware\Plugins\ViisonCommon\Classes\LoggerFactory;
use Shopware\Plugins\ViisonCommon\Classes\Subscribers\Services as ServicesSubscriber;
use Shopware\Plugins\ViisonCommon\Components\ControllerFactoryService;
use Shopware\Plugins\ViisonCommon\Components\DocumentComponentListenerService;
use Shopware\Plugins\ViisonCommon\Components\ExceptionTranslation\ExceptionTranslator;
use Shopware\Plugins\ViisonCommon\Components\FileStorage\FileStorageServiceFactory;
use Shopware\Plugins\ViisonCommon\Components\GlobalLocking\GlobalLockingService;
use Shopware\Plugins\ViisonCommon\Components\HiddenConfigStorageService;
use Shopware\Plugins\ViisonCommon\Components\ImageService;
use Shopware\Plugins\ViisonCommon\Components\Migration\MigrationService;
use Shopware\Plugins\ViisonCommon\Components\PluginConfigImporterService;
use Shopware\Plugins\ViisonCommon\Components\Plugins\SwagImportExportProfileService;
use Shopware\Plugins\ViisonCommon\Components\SubApplicationLoading\JsLoaderService;
use Shopware\Plugins\ViisonCommon\Components\SubApplicationLoading\LegacyJsLoaderService;
use Shopware\Plugins\ViisonCommon\Components\SubApplicationLoading\SubApplicationCodeGenerator\SubApplicationCodeGeneratorFactoryService;

class Services extends ServicesSubscriber
{
    /**
     * @see \Shopware\Plugins\ViisonCommon\Classes\Subscribers\Services::getServices()
     */
    public static function getServices()
    {
        return [
            'viison_common.document_component_listener_service' => function (Container $container) {
                return new DocumentComponentListenerService(
                    $container->get('events'),
                    $container->get('db')
                );
            },
            'viison_common.document_rendering_context_factory' => 'Shopware\\Plugins\\ViisonCommon\\Components\\DocumentRenderingContextFactoryService',
            'viison_common.exception_translator' => function (Container $container) {
                return new ExceptionTranslator($container->get('snippets'));
            },
            // viison_common.global_locking is deprecated, please see GlobalLockingService for details
            'viison_common.global_locking' => function (Container $container) {
                $skipLockStillHeldValidation = false;
                if ($container->hasParameter('shopware.viisoncommon.globalLocking.skipLockStillHeldValidation')) {
                    $skipLockStillHeldValidation = boolval(
                        $container->getParameter('shopware.viisoncommon.globalLocking.skipLockStillHeldValidation')
                    );
                }

                return new GlobalLockingService(
                    $container->get('dbal_connection'),
                    $container->get('viison_common.logger'),
                    $skipLockStillHeldValidation
                );
            },
            'viison_common.js_loader' => function (Container $container) {
                return new JsLoaderService(
                    $container->get('events'),
                    $container->get('ViisonCommonJSLoader'),
                    $container->get('viison_common.sub_application_code_generator_factory')
                );
            },
            'viison_common.image_service' => function (Container $container) {
                return new ImageService(
                    $container->get('dbal_connection'),
                    $container->get('models'),
                    $container->get('shopware_storefront.context_service'),
                    $container->get('shopware_storefront.media_service'),
                    $container->get('shopware_media.media_service')
                );
            },
            'viison_common.plugin_config_importer' => function (Container $container) {
                return new PluginConfigImporterService(
                    $container->get('events'),
                    $container->get('shopware_plugininstaller.plugin_manager')
                );
            },
            'viison_common.snippet_manager' => 'Shopware\\Plugins\\ViisonCommon\\Components\\Localization\\SnippetManagerService',
            'viison_common.sub_application_code_generator_factory' => function (Container $container) {
                return new SubApplicationCodeGeneratorFactoryService(
                    $container,
                    $container->get('front'),
                    $container->get('loader'),
                    $container->get('events'),
                    $container->get('viison_common.controller_factory_service'),
                    $container->get('db'),
                    $container->get('snippets'),
                    $container->get('plugins')
                );
            },
            'ViisonCommonJSLoader' => function (Container $container) {
                return new LegacyJsLoaderService($container->get('events'));
            },
            'viison_common.hidden_config_storage' => function (Container $container) {
                return new HiddenConfigStorageService($container->get('models'));
            },
            'viison_common.logger' => function (Container $container) {
                // Make log level configurable via config.php:
                // 'viisonCommon' => [
                //     'logger' => [
                //         'level' => Logger::DEBUG,
                //     ],
                // ],
                // The "shopware." prefix is added by Shopware
                if ($container->hasParameter('shopware.viisoncommon.logger.level')) {
                    $logLevel = $container->getParameter('shopware.viisoncommon.logger.level');
                } else {
                    $environment = $container->getParameter('kernel.environment');
                    $logLevel = ($environment === 'production') ? Logger::INFO : Logger::DEBUG;
                }

                $loggerFactory = new LoggerFactory($container);

                return $loggerFactory->createLikePluginLogger('pickware', $logLevel);
            },
            'viison_common.migration_service' => function (Container $container) {
                return new MigrationService(
                    $container->get('events'),
                    $container->get('db'),
                    $container->get('models'),
                    $container->get('viison_common.logger')
                );
            },
            'viison_common.swag_import_export_profile_service' => function (Container $container) {
                return new SwagImportExportProfileService(
                    $container
                );
            },
            'viison_common.document_file_storage_service' => function (Container $container) {
                $storageServiceFactory = new FileStorageServiceFactory($container);

                return $storageServiceFactory->createDocumentFileStorageService();
            },
            'viison_common.controller_factory_service' => function (Container $container) {
                return new ControllerFactoryService($container);
            },
        ];
    }
}
