<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareMobile\Classes\ApiRequestCompatibility;

use Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrder;
use Shopware\Models\Shop\Currency;
use Shopware\Plugins\ViisonCommon\Classes\Util\Currency as CurrencyUtil;
use Shopware\Plugins\ViisonPickwareCommon\Classes\ApiRequestCompatibility\Layer;

class SupplierOrders20190328 extends Layer
{
    /**
     * @inheritdoc
     */
    public function getVersion()
    {
        return '2019-03-28';
    }

    /**
     * Converts all purchase prices of the fetched supplier order items from the foreign supplier order currency to the
     * default currency if necessary. This is needed for older app version that do not display foreign currencies.
     *
     * post GET /api/supplierOrders
     */
    public function postIndexAction()
    {
        $data = $this->view->data;
        if (!is_array($data)) {
            return;
        }

        /** @var Currency $defaultCurrency */
        $defaultCurrency = CurrencyUtil::getDefaultCurrency();

        foreach ($data as &$supplierOrder) {
            if (!isset($supplierOrder['items'])) {
                continue;
            }

            $supplierOrderCurrency = $supplierOrder['currency'];
            if ($supplierOrderCurrency['id'] === $defaultCurrency->getId()) {
                continue;
            }

            // Use temporary currency entity to make use of the conversion utility function
            /** @var Currency $temporaryCurrencyEntity */
            $temporaryCurrencyEntity = new Currency();
            $temporaryCurrencyEntity = $temporaryCurrencyEntity->fromArray($supplierOrderCurrency);

            foreach ($supplierOrder['items'] as &$supplierOrderItem) {
                $supplierOrderItem['price'] = CurrencyUtil::convertAmountBetweenCurrencies(
                    $supplierOrderItem['price'],
                    $temporaryCurrencyEntity,
                    $defaultCurrency
                );
            }
            unset($supplierOrderItem);

            // Remove unnecessary currency information
            unset($supplierOrder['currency']);
        }
        unset($supplierOrder);

        $this->view->data = $data;
    }

    /**
     * Converts all purchase prices of the given supplier order item from the default currency to foreign currency of
     * the supplier order if necessary. This is needed for older app versions that do not handle foreign currencies.
     *
     * pre POST /api/supplierOrders/{id}/items/{items}/purchases
     */
    public function prePostItemsPurchasesAction()
    {
        $supplierOrderId = $this->request->getParam('id', 0);
        /** @var SupplierOrder $supplierOrderItem */
        $supplierOrder = $this->get('models')->find(SupplierOrder::class, $supplierOrderId);
        if (!$supplierOrder) {
            return;
        }
        /** @var Currency $defaultCurrency */
        $defaultCurrency = CurrencyUtil::getDefaultCurrency();
        if ($defaultCurrency->getId() === $supplierOrder->getCurrency()->getId()) {
            return;
        }

        $convertedPurchasePrice = CurrencyUtil::convertAmountBetweenCurrencies(
            $this->request->get('price'),
            $defaultCurrency,
            $supplierOrder->getCurrency()
        );

        $this->request->setParam('price', $convertedPurchasePrice);
    }
}
