<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareMobile\Components\PickingBoxBarcodeLabel;

use Exception;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\AbstractBarcodeLabelType;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\BarcodeRenderer;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\BarcodeLabelField;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\BarcodeLabelFieldTranslation;
use Shopware\Plugins\ViisonPickwareERP\Components\PluginConfig\PluginConfig as PickwareErpConfig;
use Shopware_Components_Snippet_Manager;

class PickingBoxBarcodeLabelType extends AbstractBarcodeLabelType
{
    use BarcodeLabelFieldTranslation;

    /**
     * Barcode label identifier for picking box labels.
     */
    const IDENTIFIER = 'picking_box';

    /**
     * Namespace containing all picking box field descriptions.
     */
    const SNIPPETS_NAMESPACE = 'backend/viison_pickware_mobile_picking_box_barcode_label_printing/main';

    /**
     * @var BarcodeRenderer
     */
    private $barcodeRenderer;

    /**
     * @param PickwareErpConfig $pickwareErpConfig
     * @param Shopware_Components_Snippet_Manager $snippetManager
     * @param BarcodeRenderer $barcodeRenderer
     */
    public function __construct(
        PickwareErpConfig $pickwareErpConfig,
        $snippetManager,
        BarcodeRenderer $barcodeRenderer
    ) {
        parent::__construct(self::IDENTIFIER, false, $pickwareErpConfig->getBarcodeLabelTemplateDirectories());

        $this->setNamespace($snippetManager, self::SNIPPETS_NAMESPACE);
        $this->barcodeRenderer = $barcodeRenderer;
    }

    /**
     * @inheritdoc
     */
    public function createItemProvider(array $params, array $filter, array $sort)
    {
        $start = intval($params['start']);
        $end = intval($params['end']);

        if ($start < 1) {
            throw new \InvalidArgumentException(
                sprintf('Picking boxes enumeration starts with 1. The starting value of %d is not valid.', $start)
            );
        }

        if ($start > $end) {
            throw new Exception(
                sprintf('The ending picking box number "%d" can not be bigger than the starting picking box number "%d".', $start, $end)
            );
        }

        return new PickingBoxItemProvider($this->barcodeRenderer, $this, $start, $end);
    }

    /**
     * @inheritdoc
     *
     * @deprecated Use `createItemProvider()` instead.
     *
     * WARNING: When removing this method make absolutely sure to create a version constraint for ERP to ensure the
     * `AbstractBarcodeLabelType` has its `createItemProviderForFilter` method declared as non-abstract or removed.
     */
    public function createItemProviderForFilter(array $params, array $filter)
    {
        return $this->createItemProvider($params, $filter, []);
    }

    /**
     * @inheritdoc
     */
    public function createItemProviderForPreview(array $params)
    {
        throw new Exception('Preview is not supported for picking box barcode labels.');
    }

    /**
     * @inheritdoc
     */
    public function getFields()
    {
        return [
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$pickingBoxNumber}',
                '{$cellItem.pickingBoxNumber}',
                $this->getFieldDescription('field/picking_box/picking_box_number')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_BARCODE,
                '{$barcode}',
                '<img class="is--barcode is--stretchable-barcode" src="{$cellItem.barcode}" />',
                $this->getFieldDescription('field/picking_box/barcode')
            ),
        ];
    }

    /**
     * @inheritdoc
     */
    public function enqueueForPrinting($identifier, $quantity = 1)
    {
        throw new Exception('Picking box labels can not be enqueued for printing.');
    }
}
