<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonPickwareMobile\PickedQuantity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Shopware\Components\Model\ModelEntity;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation;
use Shopware\Models\Order\Detail as OrderDetail;

/**
 * @ORM\Entity(repositoryClass="PickedQuantityRepository")
 * @ORM\Table(name="pickware_wms_picked_quantities")
 */
class PickedQuantity extends ModelEntity
{
    /**
     * @var int $id
     *
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\GeneratedValue(strategy="IDENTITY")
     * @ORM\Id
     */
    protected $id;

    /**
     * @var int $orderDetailId
     *
     * @ORM\Column(name="orderDetailId", type="integer", nullable=false)
     */
    protected $orderDetailId;

    /**
     * OWNING SIDE
     *
     * @var OrderDetail $orderDetail
     *
     * @ORM\ManyToOne(targetEntity="Shopware\Models\Order\Detail")
     * @ORM\JoinColumn(name="orderDetailId", referencedColumnName="id", onDelete="CASCADE")
     */
    protected $orderDetail;

    /**
     * @var int $binLocationId
     *
     * @ORM\Column(name="binLocationId", type="integer", nullable=true)
     */
    protected $binLocationId;

    /**
     * OWNING SIDE
     *
     * @var BinLocation $binLocation
     *
     * @ORM\ManyToOne(targetEntity="Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation")
     * @ORM\JoinColumn(name="binLocationId", referencedColumnName="id", onDelete="SET NULL")
     */
    protected $binLocation;

    /**
     * @var int $quantity
     *
     * @ORM\Column(name="quantity", type="integer", nullable=false)
     */
    protected $quantity = 0;

    /**
     * INVERSE SIDE
     *
     * @var ArrayCollection $stockEntries
     *
     * @ORM\OneToMany(targetEntity="Shopware\CustomModels\ViisonPickwareMobile\PickedQuantity\StockLedgerEntryMapping", mappedBy="pickedQuantity")
     */
    protected $stockLedgerEntryMappings;

    /**
     * @param OrderDetail $orderDetail;
     * @param BinLocation|null $stockEntry;
     */
    public function __construct(OrderDetail $orderDetail, BinLocation $binLocation = null, $quantity = 0)
    {
        $this->orderDetail = $orderDetail;
        $this->binLocation = $binLocation;
        $this->quantity = $quantity;
        $this->stockLedgerEntryMappings = new ArrayCollection();

        // Explicitely set the both IDs which form the unique identifier of this entity, because Doctrine
        // won't do it, even though we have already set the assocaited entities
        $this->orderDetailId = $orderDetail->getId();
        $this->binLocationId = ($binLocation) ? $binLocation->getId() : null;
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return OrderDetail
     */
    public function getOrderDetail()
    {
        return $this->orderDetail;
    }

    /**
     * @param OrderDetail $orderDetail
     */
    public function setOrderDetail(OrderDetail $orderDetail)
    {
        $this->orderDetail = $orderDetail;
    }

    /**
     * @return BinLocation
     */
    public function getBinLocation()
    {
        return $this->binLocation;
    }

    /**
     * @param BinLocation $binLocation
     */
    public function setBinLocation(BinLocation $binLocation)
    {
        $this->binLocation = $binLocation;
    }

    /**
     * @return int
     */
    public function getQuantity()
    {
        return $this->quantity;
    }

    /**
     * @param int $quantity
     */
    public function setQuantity($quantity)
    {
        $this->quantity = $quantity;
    }

    /**
     * @return ArrayCollection
     */
    public function getStockLedgerEntryMappings()
    {
        return $this->stockLedgerEntryMappings;
    }
}
