<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareMobile\Subscribers;

use Shopware\Plugins\ViisonCommon\Classes\Subscribers\CompatibilityCheck as AbstractCompatibilityCheck;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonCommon\Structs\PluginCompatibilityConstraint;

class CompatibilityCheckSubscriber extends AbstractCompatibilityCheck
{
    /**
     * @inheritdoc
     */
    public function getPluginCompatibilityConstraints()
    {
        return [
            new PluginCompatibilityConstraint(
                'Backend',
                'RhiemRefund',
                '1.8.1'
            ),
            new PluginCompatibilityConstraint(
                'Frontend',
                'TonurAutomatedSendingOfStatusMails5',
                '9999.9999.9999', // Mark as "incompatible"
                'Das Plugin "Automatischer E-Mailversand" ist nicht kompatibel mit Pickware. Bitte deaktivieren oder deinstallieren Sie das Plugin.'
            ),
            $this->getNonPremiumDHLAdapterConstraint(),
        ];
    }

    /**
     * Returns a plugin compatibility constraint that checks if "DHL Adapter" is installed and not "DHL Adapter
     * Premium". We need to inform the customer that the usage of Pickware WMS together with "DHL Adapter" but without
     * "DHL Adapter Premium" results in limited functionality. The growl message displays a link to the documentation of
     * the DHL Adapter.
     *
     * @return PluginCompatibilityConstraint
     */
    private function getNonPremiumDHLAdapterConstraint()
    {
        $nonPremiumDHLAdapterConstraint = new PluginCompatibilityConstraint();
        $namespace = $this->get('viison_common.snippet_manager')->getNamespace(
            'viison_pickware_mobile/compatibility_check/viison_dhl'
        );
        $nonPremiumDHLAdapterConstraint->setCustomMessage($namespace->get('incompatibility_growl/message'));
        $nonPremiumDHLAdapterConstraint->setLinkTitle($namespace->get('incompatibility_growl/link/title'));
        $nonPremiumDHLAdapterConstraint->setLink($namespace->get('incompatibility_growl/link/url'));
        $nonPremiumDHLAdapterConstraint->setCustomCheck(function () {
            // Returns false (= show growl) as soon as DHL is installed and active without premium.
            return (
                !ViisonCommonUtil::isPluginInstalledAndActive(null, 'ViisonDHL')
                || ViisonCommonUtil::isPluginInstalledAndActive(null, 'ViisonDHLPremium')
            );
        });

        return $nonPremiumDHLAdapterConstraint;
    }
}
