// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareMobileAnalyticsPickTimes.view.table.PickTimes', {

    extend: 'Shopware.apps.Analytics.view.main.Table',
    alias: 'widget.analytics-table-pick-times',
    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
        'Shopware.apps.ViisonCommonAnalytics.view.main.TableHelpers',
    ],
    features: [
        { ftype: 'summary' },
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_mobile_analytics/pick_times',

    initComponent: function () {
        this.columns = {
            items: this.getColumns(),
            defaults: {
                flex: 1,
                sortable: true,
            },
        };

        this.callParent(arguments);
    },

    /**
     * Parses the time in seconds and returns a formatted string like "1h 12m 34s"
     * Returns "-" if no value is given.
     *
     * @param value
     * @returns string
     */
    timeRenderer: function (value) {
        if (value === null || Number.isNaN(value) || typeof value !== 'number') {
            return '-';
        }

        var hours = Math.floor(value / (60 * 60));
        var minutes = Math.floor((value % (60 * 60)) / 60);
        var seconds = value % 60;
        var result = seconds + 's';
        if (minutes > 0 || hours > 0) {
            result = minutes + 'm ' + result;
        }
        if (hours > 0) {
            result = hours + 'h ' + result;
        }

        return result;
    },

    /**
     * Renders the given value with 1 decimal (0.0) and returns "-" if no correct value was given.
     *
     * @param value
     * @returns string
     */
    nullableRenderer: function (value) {
        if (value === null || Number.isNaN(value) || typeof value !== 'number') {
            return '-';
        }

        return Ext.util.Format.number(value, '0.0');
    },

    getColumns: function () {
        function makeSummaryTypeFunction(forDataIndex) {
            return function () {
                return this.store.getProxy().getReader().rawData.summary[forDataIndex];
            };
        }
        function makeSummaryColumnName() {
            return function () {
                return this.getViisonSnippet('table/total');
            };
        }

        return [
            {
                dataIndex: 'name',
                text: this.getViisonSnippet('table/name'),
                align: 'left',
                summaryType: makeSummaryColumnName().bind(this),
            },
            {
                dataIndex: 'workdays',
                text: this.getViisonSnippet('table/workdays'),
                tooltip: this.getViisonSnippet('tooltip/workdays'),
                align: 'right',
                summaryType: makeSummaryTypeFunction('workdays').bind(this),
            },
            {
                dataIndex: 'pickedOrders',
                text: this.getViisonSnippet('table/pickedOrders'),
                tooltip: this.getViisonSnippet('tooltip/pickedOrders'),
                align: 'right',
                summaryType: makeSummaryTypeFunction('pickedOrders').bind(this),
            },
            {
                dataIndex: 'packedOrders',
                text: this.getViisonSnippet('table/packedOrders'),
                tooltip: this.getViisonSnippet('tooltip/packedOrders'),
                align: 'right',
                summaryType: makeSummaryTypeFunction('packedOrders').bind(this),
            },
            {
                dataIndex: 'averagePickTimeInSeconds',
                text: this.getViisonSnippet('table/averagePickTime'),
                tooltip: this.getViisonSnippet('tooltip/averagePickTime'),
                align: 'right',
                renderer: this.timeRenderer,
                summaryType: makeSummaryTypeFunction('averagePickTimeInSeconds').bind(this),
                summaryRenderer: this.timeRenderer,
            },
            {
                dataIndex: 'picksPerWorkday',
                text: this.getViisonSnippet('table/picksPerWorkday'),
                align: 'right',
                renderer: this.nullableRenderer,
                summaryType: makeSummaryTypeFunction('picksPerWorkday').bind(this),
                summaryRenderer: this.nullableRenderer,
                hidden: true,
            },
            {
                dataIndex: 'packsPerWorkday',
                text: this.getViisonSnippet('table/packsPerWorkday'),
                align: 'right',
                renderer: this.nullableRenderer,
                summaryType: makeSummaryTypeFunction('packsPerWorkday').bind(this),
                summaryRenderer: this.nullableRenderer,
                hidden: true,
            },
            {
                xtype: 'actioncolumn',
                sortable: false,
                align: 'right',
                items: [
                    {
                        iconCls: 'sprite-user--pencil',
                        tooltip: this.getViisonSnippet('tooltip/openUser'),
                        handler: function () {
                            Shopware.app.Application.addSubApplication({
                                name: 'Shopware.apps.UserManager',
                                action: 'load',
                            });
                        },
                    },
                ],
            },
        ];
    },

});
