// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

/**
 * A custom grid column, which visualizes the the custom field 'viisonPickwareMobileIsDisplayedInPickingApp' by displaying a
 * de-/activated Picking app icon with a Qtip.
 */
Ext.define('Shopware.apps.ViisonPickwareMobileOrderPickingFilterVisualization.view.PickingVisibilityColumn', {

    extend: 'Ext.grid.column.Column',
    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_mobile_order_picking_filter_visualization',
    width: 25,
    dataIndex: 'viisonPickwareMobileIsDisplayedInPickingApp',
    sortable: false,
    snippets: {
        visible: '',
        hidden: '',
    },

    /**
     * Changes the scope to this, to be able to access the snippets in the renderer.
     */
    initComponent: function () {
        this.scope = this;
        this.menuText = '<img alt="' + this.getViisonSnippet('column/picking_app_visiblity/header') + '" src="data:image/gif;base64,R0lGODlhAQABAID/AMDAwAAAACH5BAEAAAAALAAAAAABAAEAAAICRAEAOw==" class="c-sprite-picking-visible">';

        this.callParent(arguments);
    },

    /**
     * A custom renderer, which selects the icon class corresponding to the visibility
     * of the object, which is represented by the record, in the Picking app.
     * Additional it adds a Qtip to that icon, explaining it and showing the
     * unmet conditions, if any exist.
     *
     * @param value The value for the column in the currently rendered row.
     * @param metaData The meta data of the rendered cell.
     * @param record The record of the rendered row.
     * @return The HTML, which will be rendered in the status cell.
     */
    renderer: function (value, metaData, record) {
        // Add a Qtip to the cell
        var message;
        if (value) {
            // Just use the 'visible' message
            message = '<b>' + this.snippets.visible + '</b>';
        } else {
            // Use the 'hidden' message and list all unmet conditions
            message = '<b>' + this.snippets.hidden + '</b><br/>';
            message += this.getPickingFilterDescriptions(record);
        }
        metaData.tdAttr = 'data-qtip="' + message + '"';

        // Select the icon class and create a new element
        var iconCls = (value) ? 'c-sprite-picking-visible' : 'c-sprite-picking-hidden';
        var tpl = new Ext.XTemplate('<div class="customer-column-icon ' + iconCls + '"></div>');

        return tpl.html;
    },

    /**
     * Returns a formatted string about the unmet picking conditions for the given record.
     *
     * @param {Ext.data.Model} record
     * @return {string}
     */
    getPickingFilterDescriptions: function (record) {
        // Check if there are no unmet conditions to display (i.e. viisonPickwareMobilePickProfileFilterResult
        // was undefined)
        if (!Ext.isArray(record.raw.viisonPickwareMobilePickProfileFilterResult)) {
            return '';
        }

        // Prefix all descriptions
        var conditions = record.raw.viisonPickwareMobilePickProfileFilterResult.map(function (condition) {
            return '- ' + condition.replace(/"/g, '&quot;');
        });

        return conditions.join('<br/>');
    },

});
