<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonCommon\Classes\Exceptions;

use Shopware\Models\Order\Order;
use Shopware\Models\Document\Document;
use Shopware\Plugins\ViisonCommon\Components\ExceptionTranslation\LocalizableThrowable;

class DocumentException extends \Exception implements LocalizableThrowable
{
    /**
     * @var string[]
     */
    protected $snippetArguments;

    /**
     * @var string
     */
    protected $snippetName;

    /**
     * @param $message
     * @param $snippetName
     * @param array $snippetArguments
     */
    public function __construct($message, $snippetName, array $snippetArguments = [])
    {
        $this->snippetName = $snippetName;
        $this->snippetArguments = $snippetArguments;
        parent::__construct($message);
    }

    /**
     * @inheritdoc
     */
    public function getLocalizedMessage(\Enlight_Components_Snippet_Manager $snippetManager)
    {
        $snippet = $snippetManager->getNamespace('exceptions/viison_common/document_exception')->get(
            $this->snippetName
        );

        return $snippet ? vsprintf($snippet, $this->snippetArguments) : $this->getMessage();
    }

    /**
     * Use this Exception when you try to overwrite a document of a specific type but encounter multiple documents.
     *
     * @param Document $document
     * @param Order $order
     * @return DocumentException
     */
    public static function multipleDocumentsExistForOverwriting(
        Document $document,
        Order $order
    ) {
        return new DocumentException(
            sprintf(
                'Error while overwriting document of type "%s": multiple documents of this type already exist for ' .
                'the given order. Check the order "%s", remove duplicate documents and try again.',
                $document ? $document->getName() : '-',
                $order ? $order->getNumber() : '-'
            ),
            'multiple_documents_exist_for_overwriting',
            [
                $document ? $document->getName() : '-',
                $order ? $order->getNumber() : '-',
            ]
        );
    }

    /**
     * Use this Exception when you try to overwrite a document of a specific type but encounter multiple documents.
     *
     * @param Document $document
     * @param Order $order
     * @return DocumentException
     */
    public static function invoiceDocumentDoesNotExist()
    {
        return new DocumentException(
            'The invoice document type must exist and have ID 1, but no document type could be found for that ID.',
            'invoice_document_not_found'
        );
    }
}
