<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonCommon\Components\ExceptionTranslation;

/**
 * Use this trait as a standard implementation of LocalizableThrowable.
 *
 * This trait must only be used inside a class that implements \Throwable or extends \Exception.
 */
trait MessageLocalization
{
    /**
     * @var string
     */
    protected $snippetName;

    /**
     * @var string
     */
    protected $snippetNamespace;

    /**
     * @var mixed[] Array to store additional pieces of information for the exception.
     */
    protected $snippetArguments;

    /**
     * @var null|\Exception
     * @deprecated Use the third parameter $previous of \Exception::__construct() instead
     */
    protected $previousException = null;

    /**
     * @param \Enlight_Components_Snippet_Manager $snippetManager
     * @return string The localized exception message as translated using $snippetManager
     */
    public function getLocalizedMessage(\Enlight_Components_Snippet_Manager $snippetManager)
    {
        $previousException = $this->getPrevious() ?: $this->previousException;
        $snippetArguments = $this->snippetArguments;
        if ($previousException instanceof LocalizableThrowable) {
            $snippetArguments[] = $previousException->getLocalizedMessage($snippetManager);
        } elseif ($previousException !== null) {
            $snippetArguments[] = $previousException->getMessage();
        } else {
            $snippetArguments[] = '';
        }

        $snippet = $snippetManager->getNamespace($this->snippetNamespace)->get($this->snippetName);

        return $snippet ? vsprintf($snippet, $snippetArguments) : $this->getMessage();
    }

    /**
     * @param string $snippetName Name of the snippet with localized error messages.
     */
    protected function setSnippetName($snippetName)
    {
        $this->snippetName = $snippetName;
    }

    /**
     * @param string $snippetNamespace Namespace of the snippet.
     */
    protected function setSnippetNamespace($snippetNamespace)
    {
        $this->snippetNamespace = $snippetNamespace;
    }

    /**
     * @param mixed[] $snippetArguments An array of values for the placeholders in the snippet string, which will be
     *        substituted into the snippet using vsprintf().
     */
    protected function setSnippetArguments($snippetArguments)
    {
        $this->snippetArguments = $snippetArguments;
    }

    /**
     * @param \Exception|null $previousException
     * @deprecated Use the third parameter of \Exception::_construct() instead
     */
    protected function setPreviousException(\Exception $previousException = null)
    {
        $this->previousException = $previousException;
    }
}
