<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

if (!class_exists('ViisonCommon_Plugin_BootstrapV10')) {
    require_once('PluginBootstrapV10.php');
}

/**
 * Version 11 of the common plugin Bootstrap class. This version is based on ViisonCommon_Plugin_BootstrapV10,
 * but include automatic loading of the localized plugin label.
 *
 * This class needs to be manually loaded in the respective plugin via:
 *
 * if (!class_exists('ViisonCommon_Plugin_BootstrapV11')) {
 *     require_once('ViisonCommon/PluginBootstrapV11.php');
 * }
 */
abstract class ViisonCommon_Plugin_BootstrapV11 extends ViisonCommon_Plugin_BootstrapV10
{
    /**
     * {@inheritdoc}
     *
     * Dynamically selects the correct plugin label from the plugin.json file using the default shop's language.
     */
    public function getLabel()
    {
        if (!class_exists('Shopware\\Plugins\\ViisonCommon\\Classes\\Util\\Util')) {
            return parent::getLabel();
        }

        $pluginInfo = $this->getPluginInfo();
        $language = self::getDefaultShopLanguage();
        if (isset($pluginInfo['label'][$language])) {
            return $pluginInfo['label'][$language];
        } elseif (isset($pluginInfo['label']['en'])) {
            return $pluginInfo['label']['en'];
        } else {
            return parent::getLabel();
        }
    }

    /**
     * {@inheritdoc}
     *
     * Adds the localized plugin label and description to the plugin info.
     */
    public function getInfo()
    {
        $info = parent::getInfo();
        $info['label'] = $this->getLabel();
        $info['description'] = $this->getLocalizedPluginDescription();

        return $info;
    }

    /**
     * Loads and returns the plugin description HTML file in default shop's language. If no such description exists,
     * the english description us used as fallback, if available.
     *
     * @return string|null
     */
    public function getLocalizedPluginDescription()
    {
        if (!class_exists('Shopware\\Plugins\\ViisonCommon\\Classes\\Util\\Util')) {
            return null;
        }

        // Try to find description in default language
        $language = self::getDefaultShopLanguage();
        $defaultLanguageDescriptionPath = sprintf('%sdescription.%s.html', $this->Path(), $language);
        if (is_readable($defaultLanguageDescriptionPath)) {
            return file_get_contents($defaultLanguageDescriptionPath);
        }

        // Try to find an english description as fallback
        $englishDescriptionPath = sprintf('%sdescription.en.html', $this->Path());
        if (is_readable($englishDescriptionPath)) {
            return file_get_contents($englishDescriptionPath);
        }

        return null;
    }

    /**
     * The locale string (e.g. 'de_DE') of the default shop. We 'cache' it statically to prevent unnecessary database
     * queries. Don't access this variable directly, but use {@link self::getDefaultShopLanguage()} instead.
     *
     * @var string|null
     */
    protected static $defaultShopLanguage = null;

    /**
     * Returns the language identifier string (e.g. 'de') of the default shop.
     *
     * @return string
     */
    protected static function getDefaultShopLanguage()
    {
        if (!self::$defaultShopLanguage) {
            $locale = Shopware()->Container()->get('models')->getRepository('Shopware\\Models\\Shop\\Shop')->getDefault()->getLocale()->getLocale();
            list($language, $localization) = explode('_', $locale);
            self::$defaultShopLanguage = $language;
        }

        return self::$defaultShopLanguage;
    }
}
