<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

if (!class_exists('ViisonCommon_Plugin_BootstrapV11')) {
    require_once('PluginBootstrapV11.php');
}

/**
 * Version 12 of the common plugin Bootstrap class. This version is based on ViisonCommon_Plugin_BootstrapV11,
 * but contains a fixed versions several methods intruced by version 11.
 *
 * This class needs to be manually loaded in the respective plugin via:
 *
 * if (!class_exists('ViisonCommon_Plugin_BootstrapV12')) {
 *     require_once('ViisonCommon/PluginBootstrapV12.php');
 * }
 */
abstract class ViisonCommon_Plugin_BootstrapV12 extends ViisonCommon_Plugin_BootstrapV11
{
    /**
     * {@inheritdoc}
     *
     * Overrides the parent implementation to fix the mistake of not using late static binding for static method calls,
     * which breaks or bootstrap version inheritance design.
     */
    public function getLabel()
    {
        if (!class_exists('Shopware\\Plugins\\ViisonCommon\\Classes\\Util\\Util')) {
            return parent::getLabel();
        }

        $pluginInfo = $this->getPluginInfo();
        $language = static::getDefaultShopLanguage();
        if (isset($pluginInfo['label'][$language])) {
            return $pluginInfo['label'][$language];
        } elseif (isset($pluginInfo['label']['en'])) {
            return $pluginInfo['label']['en'];
        } else {
            return parent::getLabel();
        }
    }

    /**
     * {@inheritdoc}
     *
     * Overrides the parent implementation to fix the mistake of not using late static binding for static method calls,
     * which breaks or bootstrap version inheritance design.
     */
    public function getLocalizedPluginDescription()
    {
        if (!class_exists('Shopware\\Plugins\\ViisonCommon\\Classes\\Util\\Util')) {
            return null;
        }

        // Try to find description in default language
        $language = static::getDefaultShopLanguage();
        $defaultLanguageDescriptionPath = sprintf('%sdescription.%s.html', $this->Path(), $language);
        if (is_readable($defaultLanguageDescriptionPath)) {
            return file_get_contents($defaultLanguageDescriptionPath);
        }

        // Try to find an english description as fallback
        $englishDescriptionPath = sprintf('%sdescription.en.html', $this->Path());
        if (is_readable($englishDescriptionPath)) {
            return file_get_contents($englishDescriptionPath);
        }

        return null;
    }

    /**
     * {@inheritdoc}
     *
     * Overrides the parent implementation to fix a bug in Shopware, that is caused by loading the default shop's locale
     * via its Doctrine association to the shop. That way an instance of a proxy is returned instead of an instance
     * the locale model class itself. Since Shopware saves the user locale in the backend session, deserializing the
     * session might break, if the serialized object was in instance of the proxy, because the the proxy class might not
     * be loaded upon deserialization.
     *
     * @see https://github.com/VIISON/ShopwarePickwareCommon/issues/113
     * @see https://github.com/VIISON/ShopwareDHLAdapter/issues/191
     */
    protected static function getDefaultShopLanguage()
    {
        if (!static::$defaultShopLanguage) {
            $locale = Shopware()->Container()->get('db')->fetchOne(
                'SELECT `locale`.`locale`
                FROM `s_core_shops` AS `shop`
                LEFT JOIN `s_core_locales` AS `locale`
                    ON `locale`.`id` = `shop`.`locale_id`
                WHERE `shop`.`default` = 1'
            );
            list($language, $localization) = explode('_', $locale);
            static::$defaultShopLanguage = $language;
        }

        return static::$defaultShopLanguage;
    }
}
