<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareCommon\Components\Encryption;

use Exception;
use Shopware\Plugins\ViisonCommon\Components\ExceptionTranslation\LocalizableThrowable;
use Shopware\Plugins\ViisonCommon\Components\ExceptionTranslation\MessageLocalization;

class EncryptionException extends Exception implements LocalizableThrowable
{
    use MessageLocalization;

    /**
     * @inheritdoc
     */
    public function __construct($message, $code = 0, Exception $previousException = null)
    {
        parent::__construct($message, $code, $previousException);
        $this->setSnippetNamespace('components/encryption');
    }

    /**
     * @param string $openSslError
     * @return self
     */
    public static function keyEncryptionFailed($openSslError)
    {
        $snippetArguments = [$openSslError];
        $exception = new self(vsprintf(
            'Failed to encrypt the payload encryption key: %1$s',
            $snippetArguments
        ));
        $exception->setSnippetName('encryption_exception/key_encryption_failed');
        $exception->setSnippetArguments($snippetArguments);

        return $exception;
    }

    /**
     * @param $numberOfRequestedBytes
     * @param string $openSslError
     * @return self
     */
    public static function acquiringEntropyFailed($numberOfRequestedBytes, $openSslError)
    {
        $snippetArguments = [
            $numberOfRequestedBytes,
            $openSslError,
        ];
        $exception = new self(vsprintf(
            'Failed to acquire %1$d bytes of entropy: %2$s',
            $snippetArguments
        ));
        $exception->setSnippetName('encryption_exception/failed_to_acquire_entropy');
        $exception->setSnippetArguments($snippetArguments);

        return $exception;
    }

    /**
     * @param string $openSslError
     * @return self
     */
    public static function payloadEncryptionFailed($openSslError)
    {
        $snippetArguments = [$openSslError];
        $exception = new self(vsprintf(
            'Failed to encrypt the payload: %1$s',
            $snippetArguments
        ));
        $exception->setSnippetName('encryption_exception/payload_encryption_failed');
        $exception->setSnippetArguments($snippetArguments);

        return $exception;
    }

    /**
     * @param string $openSslError
     * @return self
     */
    public static function rsaPublicKeyLoadingFailed($openSslError)
    {
        $snippetArguments = [$openSslError];
        $exception = new self(vsprintf(
            'Failed to load RSA public key: %1$s',
            $snippetArguments
        ));
        $exception->setSnippetName('encryption_exception/rsa_public_key_loading_failed');
        $exception->setSnippetArguments($snippetArguments);

        return $exception;
    }

    /**
     * @param string $signature
     * @param string $openSslError
     * @return self
     */
    public static function signatureVerificationFailed($signature, $openSslError)
    {
        $snippetArguments = [
            $signature,
            $openSslError,
        ];
        $exception = new self(vsprintf(
            'An error occurred while trying to verify signature "%1$s": %2$s',
            $snippetArguments
        ));
        $exception->setSnippetName('encryption_exception/signature_verification_failed');
        $exception->setSnippetArguments($snippetArguments);

        return $exception;
    }
}
