<?php
/**
 * premsoft
 * Copyright © 2016 Premsoft - Sven Mittreiter
 *
 * According to our dual licensing model, this program can be used either
 * under the terms of the GNU Affero General Public License, version 3,
 * or under a proprietary license.
 *
 * The texts of the GNU Affero General Public License, supplemented by an additional
 * permission, and of our proprietary license can be found
 * in the LICENSE file you have received along with this program.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * "shopware" is a registered trademark of shopware AG.
 * The licensing of the program under the AGPLv3 does not imply a
 * trademark license. Therefore any rights, titles and interests in the
 * above trademarks remain entirely with the trademark owners.
 *
 * @copyright  Copyright (c) 2016, premsoft - Sven Mittreiter (http://www.premsoft.de)
 * @author     Sven Mittreiter <info@premsoft.de>
 */

use Shopware\Components\CSRFWhitelistAware;
use Enlight_View_Default as ViewEngine;
use Shopware\Models\Shop\Shop;

class Shopware_Controllers_Frontend_PremsOnePageCheckout extends Shopware_Controllers_Frontend_Checkout implements CSRFWhitelistAware {

    public function getWhitelistedCSRFActions() {
        return [
            'index',
            'login',
            'saveBillingChanges',
            'saveShippingChanges',
            'shippingPayment',
            'saveAgbStatus',
        ];
    }

    /**
     * Check if a given version is greater or equal to
     * the currently installed shopware version.
     *
     * Attention: If your target shopware version may
     * include a version less than 4.1.3 you have to
     * use assertVersionGreaterThen().
     *
     * @since 4.1.3 introduced assertMinimumVersion($requiredVersion)
     *
     * @param string $requiredVersion string Format: 3.5.4 or 3.5.4.21111
     *
     * @return bool
     */
    protected function assertMinimumVersion($requiredVersion) {
        $version = Shopware()->Config()->version;

        if ($version === '___VERSION___') {
            return true;
        }

        return version_compare($version, $requiredVersion, '>=');
    }

    /**
     * Action zum Speichern des AGB Checked Zustandes.
     * @throws \Exception
     */
    public function saveAgbStatusAction() {
        $this->Front()->Plugins()->ViewRenderer()->setNoRender();
        if ($this->Request()->getParam('agb')) {
            Shopware()->Session()->agbChecked = 1;
        } else {
            Shopware()->Session()->agbChecked = 0;
        }
    }
    /**
     * Action zum Speichern des Kommentar Feld Zustandes.
     * @throws \Exception
     */
    public function saveCommentAction() {
        $this->Front()->Plugins()->ViewRenderer()->setNoRender();
        if ($this->Request()->getParam('comment')) {
            Shopware()->Session()->comment = $this->Request()->getParam('comment');
        } else {
            Shopware()->Session()->comment = '';
        }
    }

    /**
     * PayPalUnified method
     * @return void
     */
    private function addEcButtonBehaviour(ViewEngine $view, \SwagPaymentPayPalUnified\Models\Settings\General $generalSettings)
    {
        $view->assign('paypalUnifiedClientId', $generalSettings->getSandbox() ? $generalSettings->getSandboxClientId() : $generalSettings->getClientId());

        $shop = Shopware()->Shop();

        if (!$shop instanceof Shop) {
            throw new \UnexpectedValueException(sprintf('Expected instance of %s, got null.', Shop::class));
        }

        $view->assign('paypalUnifiedCurrency', $shop->getCurrency()->getCurrency());
    }

    /**
     * PayPalUnified method
     * @return void
     */
    private function addEcButtonStyleInfo(ViewEngine $view, \SwagPaymentPayPalUnified\Models\Settings\ExpressCheckou $expressSettings, \SwagPaymentPayPalUnified\Models\Settings\General $generalSettings)
    {
        $buttonLocaleService = Shopware()->Container()->get('paypal_unified.button_locale_service');
        $view->assign([
            'paypalUnifiedEcButtonStyleColor' => $expressSettings->getButtonStyleColor(),
            'paypalUnifiedEcButtonStyleShape' => $expressSettings->getButtonStyleShape(),
            'paypalUnifiedEcButtonStyleSize' => $expressSettings->getButtonStyleSize(),
            'paypalUnifiedButtonLocale' => $buttonLocaleService->getButtonLocale($expressSettings->getButtonLocale()),
            'paypalUnifiedIntent' => $generalSettings->getIntent(),
        ]);
    }

    /**
     * Action für Kompabitlitätsmodus:
     * Nutzer nicht eingeloggt und weitere Payment Plugins unterstützen
     */
    public function confirmAction() {
        $view = $this->View();
//$this->forward();
        $view->loadTemplate('frontend/checkout/confirm.tpl');
        $getCountryList =  Shopware()->Modules()->Admin()->sGetCountryList();
        $view->assign('country_list', $getCountryList);
        $view->assign('sUserLoggedIn', 0);
        $view->assign('config', Shopware()->Config());

        if ($this->assertPluginsPresent(array('SwagPaymentPaypalPlus'))) {
            if ((bool)Shopware()->Config()->get('paypalExpressButtonLayer')) {
                $view->assign('PaypalShowButton', true);
            }
        }

        // PayPalUnified ExpressButton im Checkout anzeigen, wenn Nutzer nicht eingeloggt ist
        if ($this->assertPluginsPresent(array('SwagPaymentPayPalUnified'))) {
            try {
                $sTargetOld = $this->Request()->getParam('sTarget');
                $sTargetActionOld = $this->Request()->getParam('sTargetAction');
                $this->Request()->setParam('sTarget', 'checkout');
                $this->Request()->setParam('sTargetAction', 'confirm');
                $expressCheckout = Shopware()->Container()->get('paypal_unified.subscriber.express_checkout');
                $args = new Enlight_Controller_ActionEventArgs(
                  [
                      'subject' => $this,
                      'request' => $this->Request()
                  ]
                );

                $expressCheckout->addExpressCheckoutButtonLogin($args);
                if ($view->getAssign('paypalUnifiedEcLoginActive')) {
                    $view->PaypalUnifiedShowButton = 1;
                }

                $this->Request()->setParam('sTarget', $sTargetOld);
                $this->Request()->setParam('sTargetAction', $sTargetActionOld);
            } catch (\Exception $e) {
                echo $e->getMessage();
            }
            //$view->assign('paypalUnifiedLanguageIso', $this->getExpressCheckoutButtonLanguage());
        }

        // Wenn mindestens SW Version 5.5.9 dann Variable setzen
        if ($this->assertMinimumVersion("5.5.9")) {
            $view->assign('isMinSw559', 1);
        }

        // Wenn mindestens SW Version 5.4.3 dann Variable setzen, die die erweiterten Datenschutzfunktionalitäten aktiviert.
        if ($this->assertMinimumVersion("5.4.3")) {
            $view->assign('isGreaterSw542', 1);
        }
        // Wenn mindestens SW Version 5.5.8 dann Variable setzen, die für die neue Gastkonto-Checkbox benötigt wird
        if ($this->assertMinimumVersion("5.5.8")) {
            $view->assign('isGreaterSw558', 1);
        }

        // Wenn mindestens SW Version 5.7.17 dann Variable setzen
        if ($this->assertMinimumVersion("5.7.17")) {
            $view->assign('isMinSw5717', 1);
        }

        // Voucher Fehler aus Session ablegen, da durch header-Redirect die Meludng verloren gegangen sein könnte falls man im
        // Kompabilitätsmodus arbeitet, nicht eingeloggt ist und beim Gutschein Formular ein Target auf "confirm" gesetzt hat.
        if (array_key_exists('sVoucherError', $_SESSION) && $_SESSION['sVoucherError']) {
            $view->sVoucherError = $_SESSION['sVoucherError'];
            unset($_SESSION['sVoucherError']);
        }

        if (!Shopware()->Config()->get('premsNoConfirmFooterBeforeLogin')) {
            $basket = $this->getBasket();
            $view->sBasket = $basket;
        }

        if ($this->basket->sCountBasket() < 1) {
            return $this->forward('cart', 'checkout');
        }

        $view->sShippingcosts = $view->sBasket['sShippingcosts'];
        $view->sShippingcostsDifference = $view->sBasket['sShippingcostsDifference'];
        $view->sAmount = $view->sBasket['sAmount'];
        $view->sAmountWithTax = $view->sBasket['sAmountWithTax'];
        $view->sAmountTax = $view->sBasket['sAmountTax'];
        $view->sAmountNet = $view->sBasket['AmountNetNumeric'];

        // Formularwerte für Registrierung bereits gesetzt? Dann an Template übergeben und Session Wert leeren
        if (Shopware()->Session()->register) {
            $view->registerData = Shopware()->Session()->register;
            Shopware()->Session()->register = false;
        }

        // Login Fehlermeldung aus Session auslesen
        if (Shopware()->Session()->loginDataError) {
            $view->loginDataError = Shopware()->Session()->loginDataError;
            Shopware()->Session()->loginDataError = false;
        }
        // Email aus Session auslesen
        if (Shopware()->Session()->email) {
            $view->sFormDataLogin = array('email' => Shopware()->Session()->email);
            Shopware()->Session()->email = '';
        }

        // Mögliche Fehlermeldung aus Session auslesen
        if (Shopware()->Session()->sErrorFlag || Shopware()->Session()->sErrorMessages) {
            $view->sErrorFlag = Shopware()->Session()->sErrorFlag;
            $view->sErrorMessages = Shopware()->Session()->sErrorMessages;
            Shopware()->Session()->sErrorFlag = false;
            Shopware()->Session()->sErrorMessages = false;
        }

        // Optin Hash Invalid
        if (Shopware()->Session()->optinhashinvalid) {
            $view->optinhashinvalid = Shopware()->Session()->optinhashinvalid;
            Shopware()->Session()->optinhashinvalid = false;
        }
        // Optin Success
        if (Shopware()->Session()->optinsuccess) {
            $view->optinsuccess = Shopware()->Session()->optinsuccess;
            Shopware()->Session()->optinsuccess = false;
        }

        $view->assign('isAccountless', Shopware()->Session()->get('isAccountless'));

        // Mögliche Fehlermeldung Registrierung ab Shopware 5.2 aus Session auslesen
        if (Shopware()->Session()->errors) {
            //$view->register = Shopware()->Session()->register;
            $view->errors = Shopware()->Session()->errors;
            Shopware()->Session()->errors = false;
        }

        // DHL Wunschpaket Plugin installiert?
        if ($this->assertPluginsPresent(array('DHLPaWunschpaket'))) {
            /** @var \DHLPaWunschpaket\Adapter\WunschpaketSDKAdapter $adapter */
            $adapter = $this->get('mopt_wunschpaket.adapter');
            if ((bool)$adapter->getPluginConfig(\DHLPaWunschpaket\Bootstrap\PluginConfigInterface::ENABLED_FIELD)){
                $view->assign('dhlWunschpaketPlugin', 1);
            }
        }
        // DHL Wunschpaket Plugin (neue Version dhlxxWunschzustellung) installiert?
        if ($this->assertPluginsPresent(array('dhlxxWunschzustellung'))) {
            /** @var \dhlxxWunschzustellung\Adapter\WunschpaketSDKAdapter $adapter */
            $adapter = $this->get('mopt_wunschpaket.adapter');
            if ((bool)$adapter->getPluginConfig(\dhlxxWunschzustellung\Bootstrap\PluginConfigInterface::ENABLED_FIELD)){
                $view->assign('dhlWunschpaketPlugin', 1);
            }
        }

        // Endereco Adressprüfung Plugin installiert (Download oder Store Version)?
        if ($this->assertPluginsPresent(['EnderecoAMS'])) {
            $config = $this->container->get('shopware.plugin.cached_config_reader')->getByPluginName('EnderecoAMS', Shopware()->Shop());
            $view->assign('enderecoAMSClient', $config['isPluginActive']);
            $view->assign('enderecoAMSClientSplitStreet', $config['splitStreet']);
            $view->assign('enderecoAMSDownloadVersion', 0);
        } else if ($this->assertPluginsPresent(['EnderecoShopware5Client'])) {
            $config = $this->container->get('shopware.plugin.cached_config_reader')->getByPluginName('EnderecoShopware5Client', Shopware()->Shop());
            $view->assign('enderecoAMSClient', $config['isPluginActive']);
            $view->assign('enderecoAMSClientSplitStreet', $config['splitStreet']);
            $view->assign('enderecoAMSDownloadVersion', 1);
        }

        if ($this->assertPluginsPresent(array('SwagCustomProducts'))) {
            $view->assign('CustomProductsActive', 1);
        }

        // Prämien Produktauswahl anzeigen/verstecken
        $view->assign('premsShowPremiumProductArea', Shopware()->Config()->get('premsShowPremiumProductArea'));
        // Seperate Lieferadresse hinzufügen aktivieren/deaktivieren
        $view->assign('premsAdditionalShippingDisable', Shopware()->Config()->get('premsAdditionalShippingDisable'));

        $view->assign('useStepsOnlyVertical', Shopware()->Config()->get('useStepsOnlyVertical'));
        $view->assign('premsAgbOnTop', Shopware()->Config()->get('premsAgbOnTop'));
        $view->assign('premsShowNoConfirmFooterBeforeLogin', Shopware()->Config()->get('premsNoConfirmFooterBeforeLogin'));
        $view->assign('premsShowArticleProperties', Shopware()->Config()->get('premsShowArticleProperties'));
        $view->assign('premsShowOrderText', Shopware()->Config()->get('premsShowOrderText'));
        $view->assign('premsOpcPaymentsRedirectToPaymentPage', Shopware()->Config()->get('premsOpcPaymentsRedirectToPaymentPage'));
        $view->assign('premsOpcCompabilityMode', Shopware()->Config()->get('premsOpcCompabilityMode'));
        $view->assign('registerBeforeLogin', Shopware()->Config()->get('registerBeforeLogin'));
        $view->assign('loginInitiallyOpen', Shopware()->Config()->get('loginInitiallyOpen'));
        $view->assign('premsRegisterInitiallyOpen', Shopware()->Config()->get('premsRegisterInitiallyOpen'));
        $view->assign('zipBeforeStreet', Shopware()->Config()->get('zipBeforeStreet'));
        $view->assign('premsRegisterOnlyAsCompany', Shopware()->Config()->get('premsRegisterOnlyAsCompany'));
        $view->assign('premsOpcShowAfterLoginRegistrationAvailableText', Shopware()->Config()->get('premsOpcShowAfterLoginRegistrationAvailableText'));
        $view->assign('premsOpcDefaultRegisterAsGuest', Shopware()->Config()->get('premsOpcDefaultRegisterAsGuest'));
        $view->assign('premsOnePageCheckoutEmailPassPosition', Shopware()->Config()->get('premsOnePageCheckoutEmailPassPosition'));
        $view->assign('premsOnePageCheckoutShowCountryDisabledDefault', Shopware()->Config()->get('premsOnePageCheckoutShowCountryDisabledDefault'));
        $view->assign('premsOnePageCheckoutVoucherOnTop', Shopware()->Config()->get('premsOnePageCheckoutVoucherOnTop'));
        $view->assign('premsOnePageCheckoutNoCommentArea', Shopware()->Config()->get('premsOnePageCheckoutNoCommentArea'));
        $view->assign('useOnePageCheckout', Shopware()->Config()->get('useOnePageCheckout'));
        $view->assign('premsOnePageCheckoutReloadBuyboy', Shopware()->Config()->get('premsOnePageCheckoutReloadBuyboy'));

        // Legt die maximal erlaubte Zeichenlänge für das Strasse Feld bei der Registrierung fest
        $maxLengthStreet = 30;
        if (Shopware()->Config()->get('premsOnePageCheckoutStreetMaxLength')) {
            $maxLengthStreet = Shopware()->Config()->get('premsOnePageCheckoutStreetMaxLength');
        }
        $view->assign('premsOnePageCheckoutStreetMaxLength', $maxLengthStreet);

        $view->assign('sEsd', $this->isEsd($view));
    }

    protected function isEsd($view)
    {
        $basketDetails = $view->sBasket['content'];
        foreach ($basketDetails as $detail) {
            if ($detail['esd'] == 1) {
                return true;
            }
        }
        return false;
    }

    protected function assertPluginsPresent(array $plugins) {
        foreach ($plugins as $plugin) {
            $sql = 'SELECT 1 FROM s_core_plugins WHERE name = ? AND active = 1';
            $test = Shopware()->Container()->get('dbal_connection')->fetchColumn($sql, [$plugin]);
            if (!$test) {
                return false;
            }
        }

        return true;
    }

    /**
     * Action to simultaneously save shipping and payment details
     */
    public function saveShippingPaymentAction()
    {
        if (!$this->Request()->isPost()) {
            return $this->forward('shippingPayment');
        }

        // Load data from request
        $dispatch = $this->Request()->getPost('sDispatch');
        $payment = $this->Request()->getPost('payment');

        // If request is ajax, we skip the validation, because the user is still editing
        if ($this->Request()->getParam('isXHR')) {
            // Save payment and shipping method data.
            $this->admin->sUpdatePayment($payment);
            $this->setDispatch($dispatch, $payment);

            return $this->forward('index');
        }
    }

    public function loginAction() {
        if ($this->Request()->isPost()) {
            $checkUser = $this->admin->sLogin();
            if (!empty($checkUser['sErrorMessages'])) {
                $this->View()->sFormDataLogin = $this->Request()->getPost();
                $this->View()->sErrorFlag = $checkUser['sErrorFlag'];
                Shopware()->Session()->loginDataError = $checkUser['sErrorMessages'];
                Shopware()->Session()->email = $this->View()->sFormDataLogin['email'];
                $this->View()->loginDataError = $checkUser['sErrorMessages'];
            } else {
                Shopware()->Modules()->Basket()->sRefreshBasket();
            }
        }

        if (Shopware()->Config()->get('always_select_payment')) {
            return $this->redirect(array(
                'action' => 'shippingPayment',
                'controller' => 'checkout',
            ));
        } else {
            return $this->redirect(array(
                'action' => 'confirm',
                'controller' => 'checkout',
            ));
        }
    }
}
