<?php

/*
 * Bootstrap class of NwdItRechtConnector Plugin
 *
 * @copyright (c) 2019, Nordisch Webdesign
 * @author Philipp Jauert <info@nordisch-webdesign.de>
 */

// To keep downwards compatibility we have to include the interface here, cause
// in lower versions fom 5.2.0 there is no CSFRWhitelistAware interface.
require_once __DIR__ . '/Components/CSRFWhitelistAware.php';

/**
 * Class Shopware_Plugins_Backend_nwdxxItRechtConnector_Bootstrap
 *
 * @author Philipp Jauert <info@nordisch-webdesign.de>
 */
class Shopware_Plugins_Backend_nwdxxItRechtConnector_Bootstrap extends Shopware_Components_Plugin_Bootstrap
{
    /**
     * Static plugin information array. Get filled after first call
     * of getPluginInfo() method.
     *
     * @var array
     */
    protected static $pluginInfo;

    /**
     * Capabilities
     * What kind of capabilities are allowed.
     *
     * @return array
     * @author Philipp Jauert <info@nordisch-webdesign.de>
     */
    public function getCapabilities()
    {
        return array(
            'install' => true,
            'update' => true,
            'enable' => true,
            'secureUninstall' => true
        );
    }

    /**
     * Get plugin version.
     *
     * @return string
     * @throws Throwable
     * @author Philipp Jauert <info@nordisch-webdesign.de>
     */
    public function getVersion()
    {
        $config = static::getPluginInfo();
        return $config['currentVersion'];
    }

    /**
     * Get plugin info. It will load contents from the plugin.json file.
     *
     * @return array
     * @throws Throwable
     * @author Philipp Jauert <info@nordisch-webdesign.de>
     */
    public static function getPluginInfo()
    {
        if(static::$pluginInfo == null){
            static::$pluginInfo = json_decode(file_get_contents(__DIR__ . DIRECTORY_SEPARATOR .'plugin.json'), true);
        }
        if (static::$pluginInfo) {
            return static::$pluginInfo;
        } else {
            throw new Exception('The plugin has an invalid version file.');
        }
    }

    /**
     * Plugin form updates.
     *
     * @param string $version Old version
     *
     * @return bool
     * @throws Exception
     * @author Philipp Jauert <info@nordisch-webdesign.de>
     */
    public function update($version)
    {
        if (version_compare($version, '2.3.2', '<')) {
            // Remove the old token from the description.
            $this->resetDescription();
        }

        return true;
    }

    private function resetDescription()
    {
        $sql = 'UPDATE s_core_config_forms SET description = :desc WHERE name = "nwdxxItRechtConnector"';
        $description = $this->getInfo()['description'];
        Shopware()->Models()->getConnection()->executeQuery($sql, array(':desc' => $description));
    }

    /**
     * Installationsroutine.
     * Wir registrieren unser Event und das Konfigurationsformular
     *
     * @return bool
     * @author Philipp Jauert <info@nordisch-webdesign.de>
     */
    public function install()
    {
        $this->registerEvents();
        $this->installForm();
        return true;
    }

    /**
     * Gibt alle notwendigen Plugin Informationen im Plugin-Manager
     * zurueck.
     *
     * @return array
     * @throws Throwable
     * @author Philipp Jauert <info@nordisch-webdesign.de>
     */
    public function getInfo()
    {
        $config = static::getPluginInfo();
        return array(
            'label' => $config['label']['de'],
            'version' => $config['currentVersion'],
            'author' => $config['author'],
            'description' => $config['description'],
            'copyright' => $config['copyright'],
            'support' => $config['support'],
            'link' => $config['link']
        );
    }

    /**
     * Since 2.0.1 this is just a wrapper method. We access here the configuration
     * of the plugin to load the config from the database.
     * The token itself will be created once while installation. It can be renewed
     * by reinstall the plugin.
     *
     * @return string
     * @throws Exception
     * @author Philipp Jauert <info@nordisch-webdesign.de>
     */
    public function getToken()
    {
        return $this->Config()->get('token');
    }

    /**
     * Erstellt das Token und gibt dieses zurueck.
     * Der Algorithmus ist wie in den aelteren Plugin Versionen geblieben,
     * damit es einen reibungslosen uebergang gibt.
     *
     * @since 1.1.1
     *
     * @return string|bool
     * @author Philipp Jauert <info@nordisch-webdesign.de>
     */
    protected static function generateToken()
    {
        // Migration step for older versions.
        $tokenFile = dirname(__FILE__).DIRECTORY_SEPARATOR.'token';
        if (file_exists($tokenFile)) {
            $token = static::readToken($tokenFile);
            if ($token !== false) {
                unlink($tokenFile);
                return $token;
            }
            return false;
        }

        return md5(md5($_SERVER['HTTP_HOST']).'nwdxx'.microtime(true));
    }

    /**
     * Auslesen des "Tokenfiles" aus dem Dateisystem und rueckgabe des
     * ausgelesenen Token.
     *
     * @since 1.1.1
     *
     * @param string $sTokenFile Dateiname inklusive Pfad
     *
     * @return string|boolean Bei erfolg wird das Token zurueck gegeben sonst false
     * @deprecated This is left for migration from < 2.0.1 versions.
     * @author Philipp Jauert <info@nordisch-webdesign.de>
     */
    protected static function readToken($sTokenFile)
    {
        if (false === $sToken = file_get_contents($sTokenFile)) {
            static::getLogger()->error('Token File konnte nicht gelesen werden.');
            return false;
        }
        return $sToken;
    }

    /**
     * Hier registrieren wir unseren eigenen Controller auf das entsprechende
     * Event. Bei diesem Event wird die Methode aufgerufen, welche
     * unseren Controller zurueck gibt.
     *
     * @author Philipp Jauert <info@nordisch-webdesign.de>
     */
    private function registerEvents()
    {
        /** Connector */
        $this->subscribeEvent(
            'Enlight_Controller_Dispatcher_ControllerPath_Frontend_EscLegalConnector',
            'onGetController'
        );
    }

    /**
     * Die Methode erstellt das Plugin Formular fuer die Konfiguration
     * des Plug-Ins. In der Version 1.1.0 sind weitere Felder hinzugekommen.
     *
     * @author Philipp Jauert <info@nordisch-webdesign.de>
     */
    private function installForm()
    {
        /** Form instance */
        $form = $this->Form();

        $description = "Tragen Sie hier die Shopseiten ID ein (z.B. 12345), in welche dieser Rechtstext übertragen werden 
                        soll. Die möglichen IDs finden Sie im Hauptmenü unter Inhalte -> Shopseiten, 
                        dort links in der Klammer hinter dem jeweiligen Seitentitel.
                        Möchten Sie diesen Rechtstext nicht über die Datenschnittstelle übertragen, 
                        können Sie das Feld einfach frei lassen.";

        /** Form element, agb */
        $form->setElement('number', 'agb',
            array(
                'label' => 'AGB ID',
                'value' => NULL,
                'scope' => Shopware\Models\Config\Element::SCOPE_SHOP,
                'description' => $description,
                'required' => false
            )
        );
        /** Form element, Revocation */
        $form->setElement('number', 'widerruf',
            array(
                'label' => 'Widerrufsrecht ID',
                'value' => NULL,
                'scope' => Shopware\Models\Config\Element::SCOPE_SHOP,
                'description' => $description,
                'required' => false
            )
        );
        /** Form element, Privacy */
        $form->setElement('number', 'datenschutz',
            array(
                'label' => 'Datenschutz ID',
                'value' => NULL,
                'scope' => Shopware\Models\Config\Element::SCOPE_SHOP,
                'description' => $description,
                'required' => false
            )
        );
        /** Add Form element, Imprint */
        $form->setElement('number', 'impressum',
            array(
                'label' => 'Impressum ID',
                'value' => NULL,
                'scope' => Shopware\Models\Config\Element::SCOPE_SHOP,
                'description' => $description,
                'required' => false
            )
        );

        $form->setElement('select', 'importAgb', array(
                'label' => 'AGB als PDF importieren',
                'store' => array(
                    array(1, 'Nein, nicht importieren'),
                    array(2, 'Ja, in den Medienmanager importieren'),
                    array(3, 'Ja, in den Medienmanager importieren und konfigurierten E-Mails als Anhang beifügen')

                ),
                'scope' => Shopware\Models\Config\Element::SCOPE_SHOP
            )
        );
        $form->setElement('text', 'emailTemplateNamenAgb',
            array(
                'label' => 'eMail-Vorlagenname f&uuml;r AGB PDF-Anhang',
                'value' => 'sORDER',
                'scope' => Shopware\Models\Config\Element::SCOPE_SHOP,
                'description' => 'Die Namen finden Sie unter Einstellungen -> '
                    . 'eMail-Vorlagen. Soll das Dokument an mehrere Vorlagen angehängt werden, '
                    . 'geben Sie die Namen bitte kommasepariert ein.'
            )
        );
        $form->setElement('select', 'importWiderruf', array(
                'label' => 'Widerrufsbelehrung als PDF importieren',
                'store' => array(
                    array(1, 'Nein, nicht importieren'),
                    array(2, 'Ja, in den Medienmanager importieren'),
                    array(3, 'Ja, in den Medienmanager importieren und konfigurierten E-Mails als Anhang beifügen')
                ),
                'scope' => Shopware\Models\Config\Element::SCOPE_SHOP
            )
        );
        $form->setElement('text', 'emailTemplateNamenWiderruf',
            array(
                'label' => 'eMail-Vorlagenname f&uuml;r Widerruf PDF-Anhang',
                'value' => 'sORDER',
                'scope' => Shopware\Models\Config\Element::SCOPE_SHOP,
                'description' => 'Die Namen finden Sie unter Einstellungen -> '
                    . 'eMail-Vorlagen. Soll das Dokument an mehrere Vorlagen angehängt werden, '
                    . 'geben Sie die Namen bitte kommasepariert ein.'
            )
        );
        $form->setElement('text', 'token', array(
            'label' => 'Token zur Eingabe im Mandantenportal',
            'value' => static::generateToken(),
            'scope' => '',
            'description' => 'Das Token wird zur Verifizierung im Portal der IT-Recht Kanzlei benötigt.',
            'required' => false,
            'readOnly' => true
        ));
    }

    /**
     * Nach der Refaktorierung des Plugins
     * muessen weder Tabellen noch Cache geleert werden. Wir
     * geben nur rueckmeldung, dass die Deinstallation funktioniert hat.
     *
     * @return bool
     * @author Philipp Jauert <info@nordisch-webdesign.de>
     */
    public function uninstall()
    {
        return true;
    }

    /**
     * OnGetController
     * Die Methode waehlt unseren Controller beim aufruf der URL
     * /esclegalconnector an, welche von der Schnittstelle der
     * IT-Recht Kanzlei aufgerufen wird.
     *
     * @since 1.1.0 Register Plugin Namespace
     *
     * @param Enlight_Event_EventArgs $arguments
     *
     * @return string
     * @author Philipp Jauert <info@nordisch-webdesign.de>
     */
    public function onGetController(Enlight_Event_EventArgs $arguments)
    {
        $this->registerPluginNamespace();

        /** Path to plugin controller */
        return $this->Path() . 'Controllers/Frontend/EscLegalConnector.php';
    }

    /**
     * Register custom namespace.
     *
     * @since 1.1.0
     *
     * @author Philipp Jauert <info@nordisch-webdesign.de>
     */
    public function registerPluginNamespace()
    {
        $this->Application()->Loader()->registerNamespace(
            'Shopware\\NwdxxItRecht',
            $this->Path(),
            '\\'
        );
    }

    /**
     * Using the container to get the logger instance.
     * Shopware()->PluginLogger() is deprecated.
     *
     * @return \Shopware\Components\Logger
     * @author Philipp Jauert <info@nordisch-webdesign.de>
     */
    protected static function getLogger()
    {
        return Shopware()->Container()->get('pluginlogger');
    }
}
