<?php

/*
 * Shopware frontend controller which will be requested by the
 * law agency system to update site content of custom configured pages.
 * It also provides pdf download and storing into the media manager
 * and hanging in these creaded media objects as attachment relation
 * in configured email templates.
 *
 * @copyright (c) 2019, Nordisch Webdesign
 * @author Philipp Jauert <info@nordisch-webdesign.de>
 */

use Shopware\Components\CSRFWhitelistAware;
use Shopware\Models\Mail\Attachment;
use Shopware\Models\Media\Media;
use Shopware\Models\Media\Repository;
use Shopware\Models\Shop\Shop;
use Shopware\Models\Site\Site;
use Symfony\Component\HttpFoundation\File\File;

require_once(__DIR__.'/../../sdk/require_all.php');

/**
 * Class Shopware_Controllers_Frontend_EscLegalConnector
 *
 * @author Philipp Jauert <info@nordisch-webdesign.de>
 */
class Shopware_Controllers_Frontend_EscLegalConnector extends Enlight_Controller_Action implements CSRFWhitelistAware
{

    const IMPORT_CONFIG_IMPORT = '2';
    const IMPORT_CONFIG_IMPORT_AND_ATTACHMENT = '3';

    const MEDIA_MANAGER_FOLDER_ID = -10;

    private $configs = [];

    /**
     * Disable CSFR Check.
     *
     * @return array
     * @author Philipp Jauert <info@nordisch-webdesign.de>
     */
    public function getWhitelistedCSRFActions()
    {
        return array(
            'index'
        );
    }

    /**
     * Disable the renderer to avoid the try of fetch it which
     * would results in an exception.
     *
     * @throws Exception
     * @author Philipp Jauert <info@nordisch-webdesign.de>
     */
    public function preDispatch()
    {
        // Render without template.
        $this->Front()->Plugins()->ViewRenderer()->setNoRender();
        parent::preDispatch();
    }

    /**
     * Get the current shop version
     *
     * @return string
     * @author Philipp Jauert <info@nordisch-webdesign.de>
     */
    protected function getShopVersion()
    {
        $version = Shopware()->Config()->get('version');
        if (null === $version) {
            $version = 'unknown';
        }
        return $version;
    }

    /**
     * Get the current plugin version
     *
     * @return string
     * @author Philipp Jauert <info@nordisch-webdesign.de>
     */
    protected function getPluginVersion()
    {
        $version = Shopware()->Plugins()->Backend()->nwdxxItRechtConnector()->getVersion();
        if (null === $version) {
            $version = 'unknown';
        }
        return $version;
    }

    /**
     * Using the container to get the logger instance.
     * Shopware()->PluginLogger() is deprecated.
     *
     * @return \Shopware\Components\Logger
     * @throws \ITRechtKanzlei\LTIError
     * @author Philipp Jauert <info@nordisch-webdesign.de>
     */
    protected function getLogger()
    {
        try {
            return $this->get('pluginlogger');
        } catch (\Throwable $e) {
            throw new \ITRechtKanzlei\LTIError(
                'Unable to load the plugin logger instance.',
                \ITRechtKanzlei\LTIError::UNKNOWN_ERROR
            );
        }
    }

    /**
     * Load the token.
     *
     * @return string
     * @throws OutputException
     * @author Philipp Jauert <info@nordisch-webdesign.de>
     */
    public function getToken()
    {
        $token = Shopware()->Plugins()->Backend()->nwdxxItRechtConnector()->getToken();
        if (false === $token) {
            $this->getLogger()->error('The client token is empty!');
            return '';
        }

        return $token;
    }

    protected function getShopById(string $id): Shop
    {
        /** @var \Shopware\Models\Shop\Repository $shopRepository */
        $shopRepository = Shopware()->Models()->getRepository('Shopware\Models\Shop\Shop');

        /** @var Shopware\Models\Shop\Shop $shop */
        $shop = $shopRepository->findOneBy([
            'active' => 1,
            'id' => $id
        ]);

        if ($shop === null) {
            throw new \ITRechtKanzlei\LTIError(
                sprintf('Invalid user account id "%d".', $id),
                \ITRechtKanzlei\LTIError::INVALID_USER_ACCOUNT_ID
            );
        }

        return $shop;
    }

    private function getMainShopRegardingNull(Shop $shop) {
        return ($shop->getMain() === null) ? $shop : $shop->getMain();
    }

    private function getConfigForShop(Shopware\Models\Shop\Shop $shop)
    {
        if (isset($this->configs[$shop->getId()])) {
            return $this->configs[$shop->getId()];
        }
        /** @var \Enlight_Plugin_PluginManager $pluginManager */
        $pluginManager = Shopware()->Container()->get('plugins');

        /** @var \Shopware_Components_Plugin_Namespace $namespace */
        $namespace = $pluginManager->get('Backend');

        /** @var \Enlight_Config $config */
        $this->configs[$shop->getId()] = $namespace->getConfig('nwdxxItRechtConnector', $shop);

        $this->getLogger()->info(sprintf('Loaded configuration for shop %s', $shop->getName()), [
            'config' => $this->configs[$shop->getId()]->toArray()
        ]);
        return $this->configs[$shop->getId()];
    }

    /**
     * Get plugin config param loaded for requested shop.
     *
     * @param Shop $shop
     * @param string $name
     *
     * @return mixed
     * @throws OutputException
     * @author Philipp Jauert <info@nordisch-webdesign.de>
     */
    protected function getConfigParam(Shopware\Models\Shop\Shop $shop, $name)
    {
        return $this->getConfigForShop($shop)->get($name);
    }

    /**
     * @param SimpleXMLElement $xml
     * @return array
     * @throws OutputException
     */
    protected function getActiveShopsForLocale(Shop $requestedShop, string $stringLocale)
    {
        /** @var \Shopware\Models\Shop\Repository $shopRepository */
        $shopRepository = Shopware()->Models()->getRepository('Shopware\Models\Shop\Shop');
        $mainShop = $this->getMainShopRegardingNull($requestedShop);

        $collection = [];
        /** @var Shop $shop */
        foreach ($shopRepository->getActiveShops() as $shop) {
            /** @var \Shopware\Models\Shop\Locale $locale */
            $locale = $shop->getLocale();
            $mainShopForShop = $this->getMainShopRegardingNull($shop);
            if ($stringLocale === $locale->getLocale() && $mainShop->getId() === $mainShopForShop->getId()) {
                $collection[] = $shop;
            }
        }

        if (empty($collection)) {
            $this->getLogger()->error('Could not find shop for legal text translation.', [
                'requested-shop' => $requestedShop->getId(),
                'locale' => $stringLocale,
            ]);
        }

        return $collection;
    }

    private function createTargetUrl(Shop $shop, Site $site) {
        $mainShop = $this->getMainShopRegardingNull($shop);
        return sprintf('%s://%s%s/%s',
            'http',
            $mainShop->getHost(),
            $shop->getBaseUrl() ? $shop->getBaseUrl() : '',
            $site->getLink() ? $site->getLink() : 'custom/index/sCustom/' . $site->getId()
        );
    }

    protected function processPdf(Shop $requestedShop, \ITRechtKanzlei\LTIPushData $legalText) {
        if (!$legalText->hasPdf()) {
            return;
        }
        $importSetting  = (string)$this->getConfigParam(
            $requestedShop,
            'import' . ucfirst(strtolower((string)$legalText->getType()))
        );

        if (!in_array($importSetting, [self::IMPORT_CONFIG_IMPORT, self::IMPORT_CONFIG_IMPORT_AND_ATTACHMENT], true)) {
            return;
        }

        $legalFileIdentifier = strtolower(
            'it-recht-connector-%d'
            . '-' . $legalText->getType()
            . '-' . $legalText->getLanguageIso639_1()
            . '-' . $legalText->getCountry()
        );

        /** @var Repository $mediaRepository */
        $mediaRepository = Shopware()->Models()->getRepository('Shopware\Models\Media\Media');

        $attachmentRepository = Shopware()->Models()->getRepository('Shopware\Models\Mail\Attachment');

        $media = $mediaRepository->findOneBy(['description' => sprintf($legalFileIdentifier, $requestedShop->getId())]);
        // Fallback to get wrongly assinged pdfs
        if ((!$media instanceof Media) && (($mainShop = $requestedShop->getMain()) !== null)) {
            $media = $mediaRepository->findOneBy(['description' => sprintf($legalFileIdentifier, $mainShop->getId())]);
        }

        if ($media instanceof Media) {
            $tmpFileName = $media->getName();
        } else {
            // Write tmp file with pattern {type}-{shop_name}-{shop_id}-{language}-{country}
            $tmpFileName = $legalText->getLocalizedFileName()
                . '-' . strtr($requestedShop->getName(), [ 'ä' => 'ae', 'ö' => 'oe', 'ü' => 'ue', 'Ä' => 'Ae', 'Ö' => 'Oe', 'Ü' => 'Ue', 'ß' => 'ss' ])
                . '-' . $requestedShop->getId();
            $tmpFileName = (new Media())->setName($tmpFileName)->getName();
        }

        $tmpFilePath = dirname(dirname(__DIR__)) . '/tmp/' . $tmpFileName;

        #echo $tmpFilePath;exit();

        // Write the contents.
        $handle = fopen($tmpFilePath, "w");

        // If it is not possible to write the file,
        // we left an error log and return false.
        if ($handle === false) {
            throw (new \ITRechtKanzlei\LTIError(
                'Das PDF kann nicht zwischengespeichert werden. Bitte korrigieren Sie die Dateiberechtigungen.',
                \ITRechtKanzlei\LTIError::SAVE_PDF_ERROR
            ))->addContext([
                'path' => $tmpFilePath
            ]);
        }

        fwrite($handle, $legalText->getPdf());
        fclose($handle);

        $file = new File($tmpFilePath);

        if ($media instanceof Media) {
            /** @var Attachment $attachments */
            $affectedAttachments = $attachmentRepository->findBy([
                'media' => $media
            ]);
            // If we got an affected attachments (now corrupted) but attachment
            // handling is inactive, remove the attachments.
            if (!empty($affectedAttachments)
                && ($importSetting !== self::IMPORT_CONFIG_IMPORT_AND_ATTACHMENT)
            ) {
                foreach ($affectedAttachments as $affectedAttachment) {
                    Shopware()->Models()->remove($affectedAttachment);
                    Shopware()->Models()->flush($affectedAttachment);
                }
            }
            Shopware()->Models()->remove($media);
            Shopware()->Models()->flush($media);
        }

        $media = new Media();
        $albumRepository = Shopware()->Models()->getRepository('Shopware\Models\Media\Album');
        $album = $albumRepository->find(self::MEDIA_MANAGER_FOLDER_ID);
        $media->setAlbum($album);
        $media->setDescription(sprintf($legalFileIdentifier, $requestedShop->getId()));
        $media->setUserId(0);
        $media->setCreated(new \DateTime());
        $media->setExtension('pdf');
        $media->setFile($file);

        Shopware()->Models()->persist($media);
        Shopware()->Models()->flush($media);
        unlink($tmpFilePath);

        if ($importSetting !== self::IMPORT_CONFIG_IMPORT_AND_ATTACHMENT) {
            return;
        }

        $templateNames  = explode(',', (string)$this->getConfigParam(
            $requestedShop,
            'emailTemplateNamen' . ucfirst(strtolower((string)$legalText->getType()))
        ));

        if (empty($templateNames)) {
            throw new \ITRechtKanzlei\LTIError(
                'Bitte hinterlegen Sie die E-Mail Template Namen, zu denen die Anhänge hinzugefügt werden sollen.',
                \ITRechtKanzlei\LTIError::CONFIGURATION_INCOMPLETE
            );
        }

        foreach ($templateNames as $templateName) {
            $templateName = trim($templateName);
            if (empty($templateName)) {
                continue;
            }

            /** @var Shopware\Models\Mail\Repository $mailRepository */
            $mailRepository = Shopware()->Models()->getRepository('Shopware\Models\Mail\Mail');

            /** @var \Shopware\Models\Mail\Mail $mailEntity */
            $mailEntity = $mailRepository->findOneBy([
                'name' => $templateName
            ]);
            if ($mailEntity === null) {
                throw (new \ITRechtKanzlei\LTIError(
                    sprintf(
                        'Das E-Mail Template mit dem Namen %s konnte nicht geladen werden. '
                            .'Bitte stellen Sie sicher, dass Sie den Namen nicht falsch '
                            .'eingegeben haben und das Template existiert.',
                        $templateName
                    ),
                    \ITRechtKanzlei\LTIError::CONFIGURATION_INCOMPLETE
                ))->addContext([
                    'email-template-name' => $templateName
                ]);
            }

            if (isset($affectedAttachments) && !empty($affectedAttachments)) {
                foreach ($affectedAttachments as $affectedAttachment) {
                    $newAttachment = new Attachment($mailEntity, $media, $affectedAttachment->getShop());
                    Shopware()->Models()->persist($newAttachment);
                    Shopware()->Models()->flush($newAttachment);
                    Shopware()->Models()->remove($affectedAttachment);
                    Shopware()->Models()->flush($affectedAttachment);
                }
            } else {
                // No attachment, create a new one.
                if ($requestedShop instanceof Shop) {
                    $attachment = new Attachment($mailEntity, $media, $requestedShop);
                    Shopware()->Models()->persist($attachment);
                    Shopware()->Models()->flush($attachment);
                }
            }
        }
    }

    /**
     * @return string
     */
    public function processLegalText(\ITRechtKanzlei\LTIPushData $legalText)
    {
        $requestedShop = $this->getShopById($legalText->getMultiShopId());
        $contentId = trim((string)$this->getConfigParam($requestedShop, $legalText->getType()));

        if (empty($contentId) || ($contentId === 0)) {
            throw new \ITRechtKanzlei\LTIError(
                sprintf(
                    'For store "%s", the page assignment for the legal text "%s" is missing.',
                    $requestedShop->getName(),
                    $legalText->getType()
                ),
                \ITRechtKanzlei\LTIError::CONFIGURATION_INCOMPLETE
            );
        }

        /** @var Shopware\Models\Site\Repository $siteRepository */
        $siteRepository = Shopware()->Models()->getRepository('Shopware\Models\Site\Site');
        /** @var \Shopware\Models\Site\Site $site */
        $site = $siteRepository->find($contentId);
        if ($site === null) {
            throw new \ITRechtKanzlei\LTIError(
                sprintf('The CMS page with ID %d could not be loaded.', $contentId),
                \ITRechtKanzlei\LTIError::CONFIGURATION_DOCUMENT_NOT_FOUND
            );
        }

        $destinationShop = null;
        $targetUrl = '';

        if ($this->container->has('translation')) {
            /** @var Shopware_Components_Translation $translator */
            $translator = $this->get('translation');

            // Get shop by locale to translate its html field.
            $languageShops = $this->getActiveShopsForLocale($requestedShop, $legalText->getLocale());

            /** @var Shop $shopForLocale */
            foreach ($languageShops as $languageShop) {
                // Don't add a translation for the main-shops language.
                if (($languageShop->getId() === $requestedShop->getId())
                    && ($requestedShop->getLocale()->getLocale() === $legalText->getLocale())
                ) {
                    $translator->delete($languageShop->getId(), 'page', $contentId);
                    continue;
                }

                $data = $translator->read($languageShop->getId(), 'page', $contentId);
                $translator->write(
                    $languageShop->getId(),
                    'page',
                    $contentId,
                    array_merge($data, ['html' => $legalText->getTextHtml()]),
                    false
                );
                $destinationShop = $languageShop;
                $targetUrl = $this->createTargetUrl($languageShop, $site);

                $this->getLogger()->info('Written translation for shop.', [
                    'requested-shop' => $requestedShop->getId(),
                    'destination-shop' => $languageShop->getId(),
                ]);
            }
        }

        // If the text in a language store has already been updated,
        // do not update the text from the main store as well,
        // as this could overwrite the text of another language.
        if (($destinationShop === null)
            && ($requestedShop->getLocale()->getLocale() === $legalText->getLocale())
        ) {
            $site->setHtml($legalText->getTextHtml());
            Shopware()->Models()->persist($site);
            Shopware()->Models()->flush($site);
            $destinationShop = $requestedShop;
            $targetUrl = $this->createTargetUrl($requestedShop, $site);
        }

        if ($destinationShop === null) {
            throw new \ITRechtKanzlei\LTIError(
                'Für die Locale (Sprache- und Landkombination) des übertragenen Rechtstexts wurde kein passender (Sprach-)Shop gefunden.',
                \ITRechtKanzlei\LTIError::CONFIGURATION_LANGUAGE_NOT_SUPPORTED
            );
        }

        // Clear cache by fire invalidation event.
        $eventManager = Shopware()->Events();
        $eventName = 'Shopware_Plugins_HttpCache_InvalidateCacheId';

        // This is how the cache id is built in Shopware.
        // @see \ShopwarePlugins\HttpCache\CacheIdCollector::getStaticSiteCacheIds(Request $request)
        $eventParam = array('cacheId' => 's' . (int)$site->getId());
        $eventManager->notify($eventName, $eventParam);

        $this->processPdf($destinationShop, $legalText);

        return $targetUrl;
    }

    /**
     * Index Action / IT-Recht RESTful interface endpoint.
     */
    public function indexAction()
    {
        $handler = new class($this) extends \ITRechtKanzlei\LTIHandler {
            private $c;

            public function __construct(Shopware_Controllers_Frontend_EscLegalConnector $controller)
            {
                $this->c = $controller;
            }

            /**
             * Check whether the sent token is valid or not.
             */
            public function isTokenValid(string $token): bool
            {
                return $this->c->getToken() === $token;
            }

            public function handleActionGetAccountList(): \ITRechtKanzlei\LTIAccountListResult
            {
                $result = new \ITRechtKanzlei\LTIAccountListResult();
                $activeShops = Shopware()->Models()
                    ->getRepository('Shopware\Models\Shop\Shop')
                    ->getActiveShops();

                foreach ($activeShops as $shop) {
                    /** @var Shopware\Models\Shop\Shop $shop */
                    $shopType = ($shop->getMain() !== null)
                        ? 'Sprachshop'
                        : ($shop->getDefault() ? 'Hauptshop' : 'Subshop');
                    $result->addAccount(
                        $shop->getId(),
                        $shop->getName().' ('.$shopType.')',
                        [$shop->getLocale()->getLocale()]
                    );
                }
                return $result;
            }

            public function handleActionPush(
                \ITRechtKanzlei\LTIPushData $legalText
            ): \ITRechtKanzlei\LTIPushResult
            {
                return new \ITRechtKanzlei\LTIPushResult(
                    $this->c->processLegalText($legalText)
                );
            }
        };

        $lti = new \ITRechtKanzlei\LTI(
            $handler,
            $this->getShopVersion(),
            $this->getPluginVersion()
        );
        $lti->setErrorCallback(function (\Throwable $e) {
            if (!$e instanceof \ITRechtKanzlei\LTIError) {
                // Only log internal errors not generated by the LTI SDK or by processing the request.
                try {
                    $this->getLogger()->error($e->getMessage(), [$e]);
                } catch (\Throwable $t) {}
            }
        });
        $rXml = $this->Request()->getRawBody();
        if (substr($rXml, 0, 6) === '<?xml ') {
            $sXML = $rXml;
        } else {
            $rXml = urldecode($rXml);
            $sXML = substr($rXml, 4);
        }

        $response = $lti->handleRequest($sXML);
        return $this->Response()
            ->setHeader('Content-type', 'application/xml')
            ->setBody($response);
    }

}
