<?php
/**
 * Shopware 4
 * Copyright © shopware AG
 *
 * According to our dual licensing model, this program can be used either
 * under the terms of the GNU Affero General Public License, version 3,
 * or under a proprietary license.
 *
 * The texts of the GNU Affero General Public License with an additional
 * permission and of our proprietary license can be found at and
 * in the LICENSE file you have received along with this program.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * "Shopware" is a registered trademark of shopware AG.
 * The licensing of the program under the AGPLv3 does not imply a
 * trademark license. Therefore any rights, title and interest in
 * our trademarks remain entirely with us.
 */

use Shopware\Bundle\CookieBundle\CookieCollection;
use Shopware\Bundle\CookieBundle\Structs\CookieGroupStruct;
use Shopware\Bundle\CookieBundle\Structs\CookieStruct;
use Shopware\Components\Plugin;

/**
 * Shopware SwagETracker Plugin - Bootstrap
 *
 * @category  Shopware
 * @package   Shopware\Plugins\SwagETracker
 * @copyright Copyright © shopware AG (http://www.shopware.de)
 */
class Shopware_Plugins_Frontend_SwagETracker_Bootstrap extends Shopware_Components_Plugin_Bootstrap
{

    const OWN_SCHEMA_ID = 1;

    /**
     * Returns the meta information about the plugin
     * as an array.
     * Keep in mind that the plugin description located
     * in the info.txt.
     *
     * @public
     * @return array
     */
    public function getInfo()
    {
        return array(
            'version' => $this->getVersion(),
            'label' => $this->getLabel(),
            'link' => 'https://www.etracker.com/',
            'description' => file_get_contents($this->Path() . 'info_de.txt') . file_get_contents($this->Path() . 'info_en.txt')
        );
    }

    /**
     * Returns the label of the plugin.
     * @return string
     */
    public function getLabel()
    {
        return 'Shop-Optimierung mit etracker';
    }

    /**
     * Returns the newest plugin-version.
     * @return string
     */
    public function getVersion()
    {
        $info = json_decode(file_get_contents(__DIR__ . DIRECTORY_SEPARATOR .'plugin.json'), true);

        if ($info) {
            return $info['currentVersion'];
        } else {
            throw new Exception('The plugin has an invalid version file.');
        }
    }

    /**
     * Plugin install method
     * It subscribes all needed events and creates the menu-entry, the databases and the new model-attributes
     *
     * @return bool success
     */
    public function install()
    {
        $this->subscribeEvents();
        $this->createMenu();
        $this->createDatabase();
        $this->createAttribute();

        return true;
    }

    /**
     * Plugin update method
     *
     * @param string $oldversion
     * @return bool success
     */
    public function update($oldversion)
    {
        $this->createConfigTable();

        if (\version_compare($oldversion, '1.5.0', '<')) {
            $this->subscribeEvent( 'CookieCollector_Collect_Cookies', 'addStatisticCookie');
        }

        return true;
    }

    /**
     * Helper method to create the new model-attribute.
     * It will be used to extend the emotion-module by a new field for a landing-page ID given by etracker
     *
     * @throws Exception
     */
    private function createAttribute()
    {
        /** @var \Shopware\Bundle\AttributeBundle\Service\CrudService $crudService */
        $crudService = Shopware()->Container()->get('shopware_attribute.crud_service');
        $crudService->update('s_emotion_attributes', 'swag_e_tracker_landing_page_id', 'text');

        /** @var \Doctrine\Common\Cache\ClearableCache $metaDataCache */
        $metaDataCache = Shopware()->Models()->getConfiguration()->getMetadataCacheImpl();
        $metaDataCache->deleteAll();
        Shopware()->Models()->generateAttributeModels(['s_emotion_attributes']);
    }

    /**
     * Helper method to subscribe the events.
     * Those events listen to the global PostDispatch, two backend-controllers and to the backend index-controller
     * to extend the css-file.
     */
    private function subscribeEvents()
    {
        $this->subscribeEvent('Enlight_Controller_Action_PostDispatchSecure_Frontend', 'onPostDispatch');

        $this->subscribeEvent('Enlight_Controller_Dispatcher_ControllerPath_Frontend_SwagETracker', 'getFrontendController');

        $this->subscribeEvent('Enlight_Controller_Dispatcher_ControllerPath_Backend_SwagETracker', 'getBackendController');
        $this->subscribeEvent('Enlight_Controller_Action_PostDispatchSecure_Backend_Emotion', 'onGetControllerPathBackend');

        $this->subscribeEvent('Enlight_Controller_Action_PostDispatchSecure_Backend_Index', 'onPostDispatchIndex');

        // Subscribe the needed event for js merge and compression
        $this->subscribeEvent('Theme_Compiler_Collect_Plugin_Javascript', 'addJsFiles');

        $this->subscribeEvent( 'CookieCollector_Collect_Cookies', 'addStatisticCookie');
    }

    /**
     * Helper method to create all the needed databases and its values.
     */
    private function createDatabase()
    {
        $this->createConfigTable();
        $this->createAreaTable();
        $this->createDivisionTable();
        $this->createControllerTable();
        $this->createAreaCategoriesTable();
        $this->createAreaControllersTable();
        $this->createAreaShopsTable();
    }

    /**
     * Helper method to create the mapping-table "s_etracker_areas_shops".
     * It contains the mapping from the areas to the shops.
     */
    private function createAreaShopsTable()
    {
        $sql= 'CREATE TABLE IF NOT EXISTS `s_etracker_areas_shops` (
			 `id` int(11) NOT NULL AUTO_INCREMENT,
			 `area_id` int(11) NOT NULL,
			 `shop_id` int(11) NOT NULL,
			 PRIMARY KEY (`id`)
			) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci';
        Shopware()->Db()->query($sql, array());
    }

    /**
     * Helper method to create the mapping-table "s_etracker_areas_controllers".
     * It contains the mapping from the areas to the controllers, which are contained in the table "s_etracker_controllers".
     */
    private function createAreaControllersTable()
    {
        $sql= 'CREATE TABLE IF NOT EXISTS `s_etracker_areas_controllers` (
			 `id` int(11) NOT NULL AUTO_INCREMENT,
			 `area_id` int(11) NOT NULL,
			 `controller_id` int(11) NOT NULL,
			 PRIMARY KEY (`id`)
			) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci';
        Shopware()->Db()->query($sql, array());
    }

    /**
     * Helper method to create the mapping-table "s_etracker_areas_categories".
     * It contains the mapping from the areas to the categories.
     */
    private function createAreaCategoriesTable()
    {
        $sql= 'CREATE TABLE IF NOT EXISTS `s_etracker_areas_categories` (
			 `id` int(11) NOT NULL AUTO_INCREMENT,
			 `area_id` int(11) NOT NULL,
			 `category_id` int(11) NOT NULL,
			 PRIMARY KEY (`id`)
			) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci';
        Shopware()->Db()->query($sql, array());
    }

    /**
     * Helper method to create the table "s_etracker_controllers" and its values.
     */
    private function createControllerTable()
    {
        $sql= 'CREATE TABLE IF NOT EXISTS `s_etracker_controllers` (
			  `id` int(11) NOT NULL AUTO_INCREMENT,
			  `name` text COLLATE utf8_unicode_ci NOT NULL,
			  PRIMARY KEY (`id`)
			) ENGINE=InnoDB  DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci';


        Shopware()->Db()->query($sql, array());
        $this->addControllerData();
    }

    /**
     * This method add the needed controller data if no data in the Database..
     * This mehtod is implement because if the customer do reinstall the Plugin, the data is redundant
     *
     * @throws Exception
     */
    private function addControllerData()
    {
        $db = Shopware()->Container()->get('db');
        $sql = 'SELECT name FROM s_etracker_controllers';
        $result = $db->fetchAll($sql);
        if(!$result) {
            $insertSql = 'INSERT IGNORE INTO `s_etracker_controllers` (`name`) VALUES
            ("Account"),
            ("Blog"),
            ("Campaign"),
            ("Checkout"),
            ("Compare"),
            ("Custom"),
            ("Detail"),
            ("Forms"),
            ("Index"),
            ("Listing"),
            ("Newsletter"),
            ("Note"),
            ("Register"),
            ("Search"),
            ("Sitemap"),
            ("Tellafriend"),
            ("Ticket")';
            $db->Query($insertSql);
        }

    }

    /**
     * Helper method to create the table "s_etracker_divisions" and its values.
     */
    private function createDivisionTable()
    {
        $sql= 'CREATE TABLE IF NOT EXISTS `s_etracker_divisions` (
			 `id` int(11) NOT NULL AUTO_INCREMENT,
			 `name` text COLLATE utf8_unicode_ci NOT NULL,
			 PRIMARY KEY (`id`)
			) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;

			INSERT IGNORE INTO `s_etracker_divisions` (`id`, `name`) VALUES
			(1, "Controller"),
			(2, "Category"),
			(3, "Shop")';
        Shopware()->Db()->query($sql, array());
    }

    /**
     * Helper method to create the most important table "s_etracker_configs"
     * and the dynamical values.
     */
    private function createConfigTable()
    {
        $sql = 'CREATE TABLE IF NOT EXISTS `s_etracker_configs` (
			 `id` int(11) NOT NULL AUTO_INCREMENT,
			 `store_id` int(11) NOT NULL,
			 `account_key` text COLLATE utf8_unicode_ci NOT NULL,
			 `net_price` tinyint(4) NOT NULL,
			 `own_page_names` tinyint(4) NOT NULL,
			 `shopware_default` int(11) NOT NULL,
			 `own_schema` text COLLATE utf8_unicode_ci NOT NULL,
			 `divide_areas` int(11) NOT NULL,
			 `custom_attributes` text COLLATE utf8_unicode_ci NOT NULL,
			 `has_signalize_enabled` tinyint(4) NOT NULL,
			 `custom_tracking_domain` text COLLATE utf8_unicode_ci NOT NULL,
			 `use_article_number` tinyint(4) NOT NULL,
			 PRIMARY KEY (`id`),
			 UNIQUE KEY `shop_id` (`store_id`)
			) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci';
        Shopware()->Db()->query($sql, array());

        try 
        {
            $sql = 'ALTER TABLE `s_etracker_configs` ADD COLUMN `custom_attributes` text COLLATE utf8_unicode_ci NOT NULL';
            Shopware()->Db()->query($sql, array());
        }
        catch(Exception $e)
        {
        }

        try
        {
            $sql = 'ALTER TABLE `s_etracker_configs` ADD COLUMN `has_signalize_enabled` tinyint(4) NOT NULL';
            Shopware()->Db()->query($sql, array());
        }
        catch(Exception $e)
        {
        }

        try
        {
            $sql = 'ALTER TABLE `s_etracker_configs` ADD COLUMN `custom_tracking_domain` text COLLATE utf8_unicode_ci NOT NULL';
            Shopware()->Db()->query($sql, array());
        }
        catch(Exception $e)
        {
        }

        try
        {
            $sql = 'ALTER TABLE `s_etracker_configs` ADD COLUMN `use_article_number` tinyint(4) NOT NULL';
            Shopware()->Db()->query($sql, array());
        }
        catch(Exception $e)
        {
        }

        $sql= 'INSERT IGNORE INTO `s_etracker_configs` (
                store_id,
                account_key,
                net_price,
                own_page_names,
                shopware_default,
                own_schema,
                divide_areas,
                custom_attributes,
                has_signalize_enabled,
                custom_tracking_domain,
                use_article_number
            ) SELECT
			    id,
			    "",
			    0,
			    0,
			    0,
			    "",
			    0,
			    "",
			    0,
			    "",
			    0
              FROM s_core_shops WHERE main_id IS NULL';

        Shopware()->Db()->query($sql, array());
    }

    /**
     * Helper method to create the table "s_etracker_areas", which contains the areas of a shop.
     */
    private function createAreaTable()
    {
        $sql = 'CREATE TABLE IF NOT EXISTS `s_etracker_areas` (
			 `id` int(11) NOT NULL AUTO_INCREMENT,
			 `store_id` int(11) NOT NULL,
			 `name` text NOT NULL,
			 `division_id` int(11) NOT NULL,
			 PRIMARY KEY (`id`)
			) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci';
        Shopware()->Db()->query($sql, array());
    }

    /**
     * Helper method to create the menu-entry for this plugin.
     */
    private function createMenu()
    {
        if($this->Menu()->findOneBy(array('controller' => 'SwagETracker')) === null) {
            $this->createMenuItem(array(
                'label'         => 'etracker',
                'controller'    => 'SwagETracker',
                'class'         => 'etracker-icon',
                'action'        => 'Index',
                'active'        => 1,
                'parent'        => $this->Menu()->findOneBy(array('label' => 'Marketing'))
            ));
        }
    }

    /**
     * This method simply returns the path to the backend-controller.
     * Additionally it sets the snippet-directory.
     *
     * @param Enlight_Event_EventArgs $args
     * @return string
     */
    public function getBackendController(Enlight_Event_EventArgs $args)
    {
        $this->Application()->Snippets()->addConfigDir(
            $this->Path() . 'Snippets/'
        );
        $this->Application()->Template()->addTemplateDir($this->Path() . 'Views/');
        $this->registerCustomModels();

        return $this->Path() . '/Controllers/Backend/SwagETracker.php';
    }

    /**
     * This backend-event is only needed to load the etracker-icon to be displayed in the plugin-manager
     *
     * @param Enlight_Event_EventArgs $args
     */
    public function onPostDispatchIndex(Enlight_Event_EventArgs $args)
    {
        $args->getSubject()->View()->addTemplateDir(dirname(__FILE__) . '/Views/');
        $args->getSubject()->View()->extendsTemplate('backend/etracker/menu_entry.tpl');
    }

    /**
     * This backend-event is used to extend the emotion-module by a new field.
     * It includes all needed files and sets the view-directory.
     *
     * @param Enlight_Event_EventArgs $args
     */
    public function onGetControllerPathBackend(Enlight_Event_EventArgs $args)
    {
        /** @var $subject Enlight_Controller_Action */
        $subject = $args->getSubject();
        $request = $subject->Request();
        $view = $subject->View();

        if(!in_array($request->getActionName(), array('index', 'load'))) {
            return;
        }
        $this->registerCustomModels();

        $this->Application()->Snippets()->addConfigDir(
            $this->Path() . 'Snippets/'
        );

        $this->Application()->Template()->addTemplateDir(
            $this->Path() . 'Views/', 'etracker'
        );

        if($request->getActionName() == 'load')
        {
            $view->extendsTemplate('backend/emotion/etracker/view/detail/settings.js');
            $view->extendsTemplate('backend/emotion/etracker/model/emotion.js');
        }

        if ($args->getRequest()->getActionName() === 'index') {
            $view->extendsTemplate('backend/emotion/etracker/etracker_app.js');
        }
    }

    /**
     * Method to include the etracker-code everywhere in the frontend
     *
     * It handles all the configs, formats the values as needed and returns it to the frontend.
     */
    public function onPostDispatch(Enlight_Event_EventArgs $args)
    {
        $subject = $args->getSubject();

        /**
         * @var $request Enlight_Controller_Request_RequestHttp
         */
        $request = $subject->Request();

        /**
         * @var $view Enlight_View_Default
         */
        $view = $subject->View();

        $this->registerCustomModels();

        $this->Application()->Loader()->registerNamespace(
            'Shopware\etracker',
            $this->Path() . 'Components/'
        );
        $etracker = new Shopware\etracker\etracker();

        /**
         * @var $config \Shopware\CustomModels\etracker\Config
         */
        $config = Shopware()->Models()->getRepository('Shopware\CustomModels\etracker\Config')->findOneBy(array(
            'storeId' => $etracker->getMainShopId()
        ));

        if(!$config){
            return;
        }

        $pageName = '';
        if($config->getOwnPageNames()) {
            if($config->getShopwareDefault() == self::OWN_SCHEMA_ID){
                $pageName = $etracker->buildPageNameFromSchema($request, $view, $config);
            }else{
                $pageName = $etracker->buildPageName($request, $view);
            }

            if($request->getControllerName() == 'index'){
                $pageName = '__INDEX__'.$pageName;
            }
        }

        $etrackerView['pageName'] = urlencode($pageName);
        $etrackerView['areas'] = $etracker->getAffectedAreas($request, $view, $config);
        $etrackerView['accountKey'] = $config->getAccountKey();
        $etrackerView['customAttributes'] = $this->getCustomAttributes($config->getCustomAttributes());
        $etrackerView['customTrackingDomain'] = $config->getCustomTrackingDomain();
        $etrackerView['trackingDomain'] = $etrackerView['customTrackingDomain'] ?: '//code.etracker.com';
        $etrackerView['events'] = $etracker->getNecessaryEventInformation($request, $view);

        // Signalize
        $etrackerView['signalize'] = '';
        if($config->hasSignalizeEnabled()) {
            $pluginPath = $this->Path();
            $swPath = '';
            if(strpos($pluginPath, 'engine/Shopware/Plugins/Community/Frontend/SwagETracker') !== false) {
                $swPath = '/engine/Shopware/Plugins/Community/Frontend/SwagETracker/';
            } else if(strpos($pluginPath, 'custom/Plugins/SwagETracker') !== false) {
                $swPath = '/custom/Plugins/SwagETracker/';
            } else if(strpos($pluginPath, 'Plugins/Community/Frontend/SwagETracker') !== false) {
                $swPath = '/Plugins/Community/Frontend/SwagETracker/';
            } else if(strpos($pluginPath, 'Plugins/Local/Frontend/SwagETracker') !== false) {
                $swPath = '/Plugins/Local/Frontend/SwagETracker';
            }
            if($swPath) {
                $etrackerView['signalize'] = 'data-sw-path="' . $swPath . 'sw.js " data-sw-scope="' . $swPath . '"';
            }
        }

        $etrackerView['useArticleNumber'] = $config->getUseArticleNumber();
        $etrackerView['debugJs'] = $request->getParam('debugJs') ? true : false;

        //Search
        $etrackerView['searchCount'] =  $etracker->getSearchCount($request, $view);
        $etrackerView['searchterm'] =  str_replace(
			['&auml;', '&uuml;', '&ouml;', '&Auml;', '&Uuml;', '&Ouml;'],
			['ä', 'ü', 'ö', 'Ä', 'Ü', 'Ö'],
			$etracker->getSearchTerm($request, $view)
        );

        /** @var \Shopware\Models\Plugin\Plugin $pluginModel */
        $pluginModel = Shopware()->Models()->getRepository('\Shopware\Models\Plugin\Plugin')->findOneBy(array(
            'name'  =>  'SwagEtracker'
        ));
        if($pluginModel !== null) {
            $etrackerView['pluginVersion'] = 'Shopware_' . $pluginModel->getVersion();
        } else {
            $etrackerView['pluginVersion'] = 'Shopware';
        }

        $view->etracker = $etrackerView;

        $view->addTemplateDir(dirname(__FILE__) . '/Views');
    }

    /**
     * Method, which returns the path to the frontend-controller.
     *
     * @return string
     */
    public function getFrontendController()
    {
        return $this->Path() . '/Controllers/Frontend/SwagETracker.php';
    }

    /**
     * Provide the file collection for js files
     *
     * @param Enlight_Event_EventArgs $args
     * @return \Doctrine\Common\Collections\ArrayCollection
     */
    public function addJsFiles(Enlight_Event_EventArgs $args)
    {
        $jsFiles = array(__DIR__ . '/Views/frontend/_public/src/js/jquery.etracker.js');

        return new Doctrine\Common\Collections\ArrayCollection($jsFiles);
    }

    public function addStatisticCookie()
    {
        $collection = new CookieCollection();
        $collection->add(new CookieStruct(
            'etracker',
            '/^(_et_coid|et_|signalize)/',
            'etracker Analytics',
            CookieGroupStruct::STATISTICS
        ));

        return $collection;
    }

    private function getCustomAttributes($custom_attributes)
    {
        $custom_attributes_arr = [];
        if (! empty($custom_attributes) && strlen($custom_attributes) < 100) {
            $split = explode(';', $custom_attributes);
            foreach ($split as $entry) {
                $entry = trim($entry);
                if (strpos($entry, 'data-') === 0) {
                    $split = explode('=', $entry);
                    $key = trim($split[0]);
                    if (1 === count($split)) {
                        $custom_attributes_arr[] = $key;
                    } elseif (2 === count($split)) {
                        // any combination of ' and ".
                        $value = trim(trim($split[1]), '\'"');
                        $custom_attributes_arr[] = $key . '="' . $value . '"';
                    }
                }
            }
        }
        return join(' ', $custom_attributes_arr);
    }
}
