<?php
/**
 * Shopware 4
 * Copyright © shopware AG
 *
 * According to our dual licensing model, this program can be used either
 * under the terms of the GNU Affero General Public License, version 3,
 * or under a proprietary license.
 *
 * The texts of the GNU Affero General Public License with an additional
 * permission and of our proprietary license can be found at and
 * in the LICENSE file you have received along with this program.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * "Shopware" is a registered trademark of shopware AG.
 * The licensing of the program under the AGPLv3 does not imply a
 * trademark license. Therefore any rights, title and interest in
 * our trademarks remain entirely with us.
 */
namespace Shopware\etracker;

use \Enlight_Controller_Request_RequestHttp as RequestHttp;
use \Enlight_View_Default as View;
use \Shopware\CustomModels\etracker\Config as Config;

class etracker
{
	/**
	 * Helper method to get the doctrine entity manager.
	 */
	private function getEntityManager()
	{
		return Shopware()->Models();
	}

	/**
	 * Helper method to actually build the page-name by the given-parameters.
	 *
	 * @param $pageName
	 * @param null $customName
	 * @param null $customId
	 * @return string
	 */
	private function createPageName($pageName, $customName = null, $customId = null)
	{
		if($customName != null) {
			$pageName .= '_' . strtolower(str_replace(' ', '', $customName));
		}

		if($customId != null) {
			$pageName .= '(Id: ' . $customId . ')';
		}
		return $pageName;
	}

	/**
	 * Helper method to build the page-name by a shopware-default.
	 * It has the format controller_action_customName_(Id: Id)
	 * "customName" is for example the article-name or the category-title.
	 *
	 * @param $request \Enlight_Controller_Request_RequestHttp
	 * @param $view \Enlight_View_Default
	 * @return string
	 */
	public function buildPageName(RequestHttp $request, View $view)
	{
		$pageName   = $request->getControllerName() . '_' . $request->getActionName();
		$viewAssign = $view->getAssign();

		switch($request->getControllerName()) {
			case 'blog':
				$pageName = $this->createPageName($pageName, $viewAssign['sArticle']['title'], $viewAssign['sArticle']['id']);
				break;
			case 'detail':
				$pageName = $this->createPageName($pageName, $viewAssign['sArticle']['articleName'], $viewAssign['sArticle']['articleID']);
				break;
			case 'listing':
				$pageName = $this->createPageName($pageName, $viewAssign['sCategoryContent']['name'], $viewAssign['sCategoryContent']['id']);
				break;
			case 'forms':
				$pageName = $this->createPageName($pageName, $viewAssign['sSupport']['name'], $viewAssign['sSupport']['id']);
				break;
			case 'custom':
				$pageName = $this->createPageName($pageName, $viewAssign['sCustomPage']['description'], $viewAssign['sCustomPage']['id']);
				break;
			case 'campaign':
				$pageName = $this->createPageName($pageName, $viewAssign['sBreadCrumb'][0]['name'], $viewAssign['emotionId']);
				break;
            case 'newsletter':
                $pageName = 'newsletter';

                if($request->getParam('subscribeToNewsletter') == '1') {
                    $pageName .= '_success';
                }

				break;
			default:
				break;
		}

		return trim($pageName, '-_/.,;|');
	}

    /**
     * Helper method to get the SearchTerm.
     *
     * @param $request \Enlight_Controller_Request_RequestHttp
     * @param $view \Enlight_View_Default
     * @return string
     */
    public function getSearchTerm(RequestHttp $request, View $view)
    {
        switch($request->getControllerName()) {
            case 'search':
                $searchterm = htmlentities(strip_tags($request->getParam('sSearch')));
                break;
            default:
                $searchterm = '';
                break;
        }

        return trim($searchterm, '-_/.,|');
    }

    /**
     * Helper method to get the Search Count.
     *
     * @param $request \Enlight_Controller_Request_RequestHttp
     * @param $view \Enlight_View_Default
     * @return int
     */
    public function getSearchCount(RequestHttp $request, View $view)
    {
        switch($request->getControllerName()) {
            case 'search':
                $viewAssign = $view->getAssign();
                $searchCount = intval($viewAssign['sSearchResults']['sArticlesCount']);
                break;
            default:
                $searchCount = 0;
                break;
        }

        return trim($searchCount);
    }

	/**
	 * Helper method to build the page-name from the schema, which the user may set in the config.
	 * It simple replaces the placeholders with actual values.
	 *
	 * @param $request \Enlight_Controller_Request_RequestHttp
	 * @param $view \Enlight_View_Default
	 * @param $config \Shopware\CustomModels\etracker\Config
	 * @return string
	 */
	public function buildPageNameFromSchema(RequestHttp $request, View $view, Config $config)
	{
		$schema     = $config->getOwnSchema();
		$controller = $request->getControllerName();
		$action     = $request->getActionName();

		list($custom, $id) = $this->getAdditionalInfo($request, $view);

		$schema = str_replace('%controller%', $controller, $schema);
		$schema = str_replace('%action%', $action, $schema);

		if($custom) {
			$schema = str_replace('%custom%', $custom, $schema);
		} else {
			$schema = str_replace('%custom%', '', $schema);
		}

		if($custom) {
			$schema = str_replace('%id%', $id, $schema);
		} else {
			$schema = str_replace('%id%', '', $schema);
		}

		$schema = trim($schema, '-_/.,;|()');

		return $schema;
	}

	/**
	 * Helper method to check if the user is on the order-finish page.
	 *
	 * @param $request \Enlight_Controller_Request_RequestHttp
	 * @return bool
	 */
	private function isOnCheckoutFinish(RequestHttp $request)
	{
		return $request->getControllerName() == 'checkout' && $action = $request->getActionName() == 'finish';
	}

	/**
	 * Get the net or gross price
	 *
	 * @param $useNetPrice
	 * @param $viewAssign
	 * @return mixed
	 */
	private function getPrice($useNetPrice, $viewAssign)
	{
		if($useNetPrice) {
			return str_replace(',', '.', $viewAssign['sAmountNet']);
		} else {
			if($viewAssign['sAmountWithTax']) {
				return str_replace(',', '.', $viewAssign['sAmountWithTax']);
			} else {
				return str_replace(',', '.', $viewAssign['sAmount']);
			}
		}
	}

	/**
	 * Helper method to get the affected areas for the current controllers.
	 *
	 * @param $request \Enlight_Controller_Request_RequestHttp
	 * @param $view \Enlight_View_Default
	 * @param $config \Shopware\CustomModels\etracker\Config
	 * @return string
	 */
	public function getAffectedAreas(RequestHttp $request, View $view, Config $config)
	{
		$divideInto = $config->getDivideAreas();
		$controller = $request->getControllerName();
		$viewAssign = $view->getAssign();
		$area       = '';

		switch($divideInto) {
			//Controller
			case 0:
				$area = $controller;
				break;

			//Categories
			case 1:
                $id = null;
				if($controller == 'listing') {
                    $id = $viewAssign['sCategoryContent']['id'];
				} elseif($controller == 'detail') {
                    $id = $viewAssign['sCategoryInfo']['id'];
				}

                if($id) {
                    $path = $this->getEntityManager()->getRepository('Shopware\Models\Category\Category')->getPathById($id);
                    if(is_string($path)) {
                        $area = $path;
                    } else if (is_array($path)) {
                        $area = implode('/', array_chunk($path, 4)[0]);
                    }
				}
				break;

			//Shops
			case 2:
				$area = Shopware()->Shop()->getName();
				break;

			//Own areas
			case 3:
				$area = $this->handleOwnArea($controller, $viewAssign);
				break;

            // ETracker default
            case 4:
                $controllerArray = array(
                    'detail', 'listing', 'custom', 'forms', 'blog', 'index', 'newsletter', 'sitemap'
                );

                if(in_array($controller, $controllerArray)) {
                    $path = explode('/',$request->getPathInfo());
                    $path = $path[1];

                    if(empty($path)){
                        $path = 'home';
                    }

                    $area = ucfirst($path);
                }
                break;
			default:
				break;
		}

		return urlencode($area);
	}

	/**
	 * Helper method to get the id of the main-shop.
	 *
	 * @return int ID of the main-shop
	 */
	public function getMainShopId()
	{
		if($main = Shopware()->Shop()->getMain()){
			return $main->getId();
		}else{
			return Shopware()->Shop()->getId();
		}
	}

	/**
	 * Helper method to handle the own configured areas.
	 * There may be more than one area for a controller/category/shop.
	 *
	 * @param $currentController
	 * @param $viewAssign
	 * @return string
	 */
	private function handleOwnArea($currentController, $viewAssign)
	{
		$controllerAreas = $this->getEntityManager()->getRepository('Shopware\CustomModels\etracker\Area')->findBy(array(
			'storeId'       => $this->getMainShopId(),
			'divisionId'    => 1
		));

		$categoryAreas = $this->getEntityManager()->getRepository('Shopware\CustomModels\etracker\Area')->findBy(array(
			'storeId'       => $this->getMainShopId(),
			'divisionId'    => 2
		));

		$shopAreas = $this->getEntityManager()->getRepository('Shopware\CustomModels\etracker\Area')->findBy(array(
			'storeId'       => $this->getMainShopId(),
			'divisionId'    => 3
		));

		$result = '';
		foreach($shopAreas as $shopArea) {
			$shops = $shopArea->getShops();

			if(empty($shops)) {
				$result .= $shopArea->getName() . ',';
			} else {
				foreach($shops as $shop) {
					if($shop->getId() == Shopware()->Shop()->getId()) {
						$result .= $shopArea->getName() . ',';
					}
				}
			}
		}

		foreach($controllerAreas as $controllerArea) {
			$controllers = $controllerArea->getControllers();
			if(empty($controllers)) {
				$result .= $controllerArea->getName() . ',';
			} else {
				foreach($controllers as $controller) {
					if(strtolower($controller->getName()) == $currentController) {
						$result .= $controllerArea->getName() . ',';
					}
				}
			}
		}

		foreach($categoryAreas as $categoryArea) {
			if($currentController == 'listing' || $currentController == 'detail') {
				$categories = $categoryArea->getCategories();
				if(empty($categories)) {
					$result .= $categoryArea->getName() . ',';
				} else {
					foreach($categories as $category) {
						if($category->getId() == $viewAssign['sCategoryInfo']['id']) {
							$result .= $categoryArea->getName() . ',';
						}
					}
				}
			}
		}

		return trim($result, ',');
	}

	/**
	 * Helper method to get additional informations for the current controller,
	 * e.g. article-name and article-id.
	 *
	 * @param $request \Enlight_Controller_Request_RequestHttp
	 * @param $view \Enlight_View_Default
	 * @return array|null
	 */
	private function getAdditionalInfo(RequestHttp $request, View $view)
	{
		$viewAssign = $view->getAssign();
		$info       = array();

		switch($request->getControllerName()) {
			case 'blog':
				$info[] = $viewAssign['sArticle']['title'];
				$info[] = $viewAssign['sArticle']['id'];
				break;
			case 'detail':
				$info[] = $viewAssign['sArticle']['articleName'];
				$info[] = $viewAssign['sArticle']['articleID'];
				break;
			case 'listing':
				$info[] = $viewAssign['sCategoryContent']['name'];
				$info[] = $viewAssign['sCategoryContent']['id'];
				break;
			case 'forms':
				$info[] = $viewAssign['sSupport']['name'];
				$info[] = $viewAssign['sSupport']['id'];
				break;
			case 'custom':
				$info[] = $viewAssign['sCustomPage']['description'];
				$info[] = $viewAssign['sCustomPage']['id'];
				break;
			case 'campaign':
				$info[] = $viewAssign['sBreadcrumb'][0]['name'];
				$info[] = $viewAssign['emotionId'];
				break;
			default:
				return null;
		}

		return $info;
	}

	/**
	 * Method to read out necessary information about the needed events.
	 *
	 * @param RequestHttp $request
	 * @param View $view
	 * @return array
	 */
	public function getNecessaryEventInformation(RequestHttp $request, View $view)
	{
		$controller = $request->getControllerName();
		$action     = $request->getActionName();
		$viewAssign = $view->getAssign();
		$events     = array();

		if($controller == 'detail') {
			$sArticle = $viewAssign['sArticle'];
            $eventArticle = $this->getEventArticle($sArticle, $viewAssign);
			$events[] = $this->createEvent('viewProduct', $eventArticle, 'send');
            $events[] = $this->createEvent('insertToBasket', $eventArticle, 'attach', 'basketButton');
            $events[] = $this->createEvent('insertToWatchlist', $eventArticle, 'attach', 'watchlistButton');
        } elseif($controller == 'note') {
            foreach ($viewAssign['sNotes'] as $sArticle) {
                $sArticle['categoryID'] = 'Wunsch- / Merkliste';
                $events[]               = $this->createEvent('removeFromWatchlist', $this->getEventArticle($sArticle, $viewAssign), 'attach', 'watchlistButton-' . $sArticle['articleID']);
            }
        } elseif($controller == 'wishlist') {
		    foreach($viewAssign['savedBaskets'] as $savedBasket) {
                foreach ($savedBasket['items'] as $sArticle) {
                    $sArticle               = $sArticle['article'];
                    $sArticle['categoryID'] = 'Wunsch- / Merkliste';
                    $events[]               = $this->createEvent('removeFromWatchlist', $this->getEventArticle($sArticle, $viewAssign), 'attach', 'watchlistButton-' . $sArticle['articleID']);
                }
            }
        } elseif($controller == 'listing') {
            foreach($viewAssign['sArticles'] as $sArticle) {
                $events[] = $this->createEvent('insertToWatchlist', $this->getEventArticle($sArticle, $viewAssign), 'attach', 'watchlistButton-' . $sArticle['articleID']);
            }
        } elseif($controller == 'checkout' && $action == 'finish') {
            if (!empty($viewAssign['sOrderNumber'])) {
                $events[] = $this->createEvent('order', $this->getOrderData($viewAssign), 'send');
            }
		}

		return $events;
	}

	/**
	 * Helper method to create an event-array.
	 *
	 * @param $name
	 * @param $information
	 * @param $type
	 * @param null $attachId
	 * @return array
	 */
	private function createEvent($name, $information, $type, $attachId = null)
	{
		return array('name' => $name, 'information' => $information, 'type' => $type, 'attachId' => $attachId);
	}

	/**
	 * Helper method to filter all the unnecessary information from the article-array.
	 *
	 * @param $sArticle
	 * @param $viewAssign
	 * @return mixed
	 */
	private function getEventArticle($sArticle, $viewAssign)
	{
        /** @var Shopware\CustomModels\etracker\Config $config */
        $config = Shopware()->Models()->getRepository('Shopware\CustomModels\etracker\Config')->findOneBy(array(
            'storeId' => $this->getMainShopId()
        ));

        $article['id']          = $config->getUseArticleNumber() ? $sArticle['ordernumber'] : strval($sArticle['articleID']);
		$article['name']        = $sArticle['articleName'];

/*
        foreach ($sArticle['sConfigurator'] as $configuration) {
            $groupName = $configuration['groupname'];
            $selected = $configuration['selected_value'];
            $values = $configuration['values'];
            $value = $values[$selected];
            $variantName = $value['optionname'];
        }
*/

        if(is_null($sArticle['categoryID'])) {
            if(array_key_exists('sCategoryContent', $viewAssign)) {
                $sArticle['categoryID'] = $viewAssign['sCategoryContent']['id'];
            }
        }

        if(!is_null($sArticle['categoryID'])) {
            if ($sArticle['categoryID'] == 'Wunsch- / Merkliste') {
                $article['category'] = array('Wunsch- / Merkliste');
            } else {
                $path = $this->getEntityManager()->getRepository('Shopware\Models\Category\Category')->getPathById($sArticle['categoryID'], 'name', '$$');
                if(is_string($path)) {
                    $categoryPath = $path;
                } else if (is_array($path)) {
                    $categoryPath = implode('$$', array_chunk($path, 4)[0]);
                }
                if ($categoryPath == 'Root') {
                    $article['category'] = array('Keine Kategorie');
                } else {
                    $article['category'] = explode('$$', $categoryPath);
                }
            }
        }

		$sUserData      = Shopware()->Modules()->Admin()->sGetUserData();
		$customerGroup  = Shopware()->Shop()->getCustomerGroup()->getKey();
		if($sUserData['billingaddress']['countryID']) {
			$countryId = $sUserData['billingaddress']['countryID'];
		}

		$customerGroupModel = $this->getEntityManager()->getRepository('Shopware\Models\Customer\Group')->findOneBy(array('key' => $customerGroup));
		$showTax            = $customerGroupModel->getTax();

		if($countryId) {
			$showTax = !$this->getEntityManager()->getRepository('Shopware\Models\Country\Country')->find($countryId)->getTaxFree();
		}

        $normalPrice = str_replace(',', '.', $sArticle['price']);
        $articlePrice = $this->getTaxPrice($config->getUseNetPrice(), $showTax, $normalPrice, $sArticle['tax']);

        $prices = [];
        $repo = $this->getEntityManager()->getRepository('Shopware\Models\Article\Price');
        if($repo) {
            $res = $repo->getPricesQuery($sArticle['articleDetailsID']);
            if($res) {
                $prices = $res->getArrayResult();
            }
        }

        $prices_array = [];
        foreach ($prices as $price) {
            if($price['customerGroupKey'] == $customerGroup) {
                $prices_array[] = [ 'from' => $price['from'], 'percent' => $price['percent']];
            }
        }

        // makes it easier to search for the correct percentage
        usort($prices_array, function($a, $b) {
            return $b['from'] - $a['from'];
        });

        $article['price']       = $articlePrice;
        $article['pricesArray'] = $prices_array;
		$article['currency']    = Shopware()->Shop()->getCurrency()->getCurrency();
		$article['variants']    = new \stdClass();

		return $article;
	}

    /**
     * Helper method to do some tax magic.
     *
     * @param $useNetPrice
     * @param $showTax
     * @param $normalPrice
     * @param $tax
     * @return string
     */
    private function getTaxPrice($useNetPrice, $showTax, $normalPrice, $tax)
    {
        $ret = $normalPrice;
        if ($useNetPrice && $showTax) {
            $ret = round($normalPrice / ($tax / 100 + 1), 2);
        } else if (!$useNetPrice && !$showTax) {
            $ret = round($normalPrice * ($tax / 100 + 1), 2);
        }
        return (string)$ret;
    }

    /**
	 * Helper method to create an array with all necessary basket-information.
	 *
	 * @param $viewAssign
	 * @return array
	 */
	private function getOrderData($viewAssign)
	{
		$basket     = $viewAssign['sBasket'];
		$userData   = $viewAssign['sUserData'];
		$products   = array();

		foreach($basket['content'] as $position) {
            $price = str_replace(',', '.', $position['price']);
			if($position['modus'] == 0) {
                // regular article
				$article = Shopware()->Modules()->Articles()->sGetArticleById($position['articleID']);
                $article['price'] = (string) round($price, 2);
			} else {
                // coupon
				$article = array(
					'articleID'     => $position['articleID'],
                    'ordernumber'   => $position['ordernumber'],
					'articleName'   => $position['articlename'],
					'categoryID'    => 1,
                    'price'         => (string) round($price, 2),
                    'currency'      => Shopware()->Shop()->getCurrency()->getCurrency(),
                    'variants'      => new \stdClass()
				);
			}
			if(!empty($article['articleID'])) {
                $eventArticle = $this->getEventArticle($article, $viewAssign);
                unset ($eventArticle['pricesArray']); // remove pricesArray from eventArticle, because it is not needed in 'order' event
				$products[] = array('product' => $eventArticle, 'quantity' => (int) $position['quantity']);
			}
		}

        // Versandkosten
        $products[] = array(
            'product'   =>  array(
                'id'            => 'Versandkosten',
                'name'          => 'Versandkosten',
                'category'      => array('Keine Kategorie'),
                'price'         => strval($basket['sShippingcostsWithTax']),
                'variants'      => new \stdClass(),
                'currency'      => Shopware()->Shop()->getCurrency()->getCurrency()
            ),
            'quantity'  =>  1
        );

		$config = Shopware()->Models()->getRepository('Shopware\CustomModels\etracker\Config')->findOneBy(array(
			'storeId' => $this->getMainShopId()
		));
		$orderPrice = $this->getPrice($config->getUseNetPrice(), $viewAssign);

		$data = array(
			'orderNumber'           => (string) $viewAssign['sOrderNumber'],
			'status'                => 'sale',
			'orderPrice'            => (string) $orderPrice,
			'currency'              => Shopware()->Shop()->getCurrency()->getCurrency(),
			'basket'                => array(
				'id'        => (string) $this->getEntityManager()->getRepository('Shopware\Models\Order\Order')->findOneBy(array('number' => $viewAssign['sOrderNumber']))->getId(),
				'products'  => $products
			),
			'customerGroup'         => $userData['additional']['user']['customergroup'],
            'deliveryConditions'    => $viewAssign['sDispatch']['name'],
            'paymentConditions'     => $viewAssign['sPayment']['name']
		);

		return $data;
	}
}