<?php
/**
 * Shopware 4
 * Copyright © shopware AG
 *
 * According to our dual licensing model, this program can be used either
 * under the terms of the GNU Affero General Public License, version 3,
 * or under a proprietary license.
 *
 * The texts of the GNU Affero General Public License with an additional
 * permission and of our proprietary license can be found at and
 * in the LICENSE file you have received along with this program.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * "Shopware" is a registered trademark of shopware AG.
 * The licensing of the program under the AGPLv3 does not imply a
 * trademark license. Therefore any rights, title and interest in
 * our trademarks remain entirely with us.
 */
class Shopware_Controllers_Backend_SwagETracker extends Shopware_Controllers_Backend_ExtJs
{
	/**
	 * Entity Manager
	 * @var null
	 */
	protected $manager = null;

	/**
	 * @var \Shopware\Components\Model\ModelRepository
	 */
	protected $configRepository = null;

	/**
	 * @var \Shopware\Components\Model\ModelRepository
	 */
	protected $areaRepository = null;

	/**
	 * @var \Shopware\Components\Model\ModelRepository
	 */
	protected $divisionRepository = null;

	/**
	 * @var \Shopware\Components\Model\ModelRepository
	 */
	protected $controllerRepository = null;

	/**
	 * @var \Shopware\Models\Shop\Repository
	 */
	protected $shopRepository = null;

	/**
	 * @var \Shopware\Models\Category\Repository
	 */
	protected $categoryRepository = null;

	/**
	 * @var \Shopware\Components\Model\QueryBuilder
	 */
	protected $queryBuilder = null;

	/**
	 * Internal helper function to get access to the entity manager.
	 * @return Shopware\Components\Model\ModelManager
	 */
	protected function getManager()
	{
		if ($this->manager === null) {
			$this->manager = Shopware()->Models();
		}

		return $this->manager;
	}

	/**
	 * Helper method to get the repository of the config-model.
	 * @return \Shopware\Components\Model\ModelRepository
	 */
	protected function getConfigRepository()
	{
		if($this->configRepository == null)
		{
			$this->configRepository = $this->getManager()->getRepository('Shopware\CustomModels\etracker\Config');
		}
		return $this->configRepository;
	}

	/**
	 * Helper method to get the repository of the area-model.
	 * @return \Shopware\Components\Model\ModelRepository
	 */
	protected function getAreaRepository()
	{
		if($this->areaRepository == null)
		{
			$this->areaRepository = $this->getManager()->getRepository('Shopware\CustomModels\etracker\Area');
		}
		return $this->areaRepository;
	}

	/**
	 * Helper method to get the repository of the division-model.
	 * @return \Shopware\Components\Model\ModelRepository
	 */
	protected function getDivisionRepository()
	{
		if($this->divisionRepository == null)
		{
			$this->divisionRepository = $this->getManager()->getRepository('Shopware\CustomModels\etracker\Division');
		}
		return $this->divisionRepository;
	}

	/**
	 * Helper method to get the repository of the controller-model.
	 * @return \Shopware\Components\Model\ModelRepository
	 */
	protected function getControllerRepository()
	{
		if($this->controllerRepository == null)
		{
			$this->controllerRepository = $this->getManager()->getRepository('Shopware\CustomModels\etracker\Controller');
		}
		return $this->controllerRepository;
	}

	/**
	 * Helper method to get the repository of the shop-model.
	 * @return \Shopware\Models\Shop\Repository
	 */
	protected function getShopRepository()
	{
		if($this->shopRepository == null)
		{
			$this->shopRepository = $this->getManager()->getRepository('Shopware\Models\Shop\Shop');
		}
		return $this->shopRepository;
	}

	/**
	 * Helper method to get the repository of the shop-model.
	 * @return \Shopware\Models\Category\Repository
	 */
	protected function getCategoryRepository()
	{
		if($this->categoryRepository == null)
		{
			$this->categoryRepository = $this->getManager()->getRepository('Shopware\Models\Category\Category');
		}
		return $this->categoryRepository;
	}

	/**
	 * @return \Shopware\Components\Model\QueryBuilder
	 */
	protected function getQueryBuilder()
	{
		if($this->queryBuilder == null)
		{
			$this->queryBuilder = $this->getManager()->createQueryBuilder();
		}
		return $this->queryBuilder;
	}

	/**
	 * Method to get all the possible configs for the stores
	 */
	public function getConfigAction()
	{
		$builder = $this->getConfigRepository()->createQueryBuilder('config');
		$records = $builder->select('config')
					->getQuery()->getArrayResult();

		$this->View()->assign(array(
			'success'   => true,
			'data'      => $records
		));
	}

	/**
	 * Method to update the config.
	 */
	public function saveConfigAction()
	{
		$configId = $this->Request()->getParam('id');

		if(empty($configId)) {
			$configModel = new Shopware\CustomModels\etracker\Config();
		} else {
			$configModel = $this->getConfigRepository()->find($configId);
		}

		$configModel->fromArray($this->Request()->getParams());

		$this->getManager()->persist($configModel);
		$this->getManager()->flush();

		$this->View()->assign(array(
			'success'   =>  true,
			'data'      =>  $this->getManager()->toArray($configModel)
		));
	}

	/**
	 * Method to read out all the areas by a given store-id.
	 * Additionally this method supports paging, so you can limit the results and set an offset.
	 */
	public function getAreasAction()
	{
		$builder = $this->getQueryBuilder();

		$builder->select('areas', 'categories', 'shops', 'controllers', 'division')
				->from('Shopware\CustomModels\etracker\Area', 'areas')
				->leftJoin('areas.categories', 'categories')
				->leftJoin('areas.shops', 'shops')
				->leftJoin('areas.controllers', 'controllers')
				->innerJoin('areas.division', 'division');

		//If a filter is set
		if ($this->Request()->get('filter')) {
			//Get the value itself
			$filters = $this->Request()->get('filter');

			foreach($filters as $filter) {
				if($filter['property'] == 'storeId') {
					$builder->andWhere('areas.storeId = :storeId')
						    ->setParameter('storeId', $filter['value']);
				} elseif($filter['property'] == 'searchValue') {
					$builder->andWhere('areas.name LIKE :value')
						    ->setParameter('value', '%' . $filter['value'] . '%');
				}
			}
		}

		$builder->setFirstResult($this->Request()->getParam('start'))
			    ->setMaxResults($this->Request()->getParam('limit'));

		$query      = $builder->getQuery()->setHydrationMode(\Doctrine\ORM\AbstractQuery::HYDRATE_ARRAY);
        $paginator  = Shopware()->Models()->createPaginator($query);
		$totalCount = $paginator->count();
		$result     = $this->formatResult($paginator->getIterator()->getArrayCopy());

		$this->View()->assign(array(
			'success'   => true,
			'data'      => $result,
			'total'     => $totalCount
		));
	}

	/**
	 * Helper method to format a string by the mapping-values.
	 * E.g. all the controllers in an area will be enumerated by a ","
	 *
	 * @param $areas
	 * @return mixed
	 */
	private function formatResult($areas) {
		foreach($areas as &$area) {
			$assigned = '';
			switch($area['divisionId']) {
				//Controllers
				case 1:
					foreach($area['controllers'] as $controller){
						$assigned .= $controller['name'] . ', ';
					}
					break;

				//Categories
				case 2:
					foreach($area['categories'] as $category) {
						$assigned .= $category['name'] . ', ';
					}
					break;

				//Shops
				default:
					foreach($area['shops'] as $shop){
						$assigned .= $shop['name'] . ', ';
					}
					break;
			}

			$area['assigned']       = trim($assigned, ', ');
			$area['divisionName']   = $area['division']['name'];
		}

		return $areas;
	}

	/**
	 * Method to simply delete one or multiple areas.
	 */
	public function deleteAreaAction()
	{
		$repository = $this->getAreaRepository();
		$params     = $this->getParams();

		if($params[0]) {
			foreach($params as $values) {
				$id     = $values['id'];
				$model  = $repository->find($id);

				$this->getManager()->remove($model);
				$this->getManager()->flush();
			}
		} else {
			$id     = $this->Request()->get('id');
			$model  = $repository->find($id);

			$this->getManager()->remove($model);
			$this->getManager()->flush();
		}

		$this->View()->assign(array('success' => true));
	}

	/**
	 * Method to update an existing area.
	 */
	public function updateAreaAction()
	{
		$params     = $this->getParams();
		$areaModel  = $this->getAreaRepository()->find($params['id']);

		$areaModel->setName($params['name']);
		$areaModel->setDivision($this->getDivisionRepository()->find($params['divisionId']));

		$this->getManager()->persist($areaModel);
		$this->getManager()->flush();

		$this->View()->assign(array('success' => true, 'data' => $params));
	}

	/**
	 * Method to create a new area.
	 */
	public function createAreaAction()
	{
		$params = $this->getParams();

		$areaModel = new \Shopware\CustomModels\etracker\Area();
		$areaModel->fromArray($params);

		/**
		 * @var \Shopware\CustomModels\etracker\Division $divisionModel
		 */
		$divisionModel = $this->getDivisionRepository()->find($this->Request()->getParam('divisionId'));
		$areaModel->setDivision($divisionModel);

		$this->getManager()->persist($areaModel);
		$this->getManager()->flush();

		$this->View()->assign(array('success' => true));
	}

	/**
	 * Method to delete an area-shop mapping.
	 */
	public function deleteAreaShopAction()
	{
		$params = $this->getParams();
		$this->removeAssociation($params['id'], $params['areaId'], 'Shops');

		$this->View()->assign(array('success' => true));
	}

	/**
	 * Method to delete an area-controller mapping.
	 */
	public function deleteAreaControllerAction()
	{
		$params = $this->getParams();
		$this->removeAssociation($params['id'], $params['areaId'], 'Controllers');

		$this->View()->assign(array('success' => true));
	}

	/**
	 * Method to delete an area-category mapping.
	 */
	public function deleteAreaCategoryAction()
	{
		$params = $this->getParams();
		$this->removeAssociation($params['id'], $params['areaId'], 'Categories');

		$this->View()->assign(array('success' => true));
	}

	/**
	 * Helper method to remove a single mapping-entry.
	 * It reads out all the entries, searches for the entry, that is supposed to get removed and
	 * only saves back the other entries.
	 *
	 * @param $associationId
	 * @param $areaId
	 * @param $function
	 */
	private function removeAssociation($associationId, $areaId, $function)
	{
		$areaModel      = $this->getAreaRepository()->find($areaId);
		$getFunction    = 'get' . $function;
		$setFunction    = 'set' . $function;
		$items          = $areaModel->$getFunction();

		foreach($items as $key => $item) {
			if($item->getId() == $associationId) {
				unset($items[$key]);
			}
		}
		$areaModel->$setFunction($items);

		$this->getManager()->persist($areaModel);
		$this->getManager()->flush();
	}

	/**
	 * This method reads out all the categories by a given search-query.
	 * The values will be used in a dropdown in the area-detail window.
	 */
	public function getCategoriesAction()
	{
		/** @var $repository \Shopware\Models\Category\Repository */
		$repository = $this->getCategoryRepository();

		$filter = array(
			'name' => '%' . $this->Request()->getParam('query') . '%'
		);

		$query = $repository->getListQuery(
				$filter,
				$this->Request()->getParam('sort', array()),
				$this->Request()->getParam('limit'),
				$this->Request()->getParam('start')
		);

		//get total result of the query
		$total = $this->getManager()->getQueryCount($query);

		//select all shop as array
		$data = $query->getArrayResult();

		//return the data and total count
		$this->View()->assign(array('success' => true, 'data' => $data, 'total' => $total));
	}

	/**
	 * This method reads out all the shops by a given search-query.
	 * The values will be used in a dropdown in the area-detail window.
	 */
	public function getShopsAction()
	{
		$order = $order = $this->Request()->getParam('sort', array());

		$builder = $this->getQueryBuilder();
		$fields = array(
			'shop.id as id',
			'locale.id as localeId',
			'category.id as categoryId',
			'currency.id as currencyId',
			'shop.default as default',
			'shop.active as active',
			'shop.name as name'
		);

		$builder->select($fields)
			    ->from('Shopware\Models\Shop\Shop', 'shop')
                ->leftJoin('shop.locale', 'locale')
				->leftJoin('shop.category', 'category')
				->leftJoin('shop.currency', 'currency')
				->where('shop.name LIKE :name')
				->andWhere('shop.main = :storeId')
				->orderBy('default', 'DESC')
				->addOrderBy('name');

		$builder->setParameter('name', '%' . $this->Request()->getParam('query') . '%');
		$builder->setParameter('storeId', $this->Request()->getParam('storeId'));

		if($order !== null) {
			$builder->addOrderBy($order);
		}

		$query = $builder->getQuery();

		//get total result of the query
		$total = $this->getManager()->getQueryCount($query);

		//select all shop as array
		$data = $query->getArrayResult();

		//return the data and total count
		$this->View()->assign(array('success' => true, 'data' => $data, 'total' => $total));
	}

	/**
	 * This method reads out all the controllers by a given search-query.
	 * The values will be used in a dropdown in the area-detail window.
	 */
	public function getControllersAction()
	{
		$builder = $this->getQueryBuilder();
		$builder->select(array('controllers.id', 'controllers.name'))
			    ->from('Shopware\CustomModels\etracker\Controller', 'controllers')
			    ->where('controllers.name LIKE :name')
			    ->setParameter('name', '%' . $this->Request()->getParam('query') . '%');


		//get total result of the query
		$total = $this->getManager()->getQueryCount($builder->getQuery());

		//select all shop as array
		$data = $this->addDescriptionToControllernames($builder->getQuery()->getArrayResult());

		//return the data and total count
		$this->View()->assign(array('success' => true, 'data' => $data, 'total' => $total));
	}

    /**
     * This method iterate over the quereResult and gets the additional description
     * from the snippets
     *
     * @param $data
     * @return array
     */
    private function addDescriptionToControllerNames($data)
    {
        $returnValue = array();

        foreach($data as $controllerName) {
            $controller = array();
            $name       = $controllerName['name'];

            $controller['id']   = $controllerName['id'];
            $controller['name'] = $this->getDescriptionByName($name);

            $returnValue[] = $controller;
        }

        return $returnValue;
    }

    /**
     * This method gets the snippet by name
     * @param $name
     * @return string
     */
    private function getDescriptionByName($name)
    {
        $nameSpace = Shopware()->Snippets()->getNamespace('backend/etracker/view/main');
        return $name . ' (' . $nameSpace->get('controllerDescription/' . $name) . ')';
    }

	/**
	 * Method to create an area-controller mapping
	 */
	public function createAreaControllerAction()
	{
		$areaId             = $this->Request()->getParam('areaId');
		$controllerId       = $this->Request()->getParam('id');

		/** @var Shopware\CustomModels\etracker\Area $areaModel */
		$areaModel          = $this->getAreaRepository()->find($areaId);

		/** @var Shopware\CustomModels\etracker\Controller $controllerModel */
		$controllerModel    = $this->getControllerRepository()->find($controllerId);

		$areaModel->addController($controllerModel);

		$this->getManager()->flush($areaModel);

		$this->View()->assign(array('success' => true));
	}

	/**
	 * Method to create an area-shop mapping
	 */
	public function createAreaShopAction()
	{
		$areaId = $this->Request()->getParam('areaId');
		$shopId = $this->Request()->getParam('id');

		/** @var Shopware\CustomModels\etracker\Area $areaModel */
		$areaModel = $this->getAreaRepository()->find($areaId);

		/** @var Shopware\Models\Shop\Shop $shopModel */
		$shopModel = $this->getShopRepository()->find($shopId);

		$areaModel->addShop($shopModel);

		$this->getManager()->flush($areaModel);

		$this->View()->assign(array('success' => true));
	}

	/**
	 * Method to create an area-category mapping
	 */
	public function createAreaCategoryAction()
	{
		$areaId         = $this->Request()->getParam('areaId');
		$categoryId     = $this->Request()->getParam('id');

		/** @var Shopware\CustomModels\etracker\Area $areaModel */
		$areaModel      = $this->getAreaRepository()->find($areaId);

		/** @var Shopware\Models\Category\Category $categoryModel */
		$categoryModel  = $this->getCategoryRepository()->find($categoryId);

		$areaModel->addCategory($categoryModel);

		$this->getManager()->flush($areaModel);

		$this->View()->assign(array('success' => true));
	}

	/**
	 * Helper method to get the important params and remove the unnecessary ones.
	 *
	 * @return mixed
	 */
	private function getParams()
	{
		$params = $this->Request()->getParams();

		unset($params['module']);
		unset($params['controller']);
		unset($params['action']);
		unset($params['_dc']);

		return $params;
	}
}