/**
 * Shopware 4
 * Copyright © shopware AG
 *
 * According to our dual licensing model, this program can be used either
 * under the terms of the GNU Affero General Public License, version 3,
 * or under a proprietary license.
 *
 * The texts of the GNU Affero General Public License with an additional
 * permission and of our proprietary license can be found at and
 * in the LICENSE file you have received along with this program.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * "Shopware" is a registered trademark of shopware AG.
 * The licensing of the program under the AGPLv3 does not imply a
 * trademark license. Therefore any rights, title and interest in
 * our trademarks remain entirely with us.
 *
 * @category   Shopware
 * @package    SwagETracker
 * @subpackage Controller
 * @copyright  Copyright © shopware AG (http://www.shopware.de)
 * @version    $Id$
 * @author shopware AG
 */

//{namespace name=backend/etracker/view/main}
//{block name="backend/swag_e_tracker/controller/tracker"}
Ext.define('Shopware.apps.SwagETracker.controller.Tracker', {

    /**
     * Extend from the standard ExtJS 4
     * @string
     */
    extend: 'Ext.app.Controller',

    snippets: {
        windowTitle: '{s name="window_title"}etracker{/s}',
        errorTitle: '{s name="error_title"}An error occurred{/s}',
        saveArea: {
            success: {
                title: '{s name="controller/saveArea/success/title"}Area saved{/s}',
                message: '{s name="controller/saveArea/success/message"}The area was successfully saved{/s}'
            }
        },
        deleteSingleArea: {
            success: {
                title: '{s name="controller/deleteSingleArea/success/title"}Area deleted{/s}',
                message: '{s name="controller/deleteSingleArea/success/message"}The area was succesfully deleted{/s}'
            }
        },
        deleteMultipleAreas: {
            title: '{s name="controller/deleteMultipleAreas/title"}Delete areas{/s}',
            message: '{s name="controller/deleteMultipleAreas/message"}You selected [0] areas. Do you really want to delete them?{/s}',
            success: {
                title: '{s name="controller/deleteMultipleAreas/success/title"}Areas deleted{/s}',
                message: '{s name="controller/deleteMultipleAreas/success/message"}The areas were successfully deleted{/s}'
            }
        },
        saveSettings: {
            success: {
                title: '{s name="controller/saveSettings/success/title"}Settings saved{/s}',
                message: '{s name="controller/saveSettings/success/message"}The settings were successfully saved{/s}'
            }
        }
    },

    /**
     * Creates all the needed event-listeners and handles all the important actions
     */
    init: function() {
        var me = this;

        me.control({
            'swagetracker-tracker-settings button[action=saveConfig]':{
                click: me.onSaveSettings
            },
            'swagetracker-tracker-settings':{
                changeAreaCombo: me.onChangeAreaCombo,
                changeSchemaCombo: me.onChangeSchemaCombo,
            },
            'swagetracker-tracker-areas':{
                editColumn: me.onEditColumn,
                deleteColumn: me.onDeleteSingleArea,
                fieldChange: me.onSearch
            },
            'swagetracker-tracker-areas button[action="add"]': {
                click: me.onAddArea
            },
            'swagetracker-tracker-areas button[action="deleteMultipleAreas"]': {
                click: me.onDeleteMultipleAreas
            },
            'swagetracker-tracker-detail button[action=saveArea]':{
                click: me.onSaveArea
            },
            'swagetracker-tracker-detail': {
                deleteColumn: me.onDeleteAreaDivision,
                changeDivisionCombo: me.onChangeDivisionCombo
            },
            'swagetracker-tracker-detail button[action=addDivisionArea]': {
                click: me.onAddAreaDivision
            }
        });

        me.callParent(arguments);
    },

    /**
     * This method creates a new category-/store-/controller-mapping in the area-detail window.
     *
     * @param btn
     */
    onAddAreaDivision: function (btn){
        var me = this,
            win = btn.up('window'),
            grid = win.down('grid'),
            form = win.down('form'),
            store = grid.getStore(),
            formValues = form.getValues(),
            remoteComboBox = grid.down('toolbar').down('combo'),
            divisionCombo = win.down('combo'),
            model;

        switch(divisionCombo.getValue()){
            //Controller
            case 1:
                model = Ext.create('Shopware.apps.SwagETracker.model.Controller');
                break;
            //Category
            case 2:
                model = Ext.create('Shopware.apps.SwagETracker.model.Category');
                break;
            //Shops
            default:
                model = Ext.create('Shopware.apps.SwagETracker.model.Shop');
                break;
        }

        model.set('id', remoteComboBox.getValue());
        model.set('areaId', formValues.id);
        model.set('name', remoteComboBox.getRawValue());
        model.save();

        store.add(model);
    },

    /**
     * This method is triggered by changing the value of the division-combobox.
     * It is needed to dynamically set the store of the remote-combobox and the grid.
     *
     * @param field
     * @param newValue
     */
    onChangeDivisionCombo: function(field, newValue) {
        var me = this,
            win = field.up('window');

        if(!win) {
            Ext.defer(this.onChangeDivisionCombo, 100, this, [ field, newValue ]);
            return;
        }

        var combo = win.down('toolbar').down('combo'),
            grid = win.down('grid'),
            data = [];

        switch(newValue){
            //Controller
            case 1:
                combo.bindStore(Ext.create('Shopware.apps.SwagETracker.store.Controllers'));
                if(win.areaRecord){
                    win.areaRecord.getControllers().each(function(controller){
                        data.push({
                            id: controller.get('id'),
                            name: controller.get('name'),
                            divisionId: win.areaRecord.get('divisionId')
                        });
                    });
                }
                break;
            //Category
            case 2:
                combo.bindStore(Ext.create('Shopware.apps.SwagETracker.store.Categories'));
                if(win.areaRecord){
                    win.areaRecord.getCategories().each(function(category){
                        data.push({
                            id: category.get('id'),
                            name: category.get('name'),
                            divisionId: win.areaRecord.get('divisionId')
                        });
                    });
                }
                break;
            //Shops
            default:
                combo.bindStore(Ext.create('Shopware.apps.SwagETracker.store.Shops', {
                    listeners: {
                        beforeload: {
                            fn: function (store, operation, eOpts) {
                                operation.params.storeId = win.storeId;
                                return true;
                            }
                        }
                    }
                }));
                if(win.areaRecord){
                    win.areaRecord.getShops().each(function(shop){
                        data.push({
                            id: shop.get('id'),
                            name: shop.get('name'),
                            divisionId: win.areaRecord.get('divisionId')
                        });
                    });
                }
            break;
        }
        var areaId = "";
        if(win.areaRecord){
            areaId = win.areaRecord.get('id');
        }

        var newStore = Ext.create('Ext.data.Store', {
            fields: [ 'id', 'name', 'divisionId' ],
            data: data,
            areaId: areaId
        });
        grid.reconfigure(newStore);
    },

    /**
     * This method is triggered when the user wants to search for a special entry.
     *
     * @param field
     * @param grid
     */
    onSearch: function(field, grid) {
        var me = this,
            store = grid.getStore();

        store.filters.remove(store.filters.items[store.filters.findIndex('property', 'searchValue')]);
        if(field.getValue().length > 0){
            store.filter('searchValue', field.getValue());
        }else{
            store.load();
        }
    },

    /**
     * This method is called to delete a shop-/controller-/category-mapping in the area-detail window.
     *
     * @param view
     * @param rowIndex
     * @param item
     * @param colIndex
     */
    onDeleteAreaDivision: function (view, rowIndex,  item, colIndex) {
        var me = this,
            store = view.getStore(),
            record = store.getAt(rowIndex),
            model;

        store.remove(record);
        switch(record.get('divisionId')){
            //Controller
            case 1:
                model = Ext.create('Shopware.apps.SwagETracker.model.Controller', {
                    id: record.get('id'),
                    name: record.get('name'),
                    areaId: store.areaId
                });
                break;
            //Category
            case 2:
                model = Ext.create('Shopware.apps.SwagETracker.model.Category', {
                    id: record.get('id'),
                    name: record.get('name'),
                    areaId: store.areaId
                });
                break;
            //Shops
            default:
                model = Ext.create('Shopware.apps.SwagETracker.model.Shop', {
                    id: record.get('id'),
                    name: record.get('name'),
                    areaId: store.areaId
                });
                break;
        }
        model.destroy();
    },

    /**
     * This method is called, when the user clicks on the pencil-icon in the area-grid.
     * It simply opens the area-detail window with some additional information.
     *
     * @param view
     * @param item
     * @param rowIndex
     * @param colIndex
     */
    onEditColumn: function(view, item, rowIndex, colIndex) {
        var me = this,
            record = view.getStore().getAt(rowIndex),
            container = view.up('tabpanel').up('container');

        this.getView('tracker.Detail').create({ storeId: container.shopId, areaRecord: record, areaStore: view.getStore() });
    },

    /**
     * This method saves or creates an area.
     *
     * @param btn
     */
    onSaveArea: function(btn) {
        var me = this,
            win = btn.up('window'),
            form = win.down('form'),
            grid = win.down('grid'),
            formValues = form.getForm().getValues(),
            formRecord = form.getRecord(),
            data;

        data = formValues;
        if(formRecord){
            data = Ext.Object.merge(formRecord.data, formValues);
        }

        var model = Ext.create('Shopware.apps.SwagETracker.model.Area', data);
        model.set('storeId', win.storeId);
        model.save({
            callback: function(data, operation){
                var records = operation.getRecords(),
                    record = records[0],
                    rawData = record.getProxy().getReader().rawData;
                if(operation.success){
                    win.close();
                    win.areaStore.load();
                    Shopware.Notification.createGrowlMessage(me.snippets.saveArea.success.title, me.snippets.saveArea.success.message, me.snippets.windowTitle);
                }else{
                    Shopware.Notification.createGrowlMessage(me.snippets.errorTitle, rawData.errorMsg, me.snippets.windowTitle);
                }
            }
        });
    },

    /**
     * This method only deletes a single area-row.
     *
     * @param view
     * @param rowIndex
     * @param colIndex
     * @param item
     */
    onDeleteSingleArea: function(view, rowIndex, colIndex, item) {
        var me = this,
            store = view.getStore(),
            container = view.up('tabpanel').up('container'),
            record = store.getAt(rowIndex);

        record.destroy({
            callback: function(data, operation){
                var records = operation.getRecords(),
                        record = records[0],
                        rawData = record.getProxy().getReader().rawData;
                if(operation.success){
                    store.clearFilter(true);
                    store.filter('storeId', container.shopId);
                    Shopware.Notification.createGrowlMessage(me.snippets.deleteSingleArea.success.title, me.snippets.deleteSingleArea.success.message, me.snippets.windowTitle);
                }else{
                    Shopware.Notification.createGrowlMessage(me.snippets.errorTitle, rawData.errorMsg, me.snippets.windowTitle);
                }
            }
        });
    },

    /**
     * This method is responsible for deleting more than one entry at once.
     *
     * @param btn
     */
    onDeleteMultipleAreas: function(btn){
        var me = this,
            grid = btn.up('grid'),
            selModel = grid.selModel,
            store = grid.getStore(),
            selection = selModel.getSelection(),
            container = btn.up('tabpanel').up('container'),
            message = Ext.String.format(me.snippets.deleteMultipleAreas.message, selection.length);

        //Create a message-box, which has to be confirmed by the user
        Ext.MessageBox.confirm(me.snippets.deleteMultipleAreas.title, message, function (response){
            //If the user doesn't want to delete the articles
            if (response !== 'yes')
            {
                return false;
            }

            Ext.each(selection, function(item){
                store.remove(item);
            });

            store.sync({
                callback: function(batch, operation) {
                    var rawData = batch.proxy.getReader().rawData;
                    if (rawData.success) {

                        store.clearFilter(true);
                        store.filter('storeId', container.shopId);
                        Shopware.Notification.createGrowlMessage(me.snippets.deleteMultipleAreas.success.title, me.snippets.deleteMultipleAreas.success.message, me.snippets.windowTitle);
                    }else{
                        Shopware.Notification.createGrowlMessage(me.snippets.errorTitle, rawData.errorMsg, me.snippets.windowTitle);
                    }
                }
            })
        });
    },

    /**
     * This method is called when the user wants to create a new area.
     * It displays the area-detail window and sets some additional information to the window.
     *
     * @param btn
     */
    onAddArea: function(btn){
        var me = this,
            view = btn.up('grid'),
            container = view.up('tabpanel').up('container');

        me.getView('tracker.Detail').create( { storeId: container.shopId, areaStore: view.getStore() } );
    },

    /**
     * This method is responsible for saving the config-data,
     * It is triggered by clicking on the "save"-button of the config-form.
     *
     * @param btn
     */
    onSaveSettings: function(btn){
        var me = this,
            tabPanel = btn.up('tabpanel'),
            form = tabPanel.down('form'),
            values = form.getForm().getValues(),
            shopTabPanel = tabPanel.up('tabpanel');

        if(values.ownPageNames == "on")
        {
            values.ownPageNames = true;
        }

        values.storeId = shopTabPanel.getActiveTab().shopId;
        var model = Ext.create('Shopware.apps.SwagETracker.model.Config', values);

        model.save({
            callback: function(data, operation){
                var records = operation.getRecords(),
                        record = records[0],
                        rawData = record.getProxy().getReader().rawData;
                if(operation.success){
                    Shopware.Notification.createGrowlMessage(me.snippets.saveSettings.success.title, me.snippets.saveSettings.success.message, me.snippets.windowTitle);
                }else{
                    Shopware.Notification.createGrowlMessage(me.snippets.errorTitle, rawData.errorMsg, me.snippets.windowTitle);
                }
            }
        });
    },

    /**
     * This method is triggered when the user changes the area-combo.
     * It is needed to enable the area-tab if the right option is selected.
     *
     * @param combo
     */
    onChangeAreaCombo: function(combo) {
        var tabPanel = combo.up('tabpanel');
        
        if(!tabPanel) {
            Ext.defer(this.onChangeAreaCombo, 100, this, [ combo ]);
            return;
        }

        var areaTab = tabPanel.down('container[name=areaTab]');

        var store = combo.getStore(),
            selectedRecord = store.getAt(store.find('areaId', combo.getValue()));


        areaTab.setDisabled(!selectedRecord.get('eventField'));
        var grid = tabPanel.down('grid');
        if(selectedRecord.get('areaId') == 4){
            grid.setDisabled(true);
        } else {
            grid.setDisabled(false);
        }
    },

    /**
     * This method is triggered when the user changes the schema-combo.
     * It is needed to enable the schema-field if the right option is selected.
     *
     * @param combo
     */
    onChangeSchemaCombo: function(combo) {
        var schemaTextField = combo.up('fieldset');
        
        if(!schemaTextField) {
            Ext.defer(this.onChangeSchemaCombo, 100, this, [ combo ]);
            return;
        }
        
        schemaTextField = schemaTextField.down('textfield[name=ownSchema]');
        var store = combo.getStore(),
            selectedRecord = store.getAt(store.find('value', combo.getValue()));
        
        schemaTextField.setDisabled(!selectedRecord.get('eventField'));
    }
});
//{/block}