/**
 * Shopware 4
 * Copyright © shopware AG
 *
 * According to our dual licensing model, this program can be used either
 * under the terms of the GNU Affero General Public License, version 3,
 * or under a proprietary license.
 *
 * The texts of the GNU Affero General Public License with an additional
 * permission and of our proprietary license can be found at and
 * in the LICENSE file you have received along with this program.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * "Shopware" is a registered trademark of shopware AG.
 * The licensing of the program under the AGPLv3 does not imply a
 * trademark license. Therefore any rights, title and interest in
 * our trademarks remain entirely with us.
 *
 * @category   Shopware
 * @package    SwagETracker
 * @subpackage View
 * @copyright  Copyright © shopware AG (http://www.shopware.de)
 * @version    $Id$
 * @author shopware AG
 */

//{namespace name=backend/etracker/view/main}
//{block name="backend/swag_e_tracker/view/tracker/detail"}
Ext.define('Shopware.apps.SwagETracker.view.tracker.Detail', {
    extend: 'Enlight.app.Window',
    alias: 'widget.swagetracker-tracker-detail',
    cls: 'createWindow',
    modal: true,
    layout: {
        type: 'vbox',
        align: 'stretch',
        pack: 'start'
    },
    defaults: {
        flex: 1
    },
    autoShow: true,
    title: '{s name="areaDetail/windowTitle"}Area detail{/s}',
    border: 0,
    width: 600,
    height: 300,
    stateful: true,
    stateId: 'shopware-tracker-detail',
    footerButton: false,

    snippets: {
        searchEmptyText: '{s name="view/search/emptyText"}Search{/s}',
        addButton: '{s name="view/areaDetail/add"}Add{/s}',
        columns: {
            name: '{s name="view/areaDetail/columns/name"}Name{/s}'
        },
        toolTips: {
            delete: '{s name="view/toolTips/delete"}Delete entry{/s}'
        },
        form: {
            name: {
                fieldLabel: '{s name="view/areaDetail/name/fieldLabel"}Name{/s}'
            },
            division: {
                fieldLabel: '{s name="view/areaDetail/division/fieldLabel"}Division into{/s}',
                emptyText: '{s name="view/areaDetail/division/fieldLabel"}Please choose{/s}'
            }
        },
        buttons: {
            cancel: '{s name="view/buttons/cancel"}Cancel{/s}',
            save: '{s name="view/buttons/save"}Save{/s}'
        }
    },

    /**
     * This method is called when this view is initiated.
     * It handles the creation of all the important components.
     */
    initComponent:function () {
        var me = this;

        me.registerEvents();
        me.detailForm = me.createFormPanel();
        me.areaGrid = me.createAreaGrid();
        me.dockedItems = [{
            xtype: 'toolbar',
            ui: 'shopware-ui',
            dock: 'bottom',
            cls: 'shopware-toolbar',
            items: me.createButtons()
        }];

        me.items = [ me.detailForm, me.areaGrid ];
        me.callParent(arguments);
    },

    /**
     * Defines additional events which will be
     * fired from the component
     *
     * @return void
     */
    registerEvents: function () {
        this.addEvents(

                /**
                 * Event will be fired when the user clicks the delete icon in the
                 * action column
                 *
                 * @event deleteColumn
                 * @param [object] View - Associated Ext.view.Table
                 * @param [integer] rowIndex - Row index
                 * @param [integer] colIndex - Column index
                 * @param [object] item - Associated HTML DOM node
                 */
                'deleteColumn',

                /**
                 * Event is fired when the user chooses a different option from the dropdown
                 *
                 * @event changeDivisionCombo
                 * @param [object] this - The field itself
                 * @param [object] newValue - The new selected value
                 */
                'changeDivisionCombo'
        )
    },

    /**
     * This method creates the area-grid and all the components needed for the grid, e.g. the toolbar.
     *
     * @returns Ext.form.Panel
     */
    createAreaGrid: function(){
        var me = this;

        var addButton = me.getAddButton();

        var searchField = me.getSearchField(addButton),
            grid = me.getAreaGrid(searchField, addButton);

        if (me.areaRecord) {
            var data = [];
            switch(me.areaRecord.get('divisionId')){
                //Controller
                case 1:
                    me.areaRecord.getControllers().each(function(controller){
                        data.push({
                            id: controller.get('id'),
                            name: controller.get('name'),
                            divisionId: me.areaRecord.get('divisionId')
                        });
                    });
                    break;
                //Category
                case 2:
                    me.areaRecord.getCategories().each(function(category){
                        data.push({
                            id: category.get('id'),
                            name: category.get('name'),
                            divisionId: me.areaRecord.get('divisionId')
                        });
                    });
                    break;
                //Shops
                default:
                    me.areaRecord.getShops().each(function(shop){
                        data.push({
                            id: shop.get('id'),
                            name: shop.get('name'),
                            divisionId: me.areaRecord.get('divisionId')
                        });
                    });
                    break;
            }

            var store = Ext.create('Ext.data.Store', {
                fields: [ 'id', 'name', 'divisionId' ],
                data: data,
                areaId: me.areaRecord.get('id')
            });

            grid.reconfigure(store);

            searchField.setDisabled(false);
        }

        return grid;
    },

    /**
     * This method creates the small form with two simple fields.
     *
     * @returns Ext.form.Panel
     */
    createFormPanel: function(){
        var me = this,
            formPanel = Ext.create('Ext.form.Panel', {
                layout: 'anchor',
                defaults:{
                    labelWidth:200,
                    anchor: '100%'
                },
                bodyPadding: 18,
                items:[
                    {
                        xtype: 'hidden',
                        name: 'id'
                    },
                    {
                        xtype: 'textfield',
                        fieldLabel: me.snippets.form.name.fieldLabel,
                        name: 'name'
                    },{
                        xtype: 'combo',
                        fieldLabel:  me.snippets.form.division.fieldLabel,
                        displayField: 'divisionName',
                        valueField: 'divisionId',
                        emptyText: me.snippets.form.division.emptyText,
                        name: 'divisionId',
                        store: Ext.create('Ext.data.Store', {
                            fields: ['divisionId', 'divisionName'],
                            data : [
                                { divisionId: 1, divisionName: "Controller"},
                                { divisionId: 2, divisionName: "Kategorien"},
                                { divisionId: 3, divisionName: "Sprachshops"}
                            ]
                        }),
                        listeners: {
                            change: {
                                fn: function(field, newValue, oldValue) {
                                    me.fireEvent('changeDivisionCombo', field, newValue);
                                }
                            }
                        }
                    }
                ]
        });

        if(me.areaRecord){
            formPanel.loadRecord(me.areaRecord);
        }

        return formPanel;
    },

    /**
     * Helper method to create the save-/cancel-button for the form
     * @returns Array
     */
    createButtons: function(){
        var me = this;
        var buttons = ['->',
            {
                text: me.snippets.buttons.cancel,
                cls: 'secondary',
                scope:me,
                handler:me.destroy
            },
            {
                text: me.snippets.buttons.save,
                action:'saveArea',
                cls:'primary'
            }
        ];

        return buttons;
    },

    /**
     * Helper method to create the search remote-combobox
     * @param addButton
     * @returns Shopware.form.field.Search
     */
    getSearchField: function(addButton){
        var me = this,
            categoryStore = Ext.create('Shopware.apps.SwagETracker.store.Categories');
        return Ext.create('Shopware.form.field.Search', {
            action : 'searchDivisions',
            width : 170,
            enableKeyEvents : true,
            emptyText : me.snippets.searchEmptyText,
            store: categoryStore,
            displayField: "name",
            disabled: true,
            valueField: 'id',
            listeners: {
                select: {
                    fn: function(){
                        addButton.setDisabled(false);
                    }
                }
            }
        });
    },

    /**
     * Helper method to create the toolbar
     * @param searchField
     * @param addButton
     * @returns Ext.toolbar.Toolbar
     */
    getToolBar: function(searchField, addButton){
        return Ext.create('Ext.toolbar.Toolbar', {
            dock: 'top',
            items:
                [
                    {
                        xtype: 'tbspacer',
                        width: 20
                    },
                    searchField,
                    addButton
                ]
        });
    },

    /**
     * Helper method to get the button for the toolbar
     * @returns Ext.button.Button
     */
    getAddButton: function(){
        var me = this;
        return Ext.create('Ext.button.Button', {
            text: me.snippets.addButton,
            action: 'addDivisionArea',
            iconCls: 'sprite-arrow-turn-270',
            disabled: true
        })
    },

    /**
     * Helper method to get the main area-grid
     *
     * @param searchField
     * @param addButton
     * @returns Ext.grid.Panel
     */
    getAreaGrid: function(searchField, addButton){
        var me = this,
            toolBar = me.getToolBar(searchField, addButton);
        return Ext.create('Ext.grid.Panel',
                {
                    width: '100%',
                    viewConfig: {
                        autoScroll: true
                    },
                    dockedItems: [toolBar],
                    columns: [
                        {
                            dataIndex: 'id',
                            hidden: true
                        },
                        {
                            header: me.snippets.columns.name,
                            dataIndex: 'name',
                            flex: 1
                        },
                        {
                            dataIndex: 'divisionId',
                            hidden: true
                        },
                        {
                            xtype: 'actioncolumn',
                            width: 60,
                            items: [
                                Ext.create('Ext.button.Button', {
                                    iconCls: 'sprite-minus-circle',
                                    action: 'delete',
                                    cls: 'delete',
                                    tooltip: me.snippets.toolTips.delete,
                                    handler:function (view, rowIndex, colIndex, item) {
                                        me.fireEvent('deleteColumn', view, rowIndex,  item, colIndex);
                                    }
                                })
                            ]
                        }
                    ]
                }
        );
    }
});
//{/block}