;(function ($) {
    $.plugin('etracker', {

        /**
         * The default options.
         * Mainly contains the selectors for the events.
         * You can set those by using data-attributes in HTML e.g. "data-deleteArticleSelector='abc'"
         */
        defaults: {
            /**
             * The selector for the delete-button of an article in the cart/checkout
             *
             * @property deleteArticleSelector
             * @type string
             */
            deleteArticleSelector: '.column--actions .column--actions-link',

            /**
             * The selector for the quantity-form of an article in the cart/checkout
             *
             * @property quantityFormSelector
             * @type string
             */
            quantityFormSelector: 'form[name^=basket_change_quantity] select[name=sQuantity]',

            /**
             * The selector for the button on the detail page to add an article to the cart
             *
             * @property buyButtonSelector
             * @type string
             */
            buyButtonSelector: '.buybox--button',

            /**
             * The selector for the button on the detail page to add an article to the watchlist
             *
             * @property watchlistButtonSelector
             * @type string
             */
            watchlistButtonSelector: '.cart--add-to-list, .link--notepad, .action--note',

            /**
             * The selector for the button on the watchlist to remove an article from the watchlist
             *
             * @property removeFromWatchlistButtonSelector
             * @type string
             */
            removeFromWatchlistButtonSelector: '.note--delete'
        },

        /** Plugin constructor */
        init: function () {
            var me = this;

            me.logEvent("INIT etracker Plugin");

            me.applyDataAttributes();
            me.initPlugin();

            // Delete article on checkout-page
            me._on(me.opts.deleteArticleSelector, 'click', $.proxy(me.onDeleteArticleCheckout, me));
            me._on(me.opts.quantityFormSelector, 'change', $.proxy(me.onChangeQuantity, me));

            // Delete article in ajax-cart
            $.subscribe(me.getEventName('plugin/swCollapseCart/onRemoveArticle'), $.proxy(me.onDeleteArticleCart, me));
            // ETR-12 'plugin/collapseCart/onRemoveArticle' doesn't trigger event
            $.subscribe(me.getEventName('plugin/imageZoom/onLensClick'), $.proxy(me.onZoomImage, me));
            $.subscribe(me.getEventName('plugin/swAjaxVariant/onBeforeRequestData'), $.proxy(me.onVariantChangeBefore, me));
            $.subscribe(me.getEventName('plugin/swAjaxVariant/onRequestData'), $.proxy(me.onVariantChangeAfter, me));
            $.subscribe(me.getEventName('plugin/swCookieConsentManager/onBuildCookiePreferences'), $.proxy(me.onCookieConsentChange, me));
            $.subscribe(me.getEventName('plugin/swCookiePermission/onAcceptButtonClick'), $.proxy(me.onCookieConsentAcceptAll, me));
        },

        /**
         * This method initializes the main logic of the plugin.
         * E.g. it creates the needed variables and prepares some events.
         */
        initPlugin: function () {
            var me = this;

            me.initVariables();

            if (me.quantitySelect.length) {
                me.prepareQuantityChange();
            }

            etCommerce.debugMode = me.debugJs;

            var watchlistButtonSelectorCount            = 0;
            var removeFromWatchlistButtonSelectorCount  = 0;
            var buyButtonSelectorCount                  = 0;
            var products                                = new Array();

            for (var i = 0; i < me.etracker.events.length; ++i) {
                var item = me.etracker.events[i];

                if (item.type == 'send') {
                    if (item.name == 'viewProduct') {
                        var category = [];
                        for (var iteration = 0; iteration < item.information.category.length; ++iteration) {
                            var categoryItem = item.information.category[iteration];
                            category.push(categoryItem);
                        }

                        var product = {
                            id: item.information.id,
                            name: item.information.name,
                            category: category,
                            price: item.information.price,
                            currency: item.information.currency,
                            variants: {}
                        };

                        me.logEvent(item.name, product);
                        etCommerce.sendEvent(item.name, product)
                    } else if (item.name == 'order') {
                        var order = item.information;

                        me.logEvent(item.name, order);
                        etCommerce.sendEvent(item.name, order);
                    }
                } else {
                    if(item.name == 'insertToBasket' || item.name == 'insertToWatchlist' || item.name == 'removeFromWatchlist') {
                        var category = [];
                        for (var iteration = 0; iteration < item.information.category.length; ++iteration) {
                            var categoryItem = item.information.category[iteration];
                            category.push(categoryItem);
                        }

                        products[i] = {
                            id: item.information.id,
                            name: item.information.name,
                            category: category,
                            price: item.information.price,
                            currency: item.information.currency,
                            variants: {}
                        };

                        if (item.name == 'insertToBasket') {
                            $(me.opts.buyButtonSelector).eq(buyButtonSelectorCount).data('i', i);

                            $(me.opts.buyButtonSelector).eq(buyButtonSelectorCount).bind('click', function () {
                                var elementI    = $(this).data('i');
                                var item        = me.etracker.events[elementI];
                                var quantity    = ~~($('#sQuantity').val());

                                // adjust price according to quantity
                                var product = products[elementI];
                                var prices = item.information.pricesArray;
                                for (var i = 0; i < prices.length; ++i) {
                                    if(quantity >= prices[i].from) {
                                        var tmp = ((100 - prices[i].percent) * product.price) / 100;
                                        // make copy to prevent modification of original product object
                                        product = JSON.parse(JSON.stringify(product))
                                        product.price = tmp.toFixed(2);
                                        break;
                                    }
                                }

                                me.logEvent(item.name, product, quantity);
                                etCommerce.sendEvent(item.name, product, quantity);
                            });

                            buyButtonSelectorCount++;
                        } else if (item.name == 'insertToWatchlist') {
                            $(me.opts.watchlistButtonSelector).eq(watchlistButtonSelectorCount).data('i', i);

                            $(me.opts.watchlistButtonSelector).eq(watchlistButtonSelectorCount).bind('click', function () {
                                var elementI    = $(this).data('i');
                                var item        = me.etracker.events[elementI];

                                me.logEvent(item.name, products[elementI], 1);
                                etCommerce.sendEvent(item.name, products[elementI], 1);
                            });

                            watchlistButtonSelectorCount++;
                        } else if (item.name == 'removeFromWatchlist') {
                            $(me.opts.removeFromWatchlistButtonSelector).eq(removeFromWatchlistButtonSelectorCount).data('i', i);

                            $(me.opts.removeFromWatchlistButtonSelector).eq(removeFromWatchlistButtonSelectorCount).bind('click', function (event) {
                                event.preventDefault();

                                var elementI    = $(this).data('i');
                                var item        = me.etracker.events[elementI];

                                me.logEvent(item.name, products[elementI], 1);
                                etCommerce.sendEvent(item.name, products[elementI], 1);

                                $(this).parent('form').submit();
                            });

                            removeFromWatchlistButtonSelectorCount++;
                        }
                    }
                }
            }
        },

        /**
         * This event is called everytime the user changes the quantity of an article in the checkout/cart.
         * @param event
         */
        onChangeQuantity: function (event) {
            var me = this;
            var $el = $(event.currentTarget);
            var $quantityForm = $el.parents('form');
            var oldValue = $el.data('oldValue');
            var currentValue = $el.val();
            var basketId = $quantityForm.find('input[name=sArticle]').val();

            $el.data('oldValue', currentValue);

            $.ajax({
                url: me.$el.attr('data-article-url'),
                data: { basketId: basketId, useArticleNumber: me.etracker.useArticleNumber },
                beforeSend: function() { }
            }).done(function (data) {
                var product = data,
                    difference = currentValue - oldValue;
                product.variants = {};

                if (difference < 0) {
                    me.logEvent('removeFromBasket', product, Math.abs(~~(difference)));
                    etCommerce.sendEvent('removeFromBasket', product, Math.abs(~~(difference)))
                } else if (difference > 0) {
                    me.logEvent('insertToBasket', product, Math.abs(~~(difference)));
                    etCommerce.sendEvent('insertToBasket', product, Math.abs(~~(difference)));
                }

                if (!me.debugJs) {
                    if (me.loadingindicator) {
                        $.loadingIndicator.open({
                            closeOnClick: false
                        });
                    }

                    $quantityForm.submit();
                }
            });
        },

        /**
         * This event is called when the user wants to delete an article from the checkout-cart.
         * @param event
         */
        onDeleteArticleCheckout: function (event, plugin, originalEvent) {
            event.preventDefault();
            var me = this,
                outer = $(event.currentTarget).parents('.etracker--cart-product'),
                basketId = outer.data('basket_id'),
                quantity = outer.data('quantity'),
                href = (event.currentTarget ? event.currentTarget.href : null),
                formToSubmit = $(event.currentTarget).parent('form');

            me.logEvent("DEL -> Cart ITEM");
            me.deleteArticleCall(me, basketId, quantity, href, formToSubmit);
        },

        /**
         * This event is called when the user wants to delete an article from the ajax-cart.
         * @param event
         * @param plugin
         * @param originalEvent
         */
        onDeleteArticleCart: function (event, plugin, originalEvent) {
            var me = this,
                parentForm = $(originalEvent.currentTarget).parents('.cart--item'),
                basketId = parentForm.find('.etracker--basket-id').val(),
                quantity = parentForm.find('.etracker--quantity').val();

            me.logEvent("DEL -> AJAX Cart ITEM");
            me.deleteArticleCall(me, basketId, quantity);
        },

        /**
         * This is called when the user zooms into an image.
         */
        onZoomImage: function () {
            ET_Event.galleryZoom('', '');
        },

        onVariantChangeBefore: function() {
            this.destroy();
        },

        onVariantChangeAfter: function() {
            var me = this;
            $.unsubscribe(me.getEventName('plugin/swAjaxVariant/onRequestData'));
            $('.etracker--details-variants').etracker();
        },

        onCookieConsentChange: function(event, plugin, preferences) {
            var enabled = $.getCookiePreference('etracker');
            if(enabled) {
                _etrackerOnReady.push(function () {
                    _etracker.enableCookies();
                });
            } else {
                _etrackerOnReady.push(function () {
                    _etracker.disableCookies();
                });
            }
        },

        onCookieConsentAcceptAll: function(event, plugin, preferences) {
            _etrackerOnReady.push(function () {
                _etracker.enableCookies();
            });
        },

        /**
         * Helper method to send a call to the "getArticleInfo"-action.
         * @param me
         * @param basketId
         * @param quantity
         * @param href
         */
        deleteArticleCall: function (me, basketId, quantity, href, formToSubmit) {
            href = href || null;
            $.ajax({
                url: me.$el.attr('data-article-url'),
                data: { basketId: basketId, useArticleNumber: me.etracker.useArticleNumber },
                async: false,
                error: function () {
                    me.logEvent("deleteArticleCall AJAX Error!!!");
                }
            }).done(function (data) {
                var product = data;

                product.variants = {};
                me.logEvent('removeFromBasket', product);
                etCommerce.sendEvent('removeFromBasket', product, ~~(quantity))

                // ETR-12 formToSubmit was NULL
                if (formToSubmit && formToSubmit.length > 0) {
                    setTimeout(function() {
                        formToSubmit.submit();
                    }, 500);
                } else if (href !== null) {
                    setTimeout(function() {
                        window.location = href;
                    }, 500);
                }
            });
        },

        /**
         * Helper method to initially set all the needed variables.
         */
        initVariables: function () {
            var me = this;

            me.etracker = $.parseJSON(me.$el.attr('data-etracker'));
            me.debugJs = me.etracker.debugJs;

            me.quantityForm = $('form[name^=basket_change_quantity]');
            me.quantitySelect = me.quantityForm.find('select[name=sQuantity]');

            me.ajaxCartRemoveBtn = $('.action--container .action--remove');

            if (me.quantitySelect.length) {
                me.autoSubmitPlugin = me.quantitySelect.eq(0).data('plugin_autoSubmit');
                if (me.autoSubmitPlugin === undefined) {
                    me.autoSubmitPlugin = me.quantitySelect.eq(0).data('plugin_swAutoSubmit');
                }
                me.loadingindicator = me.autoSubmitPlugin.opts.loadingindicator;
            }
        },

        /**
         * Helper method to prepare the quantity-change event for etracker.
         * It sets the important values and destroys the auto-submit-plugin.
         */
        prepareQuantityChange: function () {
            var me = this;

            me.quantitySelect.each(function() {
                $(this).data('oldValue', $(this).val());

                // We have to destroy the auto_submit-plugin to prevent the auto-submit
                $(this).data('plugin_swAutoSubmit').destroy();
            });
        },

        /**
         * Helper method to log the events being thrown by etracker.
         * Will only create a log if the debugging is enabled.
         * @param eventName
         */
        logEvent: function (eventName) {
            var me = this;
            if (me.debugJs) {
                console.log('Eventname', eventName);
                console.log('Data', arguments);
            }
        },

        /** Destroys the plugin */
        destroy: function () {
            var me = this;
            $.unsubscribe(me.getEventName('plugin/swCollapseCart/onRemoveArticle'));
            $.unsubscribe(me.getEventName('plugin/imageZoom/onLensClick'));
            $.unsubscribe(me.getEventName('plugin/swAjaxVariant/onBeforeRequestData'));
            me._destroy();
        }
    });

    $(document).ready(function () {
        window.etrackerReadyInterval = window.setInterval(function() {
            if(window.etrackerReady) {
                clearInterval(window.etrackerReadyInterval);
                var details = $('.etracker--details');
                if (details.length > 0) {
                    details.etracker();
                } else {
                    $('.etracker--details-variants').etracker();
                }
            }
        }, 50);
    });
})(jQuery);