<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

// Require composer dependencies if necessary
if (file_exists(__DIR__ . '/vendor/autoload.php')) {
    require_once __DIR__ . '/vendor/autoload.php';
}

if (!class_exists('ViisonCommon_Plugin_BootstrapV14')) {
    require_once __DIR__ . '/ViisonCommon/PluginBootstrapV14.php';
}

use Shopware\Plugins\ViisonCommon\Classes\Exceptions\InstallationException;
use Shopware\Plugins\ViisonDHLPremium\Subscribers;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;

/*
 * "Declare" some methods that are not required in our bootstrap, but whose existence is checked during Shopware's
 * code review:
 *
 * public function getVersion() {}
 */

final class Shopware_Plugins_Backend_ViisonDHLPremium_Bootstrap extends ViisonCommon_Plugin_BootstrapV14
{
    /**
     * @inheritdoc
     */
    protected $codeVersion = '1.4.0.55';

    /**
     * @inheritdoc
     */
    protected $minRequiredShopwareVersion = '5.5.0';

    /**
     * {@inheritdoc}
     * @throws InstallationException
     */
    public function runUpdate($oldVersion)
    {
        // Don't allow installs/updates if licence check fails, the exception thrown gives the user a hint what to do
        $this->checkLicenseInternal();

        $oldInstallVersion = ViisonCommonUtil::convertBinaryVersionToInstallVersion($oldVersion);

        switch ($oldInstallVersion) {
            case 'install':
                $this->subscribeEvent(
                    'Enlight_Controller_Front_StartDispatch',
                    'onStartDispatch'
                );

            case '1.0.0':
                // Nothing to do
            case '1.0.1':
                // Nothing to do
            case '1.0.2':
                // Nothing to do
            case '1.0.3':
                // Nothing to do
            case '1.0.4':
                // Nothing to do
            case '1.0.5':
                // Nothing to do
            case '1.0.6':
                // Nothing to do
            case '1.0.7':
                // Nothing to do
            case '1.1.0':
                // Nothing to do
            case '1.2.0':
                // Nothing to do
            case '1.3.0':
                // Nothing to do
            case '1.3.1':
                // Nothing to do
            case '1.4.0':
                // Next release

                // *** *** *** *** ***
                // NEVER REMOVE THE FOLLOWING BREAK! All updates must be added above this comment block!
                // *** *** *** *** ***
                break;
            default:
                throw InstallationException::updateFromVersionNotSupported($this, $oldVersion, $this->getVersion());
        }
    }

    /**
     * @inheritdoc
     */
    protected function runUninstall($deleteData)
    {
        if (!$deleteData) {
            return;
        }
    }

    /**
     * @inheritdoc
     */
    protected function runActivation()
    {
        // Nothing to do
    }

    /**
     * @inheritdoc
     */
    protected function runDeactivation()
    {
        // Nothing to do
    }

    /**
     * Registers the plugin's namespaces.
     */
    public function afterInit()
    {
        $this->loadDependencies();
        $this->loadPlugin();
    }

    /**
     * This callback function is triggered at the very beginning of the dispatch process and allows
     * us to register additional events on the fly.
     *
     * @param \Enlight_Event_EventArgs $args
     */
    public function onStartDispatch(\Enlight_Event_EventArgs $args)
    {
        // Nothing to do here, since the dynamic subscribers were already registered in 'afterInit()'
    }

    /**
     * Uses the dependency loader to load the namespaces and susbcribers of all required,
     * shared dependencies.
     */
    private function loadDependencies()
    {
        // Require all shared dependencies
        $loader = VIISON\ShopwarePluginDependencyLoader\Loader::getInstance();
        $loader->requireDependencies($this->Path(), [
            'ViisonCommon'
        ]);
        Shopware\Plugins\ViisonCommon\Classes\PickwareAutoloadDependencyLoader::ensureDependenciesLoaded();

        // Add the subscribers of ViisonCommon
        $viisonCommonSubscriberRegistrator = new Shopware\Plugins\ViisonCommon\Classes\SubscriberRegistrator($this);
        $viisonCommonSubscriberRegistrator->registerSubscribers();

        // Load the Shopware polyfill
        require_once __DIR__ . '/ViisonCommon/Polyfill/Loader.php';
    }

    /**
     * First checks whether the plugin is installed and is active and, ff it is, first the namespaces
     * of this plugin are registered with the class loader, before all subscribers are instanciated
     * and added to the event manager.
     */
    private function loadPlugin()
    {
        if (!$this->checkLicenseInternal(false)) {
            return;
        }

        // Make sure this plugin is installed and active before registering the susbcribers
        if (!$this->isInstalledAndActive()) {
            return;
        }

        // Create all plugin subscribers
        $subscribers = [
            new Subscribers\CompatibilityCheck($this)
        ];

        // Make sure that the subscribers are only added once
        if (!$this->isSubscriberRegistered($subscribers[0])) {
            $eventManager = $this->get('events');
            foreach ($subscribers as $subscriber) {
                $eventManager->addSubscriber($subscriber);
            }
        }
    }

    // Removed license checking method
    // Removed license checking method
    // Removed license checking method
    // Removed license checking method
    // Removed license checking method
    // Removed license checking method
    // Removed license checking method
    // Removed license checking method
    // Removed license checking method

    /**
     * Checks whether this shop has a license to run this plugin. That is, this method first
     * checks for a valid license for this plugin. If no such license can be found, it also checks
     * for a valid license for the ViisonIntraship plugin.
     *
     * @param bool $throwException
     * @return bool True, if the plugin is licensed. Otherwise false.
     * @throws Exception
     */
    public function checkLicenseInternal($throwException = true)
    {
        $ionCubeNotLoaded = !extension_loaded('ionCube Loader');
        $fileNotEncoded = function_exists('ioncube_file_is_encoded') && !ioncube_file_is_encoded();
        if ($ionCubeNotLoaded || $fileNotEncoded) {
            return true;
        }

        if (!Shopware()->Container()->has('license')) {
            if ($throwException) {
                throw new \Exception('The license manager has to be installed and active');
            }

            return false;
        }

        $licenseException = null;

        // Check vor a ViisonDHLPremium license first
        try {
            $licenseValid = $this->checkLicenseForPlugin(
                'ViisonDHLPremium',
                'kimTbOOgp2J8sj+9SoNBNUZ2Uts=',
                'gEnQ8bZn2OeR6+fD5iBLmwfsKmM=',
                $throwException
            );
        } catch (\Exception $e) {
            // Save exception for later use
            $licenseException = $e;
        }
        if ($licenseValid) {
            return true;
        }

        // No valid ViisonDHLPremium license found, hence check for an old
        // DHL Adapter (ViisonIntraship) license
        try {
            $licenseValid = $this->checkLicenseForPlugin(
                'ViisonIntraship',
                'xBsnn+XPyY2iutOT7Je3+CeU+vw=',
                'CgyU7NdqHqx9ZPZf56csX6IiE84=',
                $throwException
            );
        } catch (\Exception $e) {
            if (!$licenseException) {
                // Save exception for later use
                $licenseException = $e;
            }
        }

        if ($throwException && !$licenseValid) {
            throw $licenseException;
        }

        return $licenseValid;
    }

    /**
     * Performs the actual check of a license for the plugin with the given $pluginName and signature.
     *
     * @param string $pluginName
     * @param string $licenseSignaturePartA
     * @param string $licenseSignaturePartB
     * @param boolean $throwException (optional)
     * @return boolean
     * @throws \Exception
     */
    private function checkLicenseForPlugin($pluginName, $licenseSignaturePartA, $licenseSignaturePartB, $throwException = true)
    {
        try {
            $s = base64_decode($licenseSignaturePartA);
            $c = base64_decode($licenseSignaturePartB);
            $r = sha1(uniqid('', true), true);
            /** @var $l Shopware_Components_License */
            $l = $this->get('license');
            $i = $l->getLicense($pluginName, $r);
            $t = $l->getCoreLicense();
            // phpcs:ignore Generic.PHP.ForbiddenFunctions.FoundWithAlternative
            $u = strlen($t) === 20 ? sha1($t . $s . $t, true) : 0;
            $r = $i === sha1($c. $u . $r, true);
            if (!$r && $throwException) {
                throw new \Exception('License check for module "' . $pluginName . '" has failed.');
            }

            return $r;
        } catch (\Exception $e) {
            if ($throwException) {
                throw new \Exception('License check for module "' . $pluginName . '" has failed.');
            }

            return false;
        }
    }
}
