<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonCommon\Classes\Document\RenderingEngine;

use Shopware\Plugins\ViisonCommon\Classes\Document\PaperLayout;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util;

/**
 * Render a document to PDF using DomPdf
 */
class DomPdfRenderingEngine implements RenderingEngine
{
    /**
     * @var array Options for the DomPdf renderer
     */
    protected $options = [];

    /**
     * @inheritDoc
     */
    public function render($html, PaperLayout $paperLayout)
    {
        $domPdf = Util::createDompdfInstance($this->options);

        $domPdf->setPaper(
            DomPdfRenderingEngine::convertToDomPdfPaperFormatString($paperLayout),
            $paperLayout->pageOrientation
        );

        $domPdf->loadHtml($html);

        // Dompdf is currently not capable of handling SVG header informations
        // like title or desc and hence outputs related warnings to the output
        // stream. To prevent pollution of the output stream dompdf outputs are
        // captured and the output stream is cleaned after the rendering.
        ob_start();
        $domPdf->render();
        ob_get_clean();

        return $domPdf->output();
    }

    /**
     * Converts a paper size from millimeters to pt (= 1/72 inch).
     *
     * @param int $width The paper width in millimeters.
     * @param int $height The paper height in millimeters.
     * @return array An array containing the respective paper size in pixel.
     */
    public static function getPaperSizeByMillimeters($width, $height)
    {
        $inch = 25.4; // 1 inch = 25,4 mm

        return [
            0,
            0,
            $width / $inch * 72,
            $height / $inch * 72,
        ];
    }

    /**
     * @return array
     */
    public function getOptions()
    {
        return $this->options;
    }

    /**
     * @param array $options
     */
    public function setOptions($options)
    {
        $this->options = $options;
    }

    /**
     * @param PaperLayout $paperLayout
     * @return string|array
     */
    public static function convertToDomPdfPaperFormatString(PaperLayout $paperLayout)
    {
        // Map page formats to DomPDF specific format strings
        switch ($paperLayout->pageFormat) {
            case PaperLayout::PAGE_FORMAT_DIN_A3:
                $documentPaper = 'a3';
                break;
            case PaperLayout::PAGE_FORMAT_DIN_A4:
                $documentPaper = 'a4';
                break;
            case PaperLayout::PAGE_FORMAT_DIN_A5:
                $documentPaper = 'a5';
                break;
            case PaperLayout::PAGE_FORMAT_DIN_A6:
                $documentPaper = 'a6';
                break;
            default:
                if ($paperLayout->pageHeightInMillimeters && $paperLayout->pageWidthInMillimeters) {
                    $documentPaper = DomPdfRenderingEngine::getPaperSizeByMillimeters(
                        $paperLayout->pageWidthInMillimeters,
                        $paperLayout->pageHeightInMillimeters
                    );
                } else {
                    $documentPaper = 'A4';
                }
        }

        return $documentPaper;
    }
}
