<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonCommon\Classes\Document\RenderingEngine;

use Mpdf\Mpdf;
use Shopware\Plugins\ViisonCommon\Classes\Document\PaperLayout;

/**
 * Render a document to PDF using mPdf
 */
class MPdfRenderingEngine implements RenderingEngine
{
    /**
     * @var array
     */
    protected $defaultMpdfConfig;

    /**
     * @var int Left margin for the document
     */
    protected $left = 0;

    /**
     * @var int Right margin for the document
     */
    protected $right = 0;

    /**
     * @var int Top margin for the document
     */
    protected $top = 0;

    /**
     * @var int bottom margin for the document
     */
    protected $bottom = 0;

    /**
     * @param array $defaultMpdfConfig
     */
    public function __construct(array $defaultMpdfConfig = [])
    {
        $this->defaultMpdfConfig = $defaultMpdfConfig;
    }

    /**
     * @inheritDoc
     */
    public function render($html, PaperLayout $paperLayout)
    {
        $mPdf = $this->createMpdfInstance($paperLayout);
        $mPdf->WriteHTML($html);

        ob_start();
        $return = $mPdf->Output('', 'S');
        ob_get_clean();

        return $return;
    }

    /**
     * Set the page margins
     *
     * @param int $left Left margin
     * @param int $right Right margin
     * @param int $top Top margin
     * @param int $bottom Bottom margin
     */
    public function setPageMargins($left, $right, $top, $bottom)
    {
        $this->left = $left;
        $this->right = $right;
        $this->top = $top;
        $this->bottom = $bottom;
    }

    /**
     * @return int
     */
    public function getLeft()
    {
        return $this->left;
    }

    /**
     * @param int $left
     */
    public function setLeft($left)
    {
        $this->left = $left;
    }

    /**
     * @return int
     */
    public function getRight()
    {
        return $this->right;
    }

    /**
     * @param int $right
     */
    public function setRight($right)
    {
        $this->right = $right;
    }

    /**
     * @return int
     */
    public function getTop()
    {
        return $this->top;
    }

    /**
     * @param int $top
     */
    public function setTop($top)
    {
        $this->top = $top;
    }

    /**
     * @return int
     */
    public function getBottom()
    {
        return $this->bottom;
    }

    /**
     * @param int $bottom
     */
    public function setBottom($bottom)
    {
        $this->bottom = $bottom;
    }

    /**
     * @param PaperLayout $paperLayout
     * @return string
     */
    public static function convertToMpdfPaperFormatString(PaperLayout $paperLayout)
    {
        // Map page formats to mPDF specific format strings
        switch ($paperLayout->pageFormat) {
            case PaperLayout::PAGE_FORMAT_DIN_A3:
                $documentPaper = 'A3';
                break;
            case PaperLayout::PAGE_FORMAT_DIN_A4:
                $documentPaper = 'A4';
                break;
            case PaperLayout::PAGE_FORMAT_DIN_A5:
                $documentPaper = 'A5';
                break;
            case PaperLayout::PAGE_FORMAT_DIN_A6:
                $documentPaper = 'A6';
                break;
            default:
                $documentPaper = 'A4';
        }

        if ($paperLayout->pageOrientation === PaperLayout::PAGE_ORIENTATION_LANDSCAPE) {
            $documentPaper .= '-L';
        }

        return $documentPaper;
    }

    /**
     * @param PaperLayout $paperLayout
     * @return \mPDF|Mpdf
     */
    private function createMpdfInstance(PaperLayout $paperLayout)
    {
        $mpdfConfig = $this->defaultMpdfConfig;
        $mpdfConfig['margin_left'] = $this->left;
        $mpdfConfig['margin_right'] = $this->right;
        $mpdfConfig['margin_top'] = $this->top;
        $mpdfConfig['margin_bottom'] = $this->bottom;
        $mpdfConfig['format'] = self::convertToMpdfPaperFormatString($paperLayout);
        $mpdfConfig['mode'] = $mpdfConfig['mode'] ?: 'utf-8';
        $mpdfConfig['default_font_size'] = $mpdfConfig['default_font_size'] ?: 0;
        $mpdfConfig['default_font'] = $mpdfConfig['default_font'] ?: '';
        $mpdfConfig['margin_header'] = $mpdfConfig['margin_header'] ?: 9;
        $mpdfConfig['margin_footer'] = $mpdfConfig['margin_footer'] ?: 9;

        if (class_exists('Mpdf\\Mpdf')) {
            // mPDF 7 (SW >= 5.5)
            return new Mpdf($mpdfConfig);
        } else {
            // mPDF 6 (SW <= 5.4)
            require_once(Shopware()->DocPath() . 'engine/Library/Mpdf/mpdf.php');

            return new \mPDF(
                $mpdfConfig['mode'],
                $mpdfConfig['format'],
                $mpdfConfig['default_font_size'],
                $mpdfConfig['default_font'],
                $mpdfConfig['margin_left'],
                $mpdfConfig['margin_right'],
                $mpdfConfig['margin_top'],
                $mpdfConfig['margin_bottom'],
                $mpdfConfig['margin_header'],
                $mpdfConfig['margin_footer']
            );
        }
    }
}
