<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonCommon\Classes\Subscribers;

use Enlight_Event_EventArgs;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Plugins\ViisonCommon\Subscribers\Api\RestApiVariantUpdateSubscriber;

/**
 * A convenience subscriber that can be used to handle the creation or updates of variants (article details) via the
 * Shopware REST API. It is just an abstraction layer on top of the plain events fired by
 * {@link RestApiVariantUpdateSubscriber}.
 */
abstract class AbstractSimpleRestApiVariantUpdateSubscriber extends AbstractBaseSubscriber
{
    /**
     * Override to modify and return the passed `$variantPostData` to e.g. prevent some fields from being saved
     * by the REST API resources when the new variant is created.
     *
     * @param array $variantPostData
     * @return array
     */
    protected function modifyVariantPostDataBeforeCreation(array $variantPostData)
    {
        return $variantPostData;
    }

    /**
     * Override this method to perform extra work after the passed `$variant` was created using the passed
     * `$originalVariantPostData`.
     *
     * @param ArticleDetail $variant
     * @param array $originalVariantPostData
     */
    protected function processCreatedVariant(ArticleDetail $variant, array $originalVariantPostData)
    {
    }

    /**
     * Override to modify and return the passed `$variantPostData` to e.g. prevent some fields from being saved
     * by the REST API resources when the passed `$variant` is updated.
     *
     * @param array $variantPostData
     * @return array
     */
    protected function modifyVariantPostDataBeforeUpdate(ArticleDetail $variant, array $variantPostData)
    {
        return $variantPostData;
    }

    /**
     * Override this method to perform extra work after the passed `$variant` was updated using the passed
     * `$originalVariantPostData`.
     *
     * @param ArticleDetail $variant
     * @param array $originalVariantPostData
     */
    protected function processUpdatedVariant(ArticleDetail $variant, array $originalVariantPostData)
    {
    }

    /**
     * @inheritdoc
     */
    public static function getSubscribedEvents()
    {
        return [
            RestApiVariantUpdateSubscriber::FILTER_EVENT_NAME_VARIANT_CREATION_BEFORE => 'onBeforeVariantCreation',
            RestApiVariantUpdateSubscriber::NOTIFY_EVENT_NAME_VARIANT_CREATION_AFTER => 'onAfterVariantCreation',
            RestApiVariantUpdateSubscriber::FILTER_EVENT_NAME_VARIANT_UPDATE_BEFORE => 'onBeforeVariantUpdate',
            RestApiVariantUpdateSubscriber::NOTIFY_EVENT_NAME_VARIANT_UPDATE_AFTER => 'onAfterVariantUpdate',
        ];
    }

    /**
     * @param Enlight_Event_EventArgs $args
     * @return array
     */
    public function onBeforeVariantCreation(Enlight_Event_EventArgs $args)
    {
        return $this->modifyVariantPostDataBeforeCreation($args->getReturn());
    }

    /**
     * @param Enlight_Event_EventArgs $args
     */
    public function onAfterVariantCreation(Enlight_Event_EventArgs $args)
    {
        $this->processCreatedVariant(
            $args->get(RestApiVariantUpdateSubscriber::EVENT_ARG_KEY_VARIANT),
            $args->get(RestApiVariantUpdateSubscriber::EVENT_ARG_KEY_ORIGINAL_VARIANT_POST_DATA)
        );
    }

    /**
     * @param Enlight_Event_EventArgs $args
     * @return array
     */
    public function onBeforeVariantUpdate(Enlight_Event_EventArgs $args)
    {
        return $this->modifyVariantPostDataBeforeUpdate(
            $args->get(RestApiVariantUpdateSubscriber::EVENT_ARG_KEY_VARIANT),
            $args->getReturn()
        );
    }

    /**
     * @param Enlight_Event_EventArgs $args
     */
    public function onAfterVariantUpdate(Enlight_Event_EventArgs $args)
    {
        $this->processUpdatedVariant(
            $args->get(RestApiVariantUpdateSubscriber::EVENT_ARG_KEY_VARIANT),
            $args->get(RestApiVariantUpdateSubscriber::EVENT_ARG_KEY_ORIGINAL_VARIANT_POST_DATA)
        );
    }
}
