<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonCommon\Components\Migration;

use Psr\Log\LoggerInterface;

/**
 * Transcript for a migration to add messages for the user, clear caches and it offers a logger interface.
 */
class MigrationTranscript
{
    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var string[] Caches to invalidated after a setup process
     */
    protected $invalidatedCaches = [];

    /**
     * Messages that will be displayed to to user after successful migration run.
     *
     * @var MigrationMessage[]
     */
    private $messages = [];

    /**
     * @param LoggerInterface $logger
     */
    public function __construct(LoggerInterface $logger)
    {
        $this->logger = $logger;
    }

    /**
     * @return array
     */
    public function getInvalidatedCaches()
    {
        return $this->invalidatedCaches;
    }

    /**
     * @param string $snippetName
     * @param string $snippetNamespace
     * @param array $snippetArguments
     */
    public function addLocalizedMessage($snippetName, $snippetNamespace, array $snippetArguments = [])
    {
        $this->messages[] = new MigrationMessage($snippetName, $snippetNamespace, $snippetArguments);
    }

    /**
     * Invalidates the caches "theme" and "http".
     */
    public function invalidateFrontendCache()
    {
        $this->invalidatedCaches[] = 'theme';
        $this->invalidatedCaches[] = 'http';

        $this->invalidatedCaches = array_unique($this->invalidatedCaches);
    }

    /**
     * Invalidates the caches "config", "template" and "proxy".
     */
    public function invalidateDefaultCaches()
    {
        $this->invalidatedCaches[] = 'config';
        $this->invalidatedCaches[] = 'template';
        $this->invalidatedCaches[] = 'proxy';

        $this->invalidatedCaches = array_unique($this->invalidatedCaches);
    }

    /**
     * @return MigrationMessage[]
     */
    public function getMessages()
    {
        return $this->messages;
    }

    /**
     * @return LoggerInterface
     */
    public function getLogger()
    {
        return $this->logger;
    }
}
