<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonCommon\Components;

use Enlight_Event_EventManager as EventManager;
use Shopware\Bundle\PluginInstallerBundle\Service\InstallerService;
use Exception;
use Shopware\Models\Plugin\Plugin;
use Shopware\Models\Shop\Shop;
use Shopware\Plugins\ViisonCommon\Classes\PluginConfigImport\PluginConfigReader;

class PluginConfigImporterService extends PluginConfigReader
{
    /**
     * Event to check if the standard config import should be used.
     */
    const NOTIFY_UNTIL_ALLOW_STANDARD_IMPORT_EVENT = 'Shopware_Plugins_ViisonPickware_AllowStandardConfigImport';

    /**
     * Event to fire after the config has been imported.
     */
    const NOTIFY_AFTER_PLUGIN_CONFIG_IMPORT = 'Shopware_Plugins_ViisonPickware_AfterConfigImport';

    /**
     * @var EventManager
     */
    private $eventManager;

    /**
     * @var InstallerService
     */
    private $pluginInstaller;

    /**
     * @param EventManager $eventManager
     * @param InstallerService $pluginInstaller
     */
    public function __construct(EventManager $eventManager, InstallerService $pluginInstaller)
    {
        $this->eventManager = $eventManager;
        $this->pluginInstaller = $pluginInstaller;
    }

    /**
     * Imports a plugin's config into Shopware.
     *
     * @param Plugin $plugin
     * @param string $configPath
     * @param Shop|null $shop
     */
    public function importConfigForPlugin(Plugin $plugin, $configPath, Shop $shop = null)
    {
        $allowStandardImportEventResult = $this->eventManager->notifyUntil(
            self::NOTIFY_UNTIL_ALLOW_STANDARD_IMPORT_EVENT,
            [
                'plugin' => $plugin,
                'configPath' => $configPath,
                'shop' => $shop,
            ]
        );

        if ($allowStandardImportEventResult === null || $allowStandardImportEventResult->getReturn() === null) {
            $this->importConfigFromFile($plugin, $configPath, $shop);
        }

        $this->eventManager->notify(
            self::NOTIFY_AFTER_PLUGIN_CONFIG_IMPORT,
            [
                'plugin' => $plugin,
                'shop' => $shop,
            ]
        );
    }

    /**
     * Imports a plugin's configuration into Shopware's standard plugin configuration.
     *
     * @param Plugin $plugin
     * @param $configPath
     * @param Shop|null $shop
     * @throws Exception
     */
    private function importConfigFromFile(Plugin $plugin, $configPath, Shop $shop = null)
    {
        $configurationWrapper = $this->getConfigFileData($configPath);

        try {
            foreach ($configurationWrapper['configuration'] as $key => $value) {
                $this->pluginInstaller->saveConfigElement($plugin, $key, $value, $shop);
            }
        } catch (Exception $exception) {
            throw new Exception('The configuration\'s structure was not well formed.', 0, $exception);
        }
    }
}
